// Scala Lecture 2
//=================
 
// - Options
// - Higher-Order Functions (short-hand notation)
// - maps (behind for-comprehensions)
// - Pattern-Matching
// - Recursion


// Function Definitions
//======================

// The general scheme for a function: you have to give a 
// type to each argument and a return type of the function
//
//  def fname(arg1: ty1, arg2: ty2,..., argn: tyn): rty = {
//     ....
//  }

def incr(x: Int) : Int = x + 1
incr(42)

def add(x: Int, y: Int) : Int = x + y


def average(ls: List[Int]) : Option[Int] = {
  val s = ls.sum
  val l = ls.length
  if (l == 0) None else Some(s / l)
}

average(List(1,2,3,4))
average(List())




// The Option Type
//=================

// in Java, if something unusually happens, you return null 
// or raise an exception
//
//in Scala you use Options instead
//   - if the value is present, you use Some(value)
//   - if no value is present, you use None


List(7,2,3,4,5,6).find(_ < 4)
List(5,6,7,8,9).find(_ < 4)

// Int:      ..., 0, 1, 2,...
// Boolean:  true false
//
// List[Int]: Nil, List(_) 
//
// Option[Int]: None, Some(0), Some(1), ...
// Option[Boolean]: None, Some(true), Some(false)
// Option[...]: None, Some(_)

def div(x: Int, y: Int) : Int = 
  x / y

def safe_div(x: Int, y: Int) : Option[Int] = 
  if (y == 0) None else Some(x / y)

def foo_calculation(x: Int) = 
  if (safe_div(x, x) == None) 24 else safe_div(x, x).get

foo_calculation(3)
foo_calculation(0)

safe_div(10 + 5, 3)  

List(1,2,3,4,5,6).indexOf(7)
List[Int]().min
List[Int](1,2,3).minOption



// better error handling with Options (no exceptions)
//
//  Try(something).getOrElse(what_to_do_in_case_of_an_exception)
//

import scala.util._      // Try,...
import io.Source         // fromURL

//val my_url = "https://nms.kcl.ac.uk/christian.urban/"
val my_url = "https://urbanchr.github.io/"

println(Try(Source.fromURL(my_url)(using "ISO-8859-1").mkString).toOption)

.mkString  
Source.fromURL(my_url)(using "ISO-8859-1").getLines().toList

Try(Source.fromURL(my_url)(using "ISO-8859-1").mkString).getOrElse("")

Try(Some(Source.fromURL(my_url)(using "ISO-8859-1").mkString)).getOrElse(None)


// the same for files

Try(Some(Source.fromFile("test.txt")(using "ISO-8859-1").mkString)).getOrElse(None)

Try(Source.fromFile("test.txt")(using "ISO-8859-1").mkString).toOption

Using(Source.fromFile("test.txt")(using "ISO-8859-1"))(_.mkString).toOption

// how to implement a function for reading 
// (lines) from files...
//
def get_contents(name: String) : List[String] = 
  Source.fromFile(name)(using "ISO-8859-1").getLines().toList

get_contents("text.txt")
get_contents("test.txt")

// slightly better - return Nil
def get_contents(name: String) : List[String] = 
  Try(Source.fromFile(name)(using "ISO-8859-1").getLines.toList).getOrElse(List())

get_contents("text.txt")

// much better - you record in the type that things can go wrong 
def get_contents(name: String) : Option[List[String]] = 
  Try(Some(Source.fromFile(name)(using "ISO-8859-1").getLines().toList)).getOrElse(None)

get_contents("text.txt")
get_contents("test.txt")


// operations on options

val lst = List(None, Some(1), Some(2), None, Some(3))

for (x <- lst) yield {
  if (x == None) None
  else Some(x.get + 1)
}

lst.flatten

Some(1).get
None.get

Some(1).isDefined
None.isDefined

for (x <- lst) yield x.getOrElse(0)



val ps = List((3, 0), (4, 2), (6, 2), 
              (2, 0), (1, 0), (1, 1))

// division where possible

for ((x, y) <- ps) yield {
  if (y == 0) None else Some(x / y)
}



// getOrElse is for setting a default value

val lst = List(None, Some(1), Some(2), None, Some(3))


// a function that turns strings into numbers 
// (similar to .toInt)
Integer.parseInt("1234")


def get_me_an_int(s: String) : Option[Int] = 
 Try(Some(Integer.parseInt(s))).getOrElse(None)


// This may not look any better than working with null in Java, but to
// see the value, you have to put yourself in the shoes of the
// consumer of the get_me_an_int function, and imagine you didn't
// write that function.
//
// In Java, if you didn't write this function, you'd have to depend on
// the Javadoc of the get_me_an_int. If you didn't look at the Javadoc, 
// you might not know that get_me_an_int could return null, and your 
// code could potentially throw a NullPointerException.


// even Scala is not immune to problems like this:

List(5,6,7,8,9).indexOf(7)
List(5,6,7,8,9).indexOf(10)
List(5,6,7,8,9)(-1)


Try({
  val x = 3
  val y = 0
  Some(x / y)
}).getOrElse(None)


// minOption 
// maxOption 
// minByOption 
// maxByOption

def altproduct(xs: List[Int]) : List[Int] = xs match {
  case Nil => Nil
  case (x::y::xs) => x * y :: altproduct(y::xs)
  case (x::Nil) => List(x)
}

altproduct(List(1,2,3,4,5))

def powerset(xs: Set[Int]) : Set[Set[Int]] = {
  if (xs == Set()) Set(Set())
  else {
    powerset(xs.tail) ++ powerset(xs.tail).map(_ + xs.head)
  }
}     

powerset(Set(1,2,3)).mkString("\n")

// Higher-Order Functions
//========================

// functions can take functions as arguments
// and produce functions as result

val foo = for (n <- (1 to 10).toList) yield n * n


def even(x: Int) : Boolean = x % 2 == 0

even(2)
even(3)

def even(x: Int) : Boolean = x % 2 == 1

val foo_fun = even

foo_fun(2)
foo_fun(3)


def odd(x: Int) : Boolean = x % 2 == 1

def inc(x: Int) : Int = x + 1
val lst = (1 to 10).toList

lst.filter(odd)
lst.find(even)
lst.find(_ % 2 == 0)
lst.sortWith(_ < _)  // (x,y) => x < y
lst.find(_ > 4)
lst.count(odd)

lst.filter(_ % 2 == 0)



lst.find(_ < 4)
lst.filter(_ < 4) 

val x = 3 < 4

def less4(x: Int) : Boolean = x < 4
lst.find(less4)
lst.find(x => !(x < 4))

lst.filter(x => x % 2 == 0)
lst.filter(_ % 2 == 0)


lst.sortWith((x, y) => x > y)
lst.sortWith(_ < _)

// but this only works when the arguments are clear, but 
// not with multiple occurences
lst.find(n => odd(n) && n > 2)


// lexicographic ordering
val ps = List((3, 0), (3, 2), (4, 2), (2, 2), 
              (2, 0), (1, 1), (1, 0))

def lex(x: (Int, Int), y: (Int, Int)) : Boolean = 
  if (x._1 == y._1) x._2 < y._2 else x._1 < y._1

ps.sortWith(lex)

ps.sortBy(x => x._1)
ps.sortBy(_._2)

ps.maxBy(x => x._2)
ps.maxBy(_._2)


// maps (lower-case)
//===================

def double(x: Int): Int = x + x
def square(x: Int): Int = x * x

val lst = (1 to 10).toList

lst.map(square)
lst.map(n => square(n))

for (n <- lst) yield square(n)

lst.map(x => (double(x), square(x)))

// works also for strings
def tweet(c: Char) = c.toUpper

"Hello\nWorld".map(tweet)


// this can be iterated

lst.map(square).filter(_ > 4)

lst.map(square).find(_ > 4)
lst.map(square).find(_ > 4).map(double)

lst.map(square)
   .find(_ > 4)
   .map(double)


// Option Type and maps
//======================

// a function that turns strings into numbers (similar to .toInt)
Integer.parseInt("12u34")

// maps on Options

import scala.util._

def get_me_an_int(s: String) : Option[Int] = 
 Try(Some(Integer.parseInt(s))).getOrElse(None)

get_me_an_int("12345").map(_ % 2 == 0)
get_me_an_int("12u34").map(_ % 2 == 0)



val lst = List("12345", "foo", "5432", "bar", "x21", "456")
for (x <- lst) yield get_me_an_int(x)

// summing up all the numbers

lst.map(get_me_an_int).flatten.sum




// this is actually how for-comprehensions are
// defined in Scala

lst.map(n => square(n))
for (n <- lst) yield square(n)

// lets define our own higher-order functions
// type of functions is for example Int => Int


def my_map_int(lst: List[Int], f: Int => Int) : List[Int] = 
{
  if (lst == Nil) Nil
  else f(lst.head) :: my_map_int(lst.tail, f)
}

my_map_int(lst, square)

// same function using pattern matching: a kind
// of switch statement on steroids (see more later on)

def my_map_int(lst: List[Int], f: Int => Int) : List[Int] = 
  lst match {
    case Nil => Nil
    case x::xs => f(x)::my_map_int(xs, f)
  }



val biglst = (1 to 10000).toList
my_map_int(biglst, double)

(1 to 10000000).toList.map(double)

// other function types
//
// f1: (Int, Int) => Int
// f2: List[String] => Option[Int]
// ... 





// Map type (upper-case)
//=======================

// Note the difference between map and Map

val m = Map(1 -> "one", 2 -> "two", 10 -> "many")

List((1, "one"), (2, "two"), (10, "many")).toMap

m.get(1)
m.get(4)

m.getOrElse(1, "")
m.getOrElse(4, "")

val new_m = m + (10 -> "ten")

new_m.get(10)

val m2 = for ((k, v) <- m) yield (k, v.toUpperCase)



// groupBy function on Maps
val lst = List("one", "two", "three", "four", "five")
lst.groupBy(_.head)

lst.groupBy(_.length)

lst.groupBy(_.length).get(3)

val grps = lst.groupBy(_.length)
grps.keySet

// naive quicksort with "On" function

def sortOn(f: Int => Int, xs: List[Int]) : List[Int] = {
  if (xs.size < 2) xs
  else {
   val pivot = xs.head
   val (left, right) = xs.partition(f(_) < f(pivot))
   sortOn(f, left) ::: pivot :: sortOn(f, right.tail)
  }
} 

sortOn(identity, List(99,99,99,98,10,-3,2)) 
sortOn(n => - n, List(99,99,99,98,10,-3,2))



// Pattern Matching
//==================

// A powerful tool which has even landed in Java during 
// the last few years (https://inside.java/2021/06/13/podcast-017/).
// ...Scala already has it for many years and the concept is
// older than your friendly lecturer, that is stone old  ;o)

// The general schema:
//
//    expression match {
//       case pattern1 => expression1
//       case pattern2 => expression2
//       ...
//       case patternN => expressionN
//    }


// recall
def my_map_int(lst: List[Int], f: Int => Int) : List[Int] = 
  lst match {
    case Nil => Nil
    case x::xs => f(x)::my_map_int(xs, f)
  }

def my_map_option(opt: Option[Int], f: Int => Int) : Option[Int] = 
  opt match {
    case None => None
    case Some(x) => Some(f(x))
  }

my_map_option(None, x => x * x)
my_map_option(Some(8), x => x * x)


// you can also have cases combined
def season(month: String) : String = month match {
  case "March" | "April" | "May" => "It's spring"
  case "June" | "July" | "August" => "It's summer"
  case "September" | "October" | "November" => "It's autumn"
  case "December" => "It's winter"
  case "January" | "February" => "It's unfortunately winter"
  case _ => "Wrong month"
}

// pattern-match on integers

def fib(n: Int) : Int = n match { 
  case 0 | 1 => 1
  case n => fib(n - 1) + fib(n - 2)
}

fib(10)

// Silly: fizz buzz
def fizz_buzz(n: Int) : String = (n % 3, n % 5) match {
  case (0, 0) => "fizz buzz"
  case (0, _) => "fizz"
  case (_, 0) => "buzz"
  case _ => n.toString  
}

for (n <- 1 to 20) 
 println(fizz_buzz(n))


val lst = List(None, Some(1), Some(2), None, Some(3)).flatten

def my_flatten(xs: List[Option[Int]]): List[Int] = 
 xs match {
   case Nil => Nil 
   case None::rest => my_flatten(rest)
   case Some(v)::rest => v :: my_flatten(rest)
 }

my_flatten(List(None, Some(1), Some(2), None, Some(3)))




// Recursion
//===========

// my_length

def my_length(xs: List[Int]) : Int = {
  if (xs == Nil) 0 else 1 + my_length(xs.tail)
}

def my_sum(xs: List[Int]) : Int = {
  if (xs == Nil) 0 else xs.head + my_sum(xs.tail)
}

my_sum((1 to 100).toList)
my_length(List())

/* my_map */
for (n <- List[Int](1,2,3)) yield n * n

List(1,2,3) ::: List(3,4,5)
3 :: List(3,4,5)

def my_map(xs: List[Int], f : Int => Int ) : List[Int] = {
  if (xs == Nil) Nil 
  else f(xs.head) :: my_map(xs.tail, f)
}

def square(n: Int) : Int = n * n


my_map(List(1,2,3), square)




/* powerset */

def powerset(xs: Set[Int]) : Set[Set[Int]] = {
  if (xs == Set()) Set(Set())
  else powerset(xs.tail) ++ powerset(xs.tail).map(_ + xs.head)
}





/* on lists */
def powerset(xs: List[Int]) : List[List[Int]] = {
  if (xs == Nil) List(Nil)
  else powerset(xs.tail) ::: powerset(xs.tail).map(xs.head :: _)
}



/* Say you have characters a, b, c.
   What are all the combinations of a certain length?

   All combinations of length 2:
  
     aa, ab, ac, ba, bb, bc, ca, cb, cc

   Combinations of length 3:
   
     aaa, baa, caa, and so on......
*/

def combs(cs: List[Char], n: Int) : List[String] = {
  if (n == 0) List("")
  else for (c <- cs; s <- combs(cs, n - 1)) yield s"$c$s"
}

combs(List('a', 'b', 'c'), 3)



def combs(cs: List[Char], l: Int) : List[String] = {
  if (l == 0) List("")
  else for (c <- cs; s <- combs(cs, l - 1)) yield s"$c$s"
}

combs("abc".toList, 2)


// When writing recursive functions you have to
// think about three points
// 
// - How to start with a recursive function
// - How to communicate between recursive calls
// - Exit conditions



// A Recursive Web Crawler / Email Harvester
//===========================================
//
// the idea is to look for links using the
// regular expression "https?://[^"]*" and for
// email addresses using another regex.

import io.Source
import scala.util._

// gets the first 10K of a web-page
def get_page(url: String) : String = {
  Try(Source.fromURL(url)(using "ISO-8859-1").take(10000).mkString).
    getOrElse { println(s"  Problem with: $url"); ""}
}

// regex for URLs and emails
val http_pattern = """"https?://[^"]*"""".r
val email_pattern = """([a-z0-9_\.-]+)@([\da-z\.-]+)\.([a-z\.]{2,6})""".r

//test case:
//email_pattern.findAllIn
//  ("foo bla christian@kcl.ac.uk 1234567").toList


// drops the first and last character from a string
def unquote(s: String) = s.drop(1).dropRight(1)

def get_all_URLs(page: String): Set[String] = 
  http_pattern.findAllIn(page).map(unquote).toSet

// naive version of crawl - searches until a given depth,
// visits pages potentially more than once
def crawl(url: String, n: Int) : Unit = {
  if (n == 0) ()
  else {
    println(s"  Visiting: $n $url")
    for (u <- get_all_URLs(get_page(url))) crawl(u, n - 1)
  }
}

// some starting URLs for the crawler
//val startURL = """https://nms.kcl.ac.uk/christian.urban/"""
val startURL = """https://urbanchr.github.io/"""


crawl(startURL, 2)


// a primitive email harvester
def emails(url: String, n: Int) : Set[String] = {
  if (n == 0) Set()
  else {
    println(s"  Visiting: $n $url")
    val page = get_page(url)
    val new_emails = email_pattern.findAllIn(page).toSet
    new_emails ++ (for (u <- get_all_URLs(page)) yield emails(u, n - 1)).flatten
  }
}

emails(startURL, 3)


// if we want to explore the internet "deeper", then we
// first have to parallelise the request of webpages:
//
// scala -cp scala-parallel-collections_2.13-0.2.0.jar 
// import scala.collection.parallel.CollectionConverters._



def powerset(xs: Set[Int]) : Set[Set[Int]] = {
  if (xs == Set()) Set(Set())
  else {
    val ps = powerset(xs.tail)  
    ps ++ ps.map(_ + xs.head)
  }
}  

def psubsets(xs: Set[Int]) = 
  powerset(xs) -- Set(Set(), xs) 

//def psubsets(xs: Set[Int]) = 
//  xs.subsets.toList -- Set(Set(), xs)  

def splits(xs: Set[Int]) :  Set[(Set[Int], Set[Int])] =
  psubsets(xs).map(s => (s, xs -- s))



enum Tree {
  case Num(i: Int)
  case Add(l: Tree, r: Tree)
  case Sub(l: Tree, r: Tree)
  case Mul(l: Tree, r: Tree)
  case Div(l: Tree, r: Tree)
}
import Tree._

def pp(tr: Tree) : String = tr match {
  case Num(n) => s"$n"
  case Add(l, r) => s"(${pp(l)} + ${pp(r)})"
  case Sub(l, r) => s"(${pp(l)} - ${pp(r)})"
  case Mul(l, r) => s"(${pp(l)} * ${pp(r)})"
  case Div(l, r) => s"(${pp(l)} / ${pp(r)})"
}

def search(nums: Set[Int]) : Set[Tree] =  nums.size match {
  case 0 => Set()
  case 1 => Set(Num(nums.head))
  case 2 => {
    val l = nums.head
    val r = nums.tail.head
    Set(Add(Num(l), Num(r)), 
        Mul(Num(l), Num(r)))
        ++ Option.when(l <= r)(Sub(Num(r), Num(l)))
        ++ Option.when(l > r)(Sub(Num(l), Num(r)))
        ++ Option.when(r > 0 && l % r == 0)(Div(Num(l), Num(r)))
        ++ Option.when(l > 0 && r % l == 0)(Div(Num(r), Num(l)))
  }
  case xs => {
    val spls = splits(nums)
    val subtrs = 
      for ((lspls, rspls) <- spls;
           lt <- search(lspls); 
          rt <- search(rspls)) yield {
        Set(Add(lt, rt), Sub(lt, rt),
            Mul(lt, rt), Div(lt, rt))
    } 
    subtrs.flatten
  }
}

println(search(Set(1,2,3,4)).mkString("\n"))

def eval(tr: Tree) : Option[Int] = tr match {
  case Num(n) => Some(n)
  case Add(l, r) => 
    for (rl <- eval(l); rr <- eval(r)) yield rl + rr
  case Mul(l, r) => 
    for (rl <- eval(l); rr <- eval(r)) yield rl * rr  
  case Sub(l, r) => 
    for (rl <- eval(l); rr <- eval(r);
         if 0 <= rl - rr) yield rl - rr   
  case Div(l, r) => 
    for (rl <- eval(l); rr <- eval(r);
         if rr > 0 && rl % rr == 0) yield rl / rr          
}

eval(Add(Num(1), Num(2)))
eval(Mul(Add(Num(1), Num(2)), Num(4)))
eval(Sub(Num(3), Num(2)))
eval(Sub(Num(3), Num(6)))
eval(Div(Num(6), Num(2)))
eval(Div(Num(6), Num(4)))

def time_needed[T](n: Int, code: => T) = {
  val start = System.nanoTime()
  for (i <- (0 to n)) code
  val end = System.nanoTime()
  (end - start) / 1.0e9
}


import scala.collection.parallel.CollectionConverters._

def check(xs: Set[Int], target: Int) =
  search(xs).find(eval(_) == Some(target))

for (sol <- check(Set(50, 5, 4, 9, 10, 8), 560)) {
  println(s"${pp(sol)} => ${eval(sol)}")
}



time_needed(1, check(Set(50, 5, 4, 9, 10, 8), 560))


println(check(Set(25, 5, 2, 10, 7, 1), 986).mkString("\n"))

for (sol <- check(Set(25, 5, 2, 10, 7, 1), 986)) {
  println(s"${pp(sol)} => ${eval(sol)}")
}

for (sol <- check(Set(25, 5, 2, 10, 7, 1), -1)) {
  println(s"${pp(sol)} => ${eval(sol)}")
}

for (sol <- check(Set(100, 25, 75, 50, 7, 10), 360)) {
  println(s"${pp(sol)} => ${eval(sol)}")
}
time_needed(1, check(Set(100, 25, 75, 50, 7, 10), 360))



time_needed(1, check(Set(25, 5, 2, 10, 7, 1), 986))
time_needed(1, check(Set(25, 5, 2, 10, 7, 1), -1))


def generate(nums: Set[Int]) : Set[(Tree, Int)] =  nums.size match {
  case 0 => Set()
  case 1 => Set((Num(nums.head), nums.head))
  case xs => {
    val spls = splits(nums)
    val subtrs =
      for ((lspls, rspls) <- spls;
           (lt, ln) <- generate(lspls); 
           (rt, rn) <- generate(rspls)) yield {
        Set((Add(lt, rt), ln + rn),
            (Mul(lt, rt), ln * rn))
        ++ Option.when(ln <= rn)((Sub(rt, lt), rn - ln)) 
        ++ Option.when(ln > rn)((Sub(lt, rt), ln - rn))
        ++ Option.when(rn > 0 && ln % rn == 0)((Div(lt, rt), ln / rn))
        ++ Option.when(ln > 0 && rn % ln == 0)((Div(rt, lt), rn / ln))
    } 
    subtrs.flatten
  }
}

def check2(xs: Set[Int], target: Int) =
  generate(xs).find(_._2 == target)

for ((sol, ev) <- check2(Set(50, 5, 4, 9, 10, 8), 560)) {
  println(s"${pp(sol)} => ${eval(sol)} / $ev")
}

time_needed(1, check(Set(50, 5, 4, 9, 10, 8), 560))
time_needed(1, check2(Set(50, 5, 4, 9, 10, 8), 560))

time_needed(1, check(Set(50, 5, 4, 9, 10, 8), -1))
time_needed(1, check2(Set(50, 5, 4, 9, 10, 8), -1))

// Jumping Towers
//================


def moves(xs: List[Int], n: Int) : List[List[Int]] = 
 (xs, n) match {
   case (Nil, _) => Nil
   case (_, 0) => Nil
   case (x::xs, n) => (x::xs) :: moves(xs, n - 1)
 }

// List(5,5,1,0) -> moves(List(5,1,0), 5)
moves(List(5,1,0), 1)
moves(List(5,1,0), 2)
moves(List(5,1,0), 5)

// checks whether a jump tour exists at all

def search(xs: List[Int]) : Boolean = xs match {
  case Nil => true
  case x::xs =>
    if (xs.length < x) true 
    else moves(xs, x).exists(search(_))
}


search(List(5,3,2,5,1,1))
search(List(3,5,1,0,0,0,1))
search(List(3,5,1,0,0,0,0,1))
search(List(3,5,1,0,0,0,1,1))
search(List(3,5,1))
search(List(5,1,1))
search(Nil)
search(List(1))
search(List(5,1,1))
search(List(3,5,1,0,0,0,0,0,0,0,0,1))


import scala.util._
List.fill(100)(Random.nextInt(2))
search(List.fill(100)(Random.nextInt(10)))

// generate *all* jump tours
//    if we are only interested in the shortes one, we could
//    shortcircut the calculation and only return List(x) in
//    case where xs.length < x, because no tour can be shorter
//    than 1
// 

def jumps(xs: List[Int]) : List[List[Int]] = xs match {
  case Nil => Nil
  case x::xs => {
    val children = moves(xs, x)
    val results = 
      children.map(cs => jumps(cs).map(x :: _)).flatten
    if (xs.length < x) List(x) :: results else results
  }
}

jumps(List(3,5,1,2,1,2,1))
jumps(List(3,5,1,2,3,4,1))
jumps(List(3,5,1,0,0,0,1))
jumps(List(3,5,1))
jumps(List(5,1,1))
jumps(Nil)
jumps(List(1))
jumps(List(5,1,2))
moves(List(1,2), 5)
jumps(List(1,5,1,2))
jumps(List(3,5,1,0,0,0,0,0,0,0,0,1))

jumps(List(5,3,2,5,1,1)).minBy(_.length)
jumps(List(1,3,5,8,9,2,6,7,6,8,9)).minBy(_.length)
jumps(List(1,3,6,1,0,9)).minBy(_.length)
jumps(List(2,3,1,1,2,4,2,0,1,1)).minBy(_.length)


