// Part 1 about finding Knight's tours
//=====================================

type Pos = (Int, Int)    // a position on a chessboard 
type Path = List[Pos]    // a path...a list of positions

// for measuring time
def time_needed[T](code: => T) : T = {
  val start = System.nanoTime()
  val result = code
  val end = System.nanoTime()
  println(f"Time needed: ${(end - start) / 1.0e9}%3.3f secs.")
  result
}

// for printing a board
def print_board(dim: Int, path: Path): Unit = {
  println
  for (i <- 0 until dim) {
    for (j <- 0 until dim) {
      print(f"${path.reverse.indexOf((j, dim - i - 1))}%3.0f ")
    }
    println
  } 
}


// 1 mark

def is_legal(dim: Int, path: Path, x: Pos): Boolean = 
  0 <= x._1 && 0 <= x._2 && x._1 < dim && x._2 < dim && !path.contains(x)

assert(is_legal(8, Nil, (3, 4)) == true)
assert(is_legal(8, List((4, 1), (1, 0)), (4, 1)) == false)
assert(is_legal(2, Nil, (0, 0)) == true)


def add_pair(x: Pos, y: Pos): Pos = 
  (x._1 + y._1, x._2 + y._2)

def moves(x: Pos): List[Pos] = 
  List(( 1,  2),( 2,  1),( 2, -1),( 1, -2),
       (-1, -2),(-2, -1),(-2,  1),(-1,  2)).map(add_pair(x, _))

// 1 mark

def legal_moves(dim: Int, path: Path, x: Pos): List[Pos] = 
  moves(x).filter(is_legal(dim, path, _))

assert(legal_moves(8, Nil, (2,2)) == 
  List((3,4), (4,3), (4,1), (3,0), (1,0), (0,1), (0,3), (1,4)))
assert(legal_moves(8, Nil, (7,7)) == List((6,5), (5,6)))
assert(legal_moves(8, List((4,1), (1,0)), (2,2)) == 
  List((3,4), (4,3), (3,0), (0,1), (0,3), (1,4)))
assert(legal_moves(8, List((6,6)), (7,7)) == List((6,5), (5,6)))
assert(legal_moves(1, Nil, (0,0)) == List())
assert(legal_moves(2, Nil, (0,0)) == List())
assert(legal_moves(3, Nil, (0,0)) == List((1,2), (2,1)))

// 2 marks

def count_tours(dim: Int, path: Path): Int = {
  if (path.length == dim * dim) 1
  else 
    (for (x <- legal_moves(dim, path, path.head)) yield count_tours(dim, x::path)).sum
}

def enum_tours(dim: Int, path: Path): List[Path] = {
  if (path.length == dim * dim) List(path)
  else 
    (for (x <- legal_moves(dim, path, path.head)) yield enum_tours(dim, x::path)).flatten
}

// as far as tasks go
// test cases for CW



def count_all_tours(dim: Int) = {
  for (i <- (0 until dim).toList; 
       j <- (0 until dim).toList) yield count_tours(dim, List((i, j)))
}

def enum_all_tours(dim: Int): List[Path] = {
  (for (i <- (0 until dim).toList; 
        j <- (0 until dim).toList) yield enum_tours(dim, List((i, j)))).flatten
}


println("Number of tours starting from (0, 0)")

for (dim <- 1 to 5) {
  println(s"${dim} x ${dim} " + time_needed(0, count_tours(dim, List((0, 0)))))
}

println("Number of tours starting from all fields")

for (dim <- 1 to 5) {
  println(s"${dim} x ${dim} " + time_needed(0, count_all_tours(dim)))
}

for (dim <- 1 to 5) {
  val ts = enum_tours(dim, List((0, 0)))
  println(s"${dim} x ${dim} ")   
  if (ts != Nil) {
    print_board(dim, ts.head)
    println(ts.head)
  }
}


// 1 mark

def first(xs: List[Pos], f: Pos => Option[Path]): Option[Path] = xs match {
  case Nil => None
  case x::xs => {
    val result = f(x)
    if (result.isDefined) result else first(xs, f)
  }
}

// test cases
def foo(x: (Int, Int)) = if (x._1 > 3) Some(List(x)) else None

first(List((1, 0),(2, 0),(3, 0),(4, 0)), foo)
first(List((1, 0),(2, 0),(3, 0)), foo)


// 1 mark

def first_tour(dim: Int, path: Path): Option[Path] = {
  if (path.length == dim * dim) Some(path)
  else
    first(legal_moves(dim, path, path.head), (x: Pos) => first_tour(dim, x::path))
}



/*
for (dim <- 1 to 8) {
  val t = first_tour(dim, List((0, 0)))
  println(s"${dim} x ${dim} " + (if (t == None) "" else { print_board(dim, t.get) ; "" }))
}
*/

// 15 secs for 8 x 8
val ts1 = time_needed(0,first_tour(8, List((0, 0))).get)

// no result for 4 x 4
val ts2 = time_needed(0, first_tour(4, List((0, 0))))

// 0.3 secs for 6 x 6
val ts3 = time_needed(0, first_tour(6, List((0, 0))))

// 15 secs for 8 x 8
time_needed(0, print_board(8, first_tour(8, List((0, 0))).get))

