// Scala Lecture 4
//=================

// pattern-matching
// tail-recursion
// polymorphic types



// Pattern Matching
//==================

// A powerful tool which has even landed in Java during 
// the last few years (https://inside.java/2021/06/13/podcast-017/).
// ...Scala already has it for many years and the concept is
// older than your friendly lecturer, that is stone old  ;o)

// The general schema:
//
//    expression match {
//       case pattern1 => expression1
//       case pattern2 => expression2
//       ...
//       case patternN => expressionN
//    }


// recall
def len(xs: List[Int]) : Int = {
    if (xs == Nil) 0
    else 1 + len(xs.tail)
}    

def len(xs: List[Int]) : Int = xs match {
    case Nil => 0
    case _::xs => 1 + len(xs)
}  

len(Nil)
len(List(1,2,3,4))


List(1,2,3,4).map(x => x * x)

def my_map_int(lst: List[Int], f: Int => Int) : List[Int] = 
  lst match {
    case Nil => Nil
    case foo::xs => f(foo) :: my_map_int(xs, f)
  }

def my_map_option(opt: Option[Int], f: Int => Int) : Option[Int] = 
  opt match {
    case None => None
    case Some(x) => {
      Some(f(x))
    }
  }

my_map_option(None, x => x * x)
my_map_option(Some(8), x => x * x)


// you can also have cases combined
def season(month: String) : String = month match {
  case "March" | "April" | "May" => "It's spring"
  case "June" | "July" | "August" => "It's summer"
  case "September" | "October" | "November" => "It's autumn"
  case "December" => "It's winter"
  case "January" | "February" => "It's unfortunately winter"
  case _ => "Wrong month"
}

// pattern-match on integers

def fib(n: Int) : Int = n match { 
  case 0 | 1 => 1
  case _ => fib(n - 1) + fib(n - 2)
}

fib(10)

// pattern-match on results

// Silly: fizz buzz
def fizz_buzz(n: Int) : String = (n % 3, n % 5) match {
  case (0, 0) => "fizz buzz"
  case (0, _) => "fizz"
  case (_, 0) => "buzz"
  case _ => n.toString  
}

for (n <- 1 to 20) 
 println(fizz_buzz(n))

// guards in pattern-matching

def foo(xs: List[Int]) : String = xs match {
  case Nil => s"this list is empty"
  case x :: xs if x % 2 == 0 
     => s"the first elemnt is even"
  case x if len(x) ==
     => s"this list has exactly two elements"   
  case x :: y :: rest if x == y
     => s"this has two elemnts that are the same"
  case hd :: tl => s"this list is standard $hd::$tl"
}

foo(Nil)
foo(List(1,2,3))
foo(List(1,1))
foo(List(1,1,2,3))
foo(List(2,2,2,3))




abstract class Colour
case object Red extends Colour 
case object Green extends Colour 
case object Blue extends Colour
case object Yellow extends Colour


def fav_colour(c: Colour) : Boolean = c match {
  case Green => true
  case Red => true
  case _  => false 
}

fav_colour(Blue)


// ... a tiny bit more useful: Roman Numerals

sealed abstract class RomanDigit 
case object I extends RomanDigit 
case object V extends RomanDigit 
case object X extends RomanDigit 
case object L extends RomanDigit 
case object C extends RomanDigit 
case object D extends RomanDigit 
case object M extends RomanDigit 

type RomanNumeral = List[RomanDigit] 

List(I, M,C,D,X,X,V,I,I, A)

/*
I    -> 1
II   -> 2
III  -> 3
IV   -> 4
V    -> 5
VI   -> 6
VII  -> 7
VIII -> 8
IX   -> 9
X    -> 10
*/

def RomanNumeral2Int(rs: RomanNumeral): Int = rs match { 
  case Nil => 0
  case M::r    => 1000 + RomanNumeral2Int(r)  
  case C::M::r => 900 + RomanNumeral2Int(r)
  case D::r    => 500 + RomanNumeral2Int(r)
  case C::D::r => 400 + RomanNumeral2Int(r)
  case C::r    => 100 + RomanNumeral2Int(r)
  case X::C::r => 90 + RomanNumeral2Int(r)
  case L::r    => 50 + RomanNumeral2Int(r)
  case X::L::r => 40 + RomanNumeral2Int(r)
  case X::r    => 10 + RomanNumeral2Int(r)
  case I::X::r => 9 + RomanNumeral2Int(r)
  case V::r    => 5 + RomanNumeral2Int(r)
  case I::V::r => 4 + RomanNumeral2Int(r)
  case I::r    => 1 + RomanNumeral2Int(r)
}

RomanNumeral2Int(List(I,V))             // 4
RomanNumeral2Int(List(I,I,I,I))         // 4 (invalid Roman number)
RomanNumeral2Int(List(V,I))             // 6
RomanNumeral2Int(List(I,X))             // 9
RomanNumeral2Int(List(M,C,M,L,X,X,I,X)) // 1979
RomanNumeral2Int(List(M,M,X,V,I,I))     // 2017

abstract class Tree
case class Leaf(x: Int)
case class Branch(tl: Tree, tr: Tree)


abstract class Rexp
case object ZERO extends Rexp                      // matches nothing
case object ONE extends Rexp                       // matches the empty string
case class CHAR(c: Char) extends Rexp              // matches a character c
case class ALT(r1: Rexp, r2: Rexp) extends Rexp    // alternative
case class SEQ(r1: Rexp, r2: Rexp) extends Rexp    // sequence
case class STAR(r: Rexp) extends Rexp              // star

def depth(r: Rexp) : Int = r match {
  case ZERO => 1
  case ONE => 1
  case CHAR(_) => 1
  case ALT(r1, r2) => 1 + List(depth(r1), depth(r2)).max
  case SEQ(r1, r2) => 1 + List(depth(r1), depth(r2)).max
  case STAR(r1) => 1 + depth(r1)
}


// Trees (example of an Algebraic Datatype)


abstract class Tree
case class Leaf(x: Int) extends Tree
case class Node(s: String, left: Tree, right: Tree) extends Tree 

val lf = Leaf(20)
val tr = Node("foo", Leaf(10), Leaf(23))

val lst : List[Tree] = List(lf, tr)



// expressions (essentially trees)

sealed abstract class Exp
case class N(n: Int) extends Exp                  // for numbers
case class Plus(e1: Exp, e2: Exp) extends Exp
case class Times(e1: Exp, e2: Exp) extends Exp

def string(e: Exp) : String = e match {
  case N(n) => s"$n"
  case Plus(e1, e2) => s"(${string(e1)} + ${string(e2)})" 
  case Times(e1, e2) => s"(${string(e1)} * ${string(e2)})"
}

val e = Plus(N(9), Times(N(3), N(4)))
println(e.toString)
println(string(e))

def eval(e: Exp) : Int = e match {
  case N(n) => n
  case Plus(e1, e2) => eval(e1) + eval(e2) 
  case Times(e1, e2) => eval(e1) * eval(e2) 
}

println(eval(e))

// simplification rules:
// e + 0, 0 + e => e 
// e * 0, 0 * e => 0
// e * 1, 1 * e => e
//
// (....9 ....)

def simp(e: Exp) : Exp = e match {
  case N(n) => N(n)
  case Plus(e1, e2) => (simp(e1), simp(e2)) match {
    case (N(0), e2s) => e2s
    case (e1s, N(0)) => e1s
    case (e1s, e2s) => Plus(e1s, e2s)
  }  
  case Times(e1, e2) => (simp(e1), simp(e2)) match {
    case (N(0), _) => N(0)
    case (_, N(0)) => N(0)
    case (N(1), e2s) => e2s
    case (e1s, N(1)) => e1s
    case (e1s, e2s) => Times(e1s, e2s)
  }  
}


val e2 = Times(Plus(N(0), N(1)), Plus(N(0), N(9)))
println(string(e2))
println(string(simp(e2)))







// Tokens and Reverse Polish Notation
abstract class Token
case class T(n: Int) extends Token
case object PL extends Token
case object TI extends Token

// transfroming an Exp into a list of tokens
def rp(e: Exp) : List[Token] = e match {
  case N(n) => List(T(n))
  case Plus(e1, e2) => rp(e1) ::: rp(e2) ::: List(PL) 
  case Times(e1, e2) => rp(e1) ::: rp(e2) ::: List(TI) 
}
println(string(e2))
println(rp(e2))

def comp(ls: List[Token], st: List[Int] = Nil) : Int = (ls, st) match {
  case (Nil, st) => st.head 
  case (T(n)::rest, st) => comp(rest, n::st)
  case (PL::rest, n1::n2::st) => comp(rest, n1 + n2::st)
  case (TI::rest, n1::n2::st) => comp(rest, n1 * n2::st)
}

comp(rp(e))

def proc(s: String) : Token = s match {
  case  "+" => PL
  case  "*" => TI
  case  _ => T(s.toInt) 
}

comp("1 2 + 4 * 5 + 3 +".split(" ").toList.map(proc), Nil)


// Tail recursion
//================

def fact(n: BigInt): BigInt = 
  if (n == 0) 1 else n * fact(n - 1)


fact(10)          
fact(1000)        
fact(100000)       


def factT(n: BigInt, acc: BigInt): BigInt =
  if (n == 0) acc else factT(n - 1, n * acc)


factT(10, 1)
println(factT(100000, 1))


// there is a flag for ensuring a function is tail recursive
import scala.annotation.tailrec

@tailrec
def factT(n: BigInt, acc: BigInt): BigInt =
  if (n == 0) acc else factT(n - 1, n * acc)

factT(100000, 1)

// for tail-recursive functions the Scala compiler
// generates loop-like code, which does not need
// to allocate stack-space in each recursive
// call; Scala can do this only for tail-recursive
// functions

// Moral: Whenever a recursive function is resource-critical
// (i.e. works with a large recursion depth), then you need to
// write it in tail-recursive fashion.
// 
// Unfortuantely, Scala because of current limitations in 
// the JVM is not as clever as other functional languages. It can 
// only optimise "self-tail calls". This excludes the cases of 
// multiple functions making tail calls to each other. Well,
// nothing is perfect. 



// Polymorphic Types
//===================

// You do not want to write functions like contains, first, 
// length and so on for every type of lists.

def length_int_list(lst: List[Int]): Int = lst match {
  case Nil => 0
  case _::xs => 1 + length_int_list(xs)
}

length_int_list(List(1, 2, 3, 4))

def length_string_list(lst: List[String]): Int = lst match {
  case Nil => 0
  case _::xs => 1 + length_string_list(xs)
}

length_string_list(List("1", "2", "3", "4"))


// you can make the function parametric in type(s)

def length[A](lst: List[A]): Int = lst match {
  case Nil => 0
  case x::xs => 1 + length(xs)
}
length(List("1", "2", "3", "4"))
length(List(1, 2, 3, 4))


length[String](List(1, 2, 3, 4))


def map[A, B](lst: List[A], f: A => B): List[B] = lst match {
  case Nil => Nil
  case x::xs => f(x)::map(xs, f) 
}

map(List(1, 2, 3, 4), (x: Int) => x.toString)


// should be
def first[A, B](xs: List[A], f: A => Option[B]) : Option[B] = ???

// Type inference is local in Scala

def id[T](x: T) : T = x

val x = id(322)          // Int
val y = id("hey")        // String
val z = id(Set(1,2,3,4)) // Set[Int]


// The type variable concept in Scala can get really complicated.
//
// - variance (OO)
// - bounds (subtyping)
// - quantification

// Java has issues with this too: Java allows
// to write the following incorrect code, and
// only recovers by raising an exception
// at runtime.

// Object[] arr = new Integer[10];
// arr[0] = "Hello World";


// Scala gives you a compile-time error, which
// is much better.

var arr = Array[Int]()
arr(0) = "Hello World"




// Function definitions again
//============================

// variable arguments

def printAll(strings: String*) = {
  strings.foreach(println)
}

printAll()
printAll("foo")
printAll("foo", "bar")
printAll("foo", "bar", "baz")

// pass a list to the varargs field
val fruits = List("apple", "banana", "cherry")

printAll(fruits: _*)


// you can also implement your own string interpolations
import scala.language.implicitConversions
import scala.language.reflectiveCalls

implicit def sring_inters(sc: StringContext) = new {
    def i(args: Any*): String = s"${sc.s(args:_*)}\n"
}

i"add ${3+2} ${3 * 3}" 


// default arguments

def length[A](xs: List[A]) : Int = xs match {
  case Nil => 0
  case _ :: tail => 1 + length(tail)
}

def lengthT[A](xs: List[A], acc : Int = 0) : Int = xs match {
  case Nil => acc
  case _ :: tail => lengthT(tail, 1 + acc)
}

lengthT(List.fill(100000)(1))


def fact(n: BigInt, acc: BigInt = 1): BigInt =
  if (n == 0) acc else fact(n - 1, n * acc)

fact(10)



// currying    (Haskell Curry)

def add(x: Int, y: Int) = x + y

List(1,2,3,4,5).map(x => add(3, x))

def add2(x: Int)(y: Int) = x + y

List(1,2,3,4,5).map(add2(3))

val a3 : Int => Int = add2(3)

// currying helps sometimes with type inference

def find[A](xs: List[A])(pred: A => Boolean): Option[A] = {
  xs match {
    case Nil => None
    case hd :: tl =>
      if (pred(hd)) Some(hd) else find(tl)(pred)
  }
}

find(List(1, 2, 3))(x => x % 2 == 0)

// Source.fromURL(url)(encoding)
// Source.fromFile(name)(encoding)







// Sudoku 
//========

// THE POINT OF THIS CODE IS NOT TO BE SUPER
// EFFICIENT AND FAST, just explaining exhaustive
// depth-first search


val game0 = """.14.6.3..
              |62...4..9
              |.8..5.6..
              |.6.2....3
              |.7..1..5.
              |5....9.6.
              |..6.2..3.
              |1..5...92
              |..7.9.41.""".stripMargin.replaceAll("\\n", "")



type Pos = (Int, Int)
val EmptyValue = '.'
val MaxValue = 9

def pretty(game: String): String = 
  "\n" + (game.grouped(MaxValue).mkString("\n"))

pretty(game0)


val allValues = "123456789".toList
val indexes = (0 to 8).toList

def empty(game: String) = game.indexOf(EmptyValue)
def isDone(game: String) = empty(game) == -1 
def emptyPosition(game: String) = {
  val e = empty(game)
  (e % MaxValue, e / MaxValue)
}

def get_row(game: String, y: Int) = 
  indexes.map(col => game(y * MaxValue + col))
def get_col(game: String, x: Int) = 
  indexes.map(row => game(x + row * MaxValue))

//get_row(game0, 0)
//get_row(game0, 1)
//get_col(game0, 0)

def get_box(game: String, pos: Pos): List[Char] = {
    def base(p: Int): Int = (p / 3) * 3
    val x0 = base(pos._1)
    val y0 = base(pos._2)
    val ys = (y0 until y0 + 3).toList
    (x0 until x0 + 3).toList
      .flatMap(x => ys.map(y => game(x + y * MaxValue)))
}


//get_box(game0, (3, 1))


// this is not mutable!!
def update(game: String, pos: Int, value: Char): String = 
  game.updated(pos, value)

def toAvoid(game: String, pos: Pos): List[Char] = 
  (get_col(game, pos._1) ++ 
   get_row(game, pos._2) ++ 
   get_box(game, pos))

def candidates(game: String, pos: Pos): List[Char] = 
  allValues.diff(toAvoid(game, pos))

//candidates(game0, (0,0))


def search(game: String): List[String] = {
  if (isDone(game)) List(game)
  else {
    val cs = candidates(game, emptyPosition(game))
    cs.map(c => search(update(game, empty(game), c))).flatten
  }
}

pretty(game0)
search(game0).map(pretty)

val game1 = """23.915...
              |...2..54.
              |6.7......
              |..1.....9
              |89.5.3.17
              |5.....6..
              |......9.5
              |.16..7...
              |...329..1""".stripMargin.replaceAll("\\n", "")

search(game1).map(pretty)

// a game that is in the hard category
val game2 = """8........
              |..36.....
              |.7..9.2..
              |.5...7...
              |....457..
              |...1...3.
              |..1....68
              |..85...1.
              |.9....4..""".stripMargin.replaceAll("\\n", "")

search(game2).map(pretty)

// game with multiple solutions
val game3 = """.8...9743
              |.5...8.1.
              |.1.......
              |8....5...
              |...8.4...
              |...3....6
              |.......7.
              |.3.5...8.
              |9724...5.""".stripMargin.replaceAll("\\n", "")

search(game3).map(pretty).foreach(println)

// for measuring time
def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  s"${(end - start) / 1.0e9} secs"
}

time_needed(1, search(game2))



// tail recursive version that searches 
// for all Sudoku solutions
import scala.annotation.tailrec

@tailrec
def searchT(games: List[String], sols: List[String]): List[String] = 
 games match {
    case Nil => sols
    case game::rest => {
      if (isDone(game)) searchT(rest, game::sols)
      else {
        val cs = candidates(game, emptyPosition(game))
        searchT(cs.map(c => update(game, empty(game), c)) ::: rest, sols)
     }
   }
 }

searchT(List(game3), List()).map(pretty)


// tail recursive version that searches 
// for a single solution

def search1T(games: List[String]): Option[String] = games match {
  case Nil => None
  case game::rest => {
    if (isDone(game)) Some(game)
    else {
      val cs = candidates(game, emptyPosition(game))
      search1T(cs.map(c => update(game, empty(game), c)) ::: rest)
    }
  }
}

search1T(List(game3)).map(pretty)
time_needed(1, search1T(List(game3)))
time_needed(1, search1T(List(game2)))

// game with multiple solutions
val game3 = """.8...9743
              |.5...8.1.
              |.1.......
              |8....5...
              |...8.4...
              |...3....6
              |.......7.
              |.3.5...8.
              |9724...5.""".stripMargin.replaceAll("\\n", "")

searchT(List(game3), Nil).map(pretty)
search1T(List(game3)).map(pretty)






// Cool Stuff in Scala
//=====================


// Implicits or How to Pimp your Library
//======================================
//
// For example adding your own methods to Strings:
// Imagine you want to increment strings, like
//
//     "HAL".increment
//
// you can avoid ugly fudges, like a MyString, by
// using implicit conversions.

print("\n")
print("""\n""")

implicit class MyString(s: String) {
  def increment = s.map(c => (c + 1).toChar) 
}

"HAL".increment


// Abstract idea:
// In that version implicit conversions were used to solve the 
// late extension problem; namely, given a class C and a class T, 
// how to have C extend T without touching or recompiling C. 
// Conversions add a wrapper when a member of T is requested 
// from an instance of C.



import scala.concurrent.duration.{TimeUnit,SECONDS,MINUTES}

case class Duration(time: Long, unit: TimeUnit) {
  def +(o: Duration) = 
    Duration(time + unit.convert(o.time, o.unit), unit)
}

implicit class Int2Duration(that: Int) {
  def seconds = Duration(that, SECONDS)
  def minutes = Duration(that, MINUTES)
}

5.seconds + 2.minutes   //Duration(125L, SECONDS )
2.minutes + 60.seconds




// Regular expressions - the power of DSLs in Scala
//==================================================

abstract class Rexp
case object ZERO extends Rexp                     // nothing
case object ONE extends Rexp                      // the empty string
case class CHAR(c: Char) extends Rexp             // a character c
case class ALT(r1: Rexp, r2: Rexp) extends Rexp   // alternative  r1 + r2
case class SEQ(r1: Rexp, r2: Rexp) extends Rexp   // sequence     r1 . r2  
case class STAR(r: Rexp) extends Rexp             // star         r*



// writing (ab)* in the format above is 
// tedious
val r0 = STAR(SEQ(CHAR('a'), CHAR('b')))


// some convenience for typing in regular expressions
import scala.language.implicitConversions    
import scala.language.reflectiveCalls 

def charlist2rexp(s: List[Char]): Rexp = s match {
  case Nil => ONE
  case c::Nil => CHAR(c)
  case c::s => SEQ(CHAR(c), charlist2rexp(s))
}

implicit def string2rexp(s: String): Rexp = 
  charlist2rexp(s.toList)

val r1 = STAR("ab")
val r2 = STAR("hello") | STAR("world")


implicit def RexpOps (r: Rexp) = new {
  def | (s: Rexp) = ALT(r, s)
  def % = STAR(r)
  def ~ (s: Rexp) = SEQ(r, s)
}

implicit def stringOps (s: String) = new {
  def | (r: Rexp) = ALT(s, r)
  def | (r: String) = ALT(s, r)
  def % = STAR(s)
  def ~ (r: Rexp) = SEQ(s, r)
  def ~ (r: String) = SEQ(s, r)
}

//example regular expressions


val digit = ("0" | "1" | "2" | "3" | "4" | 
              "5" | "6" | "7" | "8" | "9")
val sign = "+" | "-" | ""
val number = sign ~ digit ~ digit.% 




// In mandelbrot.scala I used complex (imaginary) numbers 
// and implemented the usual arithmetic operations for complex 
// numbers.

case class Complex(re: Double, im: Double) { 
  // represents the complex number re + im * i
  def +(that: Complex) = Complex(this.re + that.re, this.im + that.im)
  def -(that: Complex) = Complex(this.re - that.re, this.im - that.im)
  def *(that: Complex) = Complex(this.re * that.re - this.im * that.im,
                                 this.re * that.im + that.re * this.im)
  def *(that: Double) = Complex(this.re * that, this.im * that)
  def abs = Math.sqrt(this.re * this.re + this.im * this.im)
}

val test = Complex(1, 2) + Complex (3, 4)


// ...to allow the notation n + m * i
import scala.language.implicitConversions   

val i = Complex(0, 1)
implicit def double2complex(re: Double) = Complex(re, 0)

val inum1 = -2.0 + -1.5 * i
val inum2 =  1.0 +  1.5 * i
