// Part 3 about finding a single tour using the Warnsdorf Rule
//=============================================================


type Pos = (Int, Int)
type Path = List[Pos]

// for measuring time
def time_needed[T](n: Int, code: => T) : T = {
  val start = System.nanoTime()
  for (i <- 0 until n) code
  val result = code
  val end = System.nanoTime()
  println(f"Time needed: ${(end - start) / 1.0e9}%3.3f secs.")
  result
}

def print_board(dim: Int, path: Path): Unit = {
  println
  for (i <- 0 until dim) {
    for (j <- 0 until dim) {
      print(f"${path.reverse.indexOf((i, j))}%4.0f ")
    }
    println
  } 
}

def add_pair(x: Pos, y: Pos): Pos = 
  (x._1 + y._1, x._2 + y._2)

def is_legal(dim: Int, path: Path, x: Pos): Boolean = 
  0 <= x._1 && 0 <= x._2 && x._1 < dim && x._2 < dim && !path.contains(x)

def moves(x: Pos): List[Pos] = 
  List(( 1,  2),( 2,  1),( 2, -1),( 1, -2),
       (-1, -2),(-2, -1),(-2,  1),(-1,  2)).map(add_pair(x, _))

def legal_moves(dim: Int, path: Path, x: Pos): List[Pos] = 
  moves(x).filter(is_legal(dim, path, _))

def ordered_moves(dim: Int, path: Path, x: Pos): List[Pos] = 
  legal_moves(dim, path, x).sortBy((x) => legal_moves(dim, path, x).length)

import scala.annotation.tailrec

@tailrec
def first(xs: List[Pos], f: Pos => Option[Path]): Option[Path] = xs match {
  case Nil => None
  case x::xs => {
    val result = f(x)
    if (result.isDefined) result else first(xs, f)
  }
}


//def first[A, B](xs: List[A], f: A => Option[B]): Option[B] =
//  xs.flatMap(f(_)).headOption


def first_closed_tour_heuristics(dim: Int, path: Path): Option[Path] = {
  if (path.length == dim * dim && moves(path.head).contains(path.last)) Some(path)
  else
    first(ordered_moves(dim, path, path.head), (x: Pos) => first_closed_tour_heuristics(dim, x::path))
}

// heuristic cannot be used to search for closed tours on 7 x 7
for (dim <- 1 to 6) {
  val t = time_needed(0, first_closed_tour_heuristics(dim, List((dim / 2, dim / 2))))
  println(s"${dim} x ${dim} closed: " + (if (t == None) "" else { print_board(dim, t.get) ; "" }))
}


//@tailrec
/*
def first_tour_heuristics(dim: Int, path: Path): Option[Path] = {

  @tailrec
  def aux(dim: Int, path: Path, moves: List[Pos]): Option[Path] = 
  if (path.length == dim * dim) Some(path)
  else
    moves match {
      case Nil => None
      case x::xs => {
        val r = first_tour_heuristics(dim, x::path)
        if (r.isDefined) r else aux(dim, path, xs)
      }    
    }

  aux(dim, path, ordered_moves(dim, path, path.head)) 
}
*/ 

@tailrec
def tour_on_mega_board(dim: Int, paths: List[Path]): Option[Path] = paths match {
  case Nil => None
  case (path::rest) =>
    if (path.length == dim * dim) Some(path)
    else tour_on_mega_board(dim, ordered_moves(dim, path, path.head).map(_::path) ::: rest)
}



/*
def first_tour_heuristics(dim: Int, path: Path): Option[Path] = {
  if (path.length == dim * dim) Some(path)
  else
    for (p <- ordered_moves(dim, path, path.head))
      val r = first_tour_heuristics(dim, x::path)
    //first(ordered_moves(dim, path, path.head), (x: Pos) => first_tour_heuristics(dim, x::path))
    ordered_moves(dim, path, path.head).view.flatMap((x: Pos) => first_tour_heuristics(dim, x::path)).headOption
}
*/ 

/*
for (dim <- 1 to 50) {
  val t = first_tour_heuristics(dim, List((dim / 2, dim / 2)))
  println(s"${dim} x ${dim}: " + (if (t == None) "" else { print_board(dim, t.get) ; "" }))
}
*/

val dim = 70
println(s"${dim} x ${dim}:")
print_board(dim, time_needed(0, tour_on_mega_board(dim, List(List((0, 0)))).get))

