// Main Part 3 about Regular Expression Matching
//==============================================

object M3 {

// Regular Expressions
abstract class Rexp
case object ZERO extends Rexp
case object ONE extends Rexp
case class CHAR(c: Char) extends Rexp
case class ALTs(rs: List[Rexp]) extends Rexp  // alternatives 
case class SEQs(rs: List[Rexp]) extends Rexp  // sequences
case class STAR(r: Rexp) extends Rexp         // star


//the usual binary choice and binary sequence can be defined 
//in terms of ALTs and SEQs
def ALT(r1: Rexp, r2: Rexp) = ALTs(List(r1, r2))
def SEQ(r1: Rexp, r2: Rexp) = SEQs(List(r1, r2))

// some convenience for typing in regular expressions
import scala.language.implicitConversions    
import scala.language.reflectiveCalls 

def charlist2rexp(s: List[Char]): Rexp = s match {
  case Nil => ONE
  case c::Nil => CHAR(c)
  case c::s => SEQ(CHAR(c), charlist2rexp(s))
}
implicit def string2rexp(s: String): Rexp = charlist2rexp(s.toList)

implicit def RexpOps (r: Rexp) = new {
  def | (s: Rexp) = ALT(r, s)
  def % = STAR(r)
  def ~ (s: Rexp) = SEQ(r, s)
}

implicit def stringOps (s: String) = new {
  def | (r: Rexp) = ALT(s, r)
  def | (r: String) = ALT(s, r)
  def % = STAR(s)
  def ~ (r: Rexp) = SEQ(s, r)
  def ~ (r: String) = SEQ(s, r)
}

// (1) 
def nullable (r: Rexp) : Boolean = r match {
  case ZERO => false
  case ONE => true
  case CHAR(_) => false
  case ALTs(rs) => rs.exists(nullable)
  case SEQs(rs) => rs.forall(nullable)
  case STAR(_) => true
}

// (2) 
def der(c: Char, r: Rexp) : Rexp = r match {
  case ZERO => ZERO
  case ONE => ZERO
  case CHAR(d) => if (c == d) ONE else ZERO
  case ALTs(rs) => ALTs(rs.map(der(c, _)))
  case SEQs(Nil) => ZERO
  case SEQs(r1::rs) => 
    if (nullable(r1)) ALT(SEQs(der(c, r1)::rs), der(c, SEQs(rs)))
    else SEQs(der(c, r1):: rs)
  case STAR(r1) => SEQ(der(c, r1), STAR(r1))
}


// (3) 
def denest(rs: List[Rexp]) : List[Rexp] = rs match {
  case Nil => Nil
  case ZERO::tl => denest(tl)
  case ALTs(rs1)::rs2 => rs1 ::: denest(rs2)  
  case r::rs => r :: denest(rs) 
}

// (4)
def flts(rs: List[Rexp], acc: List[Rexp] = Nil) : List[Rexp] = rs match {
  case Nil => acc
  case ZERO::rs => ZERO::Nil
  case ONE::rs => flts(rs, acc)
  case SEQs(rs1)::rs => flts(rs, acc ::: rs1)
  case r::rs => flts(rs, acc :+ r) 
}

// (5)
def ALTs_smart(rs: List[Rexp]) : Rexp = rs match {
  case Nil => ZERO
  case r::Nil => r  
  case rs => ALTs(rs)
}

def SEQs_smart(rs: List[Rexp]) : Rexp = rs match {
  case Nil => ONE
  case ZERO::Nil => ZERO
  case r::Nil => r
  case rs => SEQs(rs) 
}

// (6) 

def simp(r: Rexp) : Rexp = r match {
  case ALTs(rs) => 
    ALTs_smart(denest(rs.map(simp)).distinct)
  case SEQs(rs) => 
    SEQs_smart(flts(rs.map(simp)))
  case r => r
}

//println("Simp tests")
//println(simp(ALT(ONE | CHAR('a'), CHAR('a') | ONE)))
//println(simp(((CHAR('a') | ZERO) ~ ONE) | 
//              (((ONE | CHAR('b')) | CHAR('c')) ~ (CHAR('d') ~ ZERO))))

// (7) 

def ders (s: List[Char], r: Rexp) : Rexp = s match {
  case Nil => r
  case c::s => ders(s, simp(der(c, r)))
}

// main matcher function
def matcher(r: Rexp, s: String) = nullable(ders(s.toList, r))

// (8) 

def size(r: Rexp): Int = r match {
  case ZERO => 1
  case ONE => 1
  case CHAR(_) => 1
  case ALTs(rs) => 1 + rs.map(size).sum
  case SEQs(rs) => 1 + rs.map(size).sum
  case STAR(r1) => 1 + size(r1)
}



// some testing data
/*
println(matcher(("a" ~ "b") ~ "c", "abc"))  // => true
println(matcher(("a" ~ "b") ~ "c", "ab"))   // => false

// the supposedly 'evil' regular expression (a*)* b
val EVIL = SEQ(STAR(STAR(CHAR('a'))), CHAR('b'))

println(matcher(EVIL, "a" * 1000 ++ "b"))   // => true
println(matcher(EVIL, "a" * 1000))          // => false

// size without simplifications
println(size(der('a', der('a', EVIL))))             // => 36
println(size(der('a', der('a', der('a', EVIL)))))   // => 83

// size with simplification
println(simp(der('a', der('a', EVIL))))          
println(simp(der('a', der('a', der('a', EVIL)))))

println(size(simp(der('a', der('a', EVIL)))))           // => 7
println(size(simp(der('a', der('a', der('a', EVIL)))))) // => 7

// Python needs around 30 seconds for matching 28 a's with EVIL. 
// Java 9 and later increase this to an "astonishing" 40000 a's in
// around 30 seconds.
//
// Lets see how long it takes to match strings with 
// 5 Million a's...it should be in the range of a 
// few seconds.

def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  "%.5f".format((end - start)/(i * 1.0e9))
}

for (i <- 0 to 5000000 by 500000) {
  println(s"$i ${time_needed(2, matcher(EVIL, "a" * i))} secs.") 
}

// another "power" test case 
println(simp(Iterator.iterate(ONE:Rexp)(r => SEQ(r, ONE | ONE)).drop(100).next()) == ONE)

// the Iterator produces the rexp
//
//      SEQ(SEQ(SEQ(..., ONE | ONE) , ONE | ONE), ONE | ONE)
//
//    where SEQ is nested 100 times.
*/ 


}





/*
// if nullable(r1)  
  ALTs(SEQs(der(c, r1)::rs)::(rs filter what is nullable) .map(der(c,_)))

*/
