(*  Title:      Nominal2_Base
    Authors:    Brian Huffman, Christian Urban

    Basic definitions and lemma infrastructure for 
    Nominal Isabelle. 
*)
theory Nominal2_Base
imports Main Infinite_Set
uses ("nominal_library.ML")
     ("nominal_atoms.ML")
begin

section {* Atoms and Sorts *}

text {* A simple implementation for atom_sorts is strings. *}
(* types atom_sort = string *)

text {* To deal with Church-like binding we use trees of  
  strings as sorts. *}

datatype atom_sort = Sort "string" "atom_sort list"

datatype atom = Atom atom_sort nat


text {* Basic projection function. *}

primrec
  sort_of :: "atom \<Rightarrow> atom_sort"
where
  "sort_of (Atom s i) = s"

primrec
  nat_of :: "atom \<Rightarrow> nat"
where
  "nat_of (Atom s n) = n"


text {* There are infinitely many atoms of each sort. *}
lemma INFM_sort_of_eq: 
  shows "INFM a. sort_of a = s"
proof -
  have "INFM i. sort_of (Atom s i) = s" by simp
  moreover have "inj (Atom s)" by (simp add: inj_on_def)
  ultimately show "INFM a. sort_of a = s" by (rule INFM_inj)
qed

lemma infinite_sort_of_eq:
  shows "infinite {a. sort_of a = s}"
  using INFM_sort_of_eq unfolding INFM_iff_infinite .

lemma atom_infinite [simp]: 
  shows "infinite (UNIV :: atom set)"
  using subset_UNIV infinite_sort_of_eq
  by (rule infinite_super)

lemma obtain_atom:
  fixes X :: "atom set"
  assumes X: "finite X"
  obtains a where "a \<notin> X" "sort_of a = s"
proof -
  from X have "MOST a. a \<notin> X"
    unfolding MOST_iff_cofinite by simp
  with INFM_sort_of_eq
  have "INFM a. sort_of a = s \<and> a \<notin> X"
    by (rule INFM_conjI)
  then obtain a where "a \<notin> X" "sort_of a = s"
    by (auto elim: INFM_E)
  then show ?thesis ..
qed

lemma atom_components_eq_iff:
  fixes a b :: atom
  shows "a = b \<longleftrightarrow> sort_of a = sort_of b \<and> nat_of a = nat_of b"
  by (induct a, induct b, simp)

section {* Sort-Respecting Permutations *}

typedef perm =
  "{f. bij f \<and> finite {a. f a \<noteq> a} \<and> (\<forall>a. sort_of (f a) = sort_of a)}"
proof
  show "id \<in> ?perm" by simp
qed

lemma permI:
  assumes "bij f" and "MOST x. f x = x" and "\<And>a. sort_of (f a) = sort_of a"
  shows "f \<in> perm"
  using assms unfolding perm_def MOST_iff_cofinite by simp

lemma perm_is_bij: "f \<in> perm \<Longrightarrow> bij f"
  unfolding perm_def by simp

lemma perm_is_finite: "f \<in> perm \<Longrightarrow> finite {a. f a \<noteq> a}"
  unfolding perm_def by simp

lemma perm_is_sort_respecting: "f \<in> perm \<Longrightarrow> sort_of (f a) = sort_of a"
  unfolding perm_def by simp

lemma perm_MOST: "f \<in> perm \<Longrightarrow> MOST x. f x = x"
  unfolding perm_def MOST_iff_cofinite by simp

lemma perm_id: "id \<in> perm"
  unfolding perm_def by simp

lemma perm_comp:
  assumes f: "f \<in> perm" and g: "g \<in> perm"
  shows "(f \<circ> g) \<in> perm"
apply (rule permI)
apply (rule bij_comp)
apply (rule perm_is_bij [OF g])
apply (rule perm_is_bij [OF f])
apply (rule MOST_rev_mp [OF perm_MOST [OF g]])
apply (rule MOST_rev_mp [OF perm_MOST [OF f]])
apply (simp)
apply (simp add: perm_is_sort_respecting [OF f])
apply (simp add: perm_is_sort_respecting [OF g])
done

lemma perm_inv:
  assumes f: "f \<in> perm"
  shows "(inv f) \<in> perm"
apply (rule permI)
apply (rule bij_imp_bij_inv)
apply (rule perm_is_bij [OF f])
apply (rule MOST_mono [OF perm_MOST [OF f]])
apply (erule subst, rule inv_f_f)
apply (rule bij_is_inj [OF perm_is_bij [OF f]])
apply (rule perm_is_sort_respecting [OF f, THEN sym, THEN trans])
apply (simp add: surj_f_inv_f [OF bij_is_surj [OF perm_is_bij [OF f]]])
done

lemma bij_Rep_perm: "bij (Rep_perm p)"
  using Rep_perm [of p] unfolding perm_def by simp

lemma finite_Rep_perm: "finite {a. Rep_perm p a \<noteq> a}"
  using Rep_perm [of p] unfolding perm_def by simp

lemma sort_of_Rep_perm: "sort_of (Rep_perm p a) = sort_of a"
  using Rep_perm [of p] unfolding perm_def by simp

lemma Rep_perm_ext:
  "Rep_perm p1 = Rep_perm p2 \<Longrightarrow> p1 = p2"
  by (simp add: fun_eq_iff Rep_perm_inject [symmetric])

instance perm :: size ..

subsection {* Permutations form a group *}

instantiation perm :: group_add
begin

definition
  "0 = Abs_perm id"

definition
  "- p = Abs_perm (inv (Rep_perm p))"

definition
  "p + q = Abs_perm (Rep_perm p \<circ> Rep_perm q)"

definition
  "(p1::perm) - p2 = p1 + - p2"

lemma Rep_perm_0: "Rep_perm 0 = id"
  unfolding zero_perm_def
  by (simp add: Abs_perm_inverse perm_id)

lemma Rep_perm_add:
  "Rep_perm (p1 + p2) = Rep_perm p1 \<circ> Rep_perm p2"
  unfolding plus_perm_def
  by (simp add: Abs_perm_inverse perm_comp Rep_perm)

lemma Rep_perm_uminus:
  "Rep_perm (- p) = inv (Rep_perm p)"
  unfolding uminus_perm_def
  by (simp add: Abs_perm_inverse perm_inv Rep_perm)

instance
apply default
unfolding Rep_perm_inject [symmetric]
unfolding minus_perm_def
unfolding Rep_perm_add
unfolding Rep_perm_uminus
unfolding Rep_perm_0
by (simp_all add: o_assoc inv_o_cancel [OF bij_is_inj [OF bij_Rep_perm]])

end


section {* Implementation of swappings *}

definition
  swap :: "atom \<Rightarrow> atom \<Rightarrow> perm" ("'(_ \<rightleftharpoons> _')")
where
  "(a \<rightleftharpoons> b) =
    Abs_perm (if sort_of a = sort_of b 
              then (\<lambda>c. if a = c then b else if b = c then a else c) 
              else id)"

lemma Rep_perm_swap:
  "Rep_perm (a \<rightleftharpoons> b) =
    (if sort_of a = sort_of b 
     then (\<lambda>c. if a = c then b else if b = c then a else c)
     else id)"
unfolding swap_def
apply (rule Abs_perm_inverse)
apply (rule permI)
apply (auto simp add: bij_def inj_on_def surj_def)[1]
apply (rule MOST_rev_mp [OF MOST_neq(1) [of a]])
apply (rule MOST_rev_mp [OF MOST_neq(1) [of b]])
apply (simp)
apply (simp)
done

lemmas Rep_perm_simps =
  Rep_perm_0
  Rep_perm_add
  Rep_perm_uminus
  Rep_perm_swap

lemma swap_different_sorts [simp]:
  "sort_of a \<noteq> sort_of b \<Longrightarrow> (a \<rightleftharpoons> b) = 0"
  by (rule Rep_perm_ext) (simp add: Rep_perm_simps)

lemma swap_cancel:
  "(a \<rightleftharpoons> b) + (a \<rightleftharpoons> b) = 0"
  by (rule Rep_perm_ext) 
     (simp add: Rep_perm_simps fun_eq_iff)

lemma swap_self [simp]:
  "(a \<rightleftharpoons> a) = 0"
  by (rule Rep_perm_ext, simp add: Rep_perm_simps fun_eq_iff)

lemma minus_swap [simp]:
  "- (a \<rightleftharpoons> b) = (a \<rightleftharpoons> b)"
  by (rule minus_unique [OF swap_cancel])

lemma swap_commute:
  "(a \<rightleftharpoons> b) = (b \<rightleftharpoons> a)"
  by (rule Rep_perm_ext)
     (simp add: Rep_perm_swap fun_eq_iff)

lemma swap_triple:
  assumes "a \<noteq> b" and "c \<noteq> b"
  assumes "sort_of a = sort_of b" "sort_of b = sort_of c"
  shows "(a \<rightleftharpoons> c) + (b \<rightleftharpoons> c) + (a \<rightleftharpoons> c) = (a \<rightleftharpoons> b)"
  using assms
  by (rule_tac Rep_perm_ext)
     (auto simp add: Rep_perm_simps fun_eq_iff)


section {* Permutation Types *}

text {*
  Infix syntax for @{text permute} has higher precedence than
  addition, but lower than unary minus.
*}

class pt =
  fixes permute :: "perm \<Rightarrow> 'a \<Rightarrow> 'a" ("_ \<bullet> _" [76, 75] 75)
  assumes permute_zero [simp]: "0 \<bullet> x = x"
  assumes permute_plus [simp]: "(p + q) \<bullet> x = p \<bullet> (q \<bullet> x)"
begin

lemma permute_diff [simp]:
  shows "(p - q) \<bullet> x = p \<bullet> - q \<bullet> x"
  unfolding diff_minus by simp

lemma permute_minus_cancel [simp]:
  shows "p \<bullet> - p \<bullet> x = x"
  and   "- p \<bullet> p \<bullet> x = x"
  unfolding permute_plus [symmetric] by simp_all

lemma permute_swap_cancel [simp]:
  shows "(a \<rightleftharpoons> b) \<bullet> (a \<rightleftharpoons> b) \<bullet> x = x"
  unfolding permute_plus [symmetric]
  by (simp add: swap_cancel)

lemma permute_swap_cancel2 [simp]:
  shows "(a \<rightleftharpoons> b) \<bullet> (b \<rightleftharpoons> a) \<bullet> x = x"
  unfolding permute_plus [symmetric]
  by (simp add: swap_commute)

lemma inj_permute [simp]: 
  shows "inj (permute p)"
  by (rule inj_on_inverseI)
     (rule permute_minus_cancel)

lemma surj_permute [simp]: 
  shows "surj (permute p)"
  by (rule surjI, rule permute_minus_cancel)

lemma bij_permute [simp]: 
  shows "bij (permute p)"
  by (rule bijI [OF inj_permute surj_permute])

lemma inv_permute: 
  shows "inv (permute p) = permute (- p)"
  by (rule inv_equality) (simp_all)

lemma permute_minus: 
  shows "permute (- p) = inv (permute p)"
  by (simp add: inv_permute)

lemma permute_eq_iff [simp]: 
  shows "p \<bullet> x = p \<bullet> y \<longleftrightarrow> x = y"
  by (rule inj_permute [THEN inj_eq])

end

subsection {* Permutations for atoms *}

instantiation atom :: pt
begin

definition
  "p \<bullet> a = (Rep_perm p) a"

instance 
apply(default)
apply(simp_all add: permute_atom_def Rep_perm_simps)
done

end

lemma sort_of_permute [simp]:
  shows "sort_of (p \<bullet> a) = sort_of a"
  unfolding permute_atom_def by (rule sort_of_Rep_perm)

lemma swap_atom:
  shows "(a \<rightleftharpoons> b) \<bullet> c =
           (if sort_of a = sort_of b
            then (if c = a then b else if c = b then a else c) else c)"
  unfolding permute_atom_def
  by (simp add: Rep_perm_swap)

lemma swap_atom_simps [simp]:
  "sort_of a = sort_of b \<Longrightarrow> (a \<rightleftharpoons> b) \<bullet> a = b"
  "sort_of a = sort_of b \<Longrightarrow> (a \<rightleftharpoons> b) \<bullet> b = a"
  "c \<noteq> a \<Longrightarrow> c \<noteq> b \<Longrightarrow> (a \<rightleftharpoons> b) \<bullet> c = c"
  unfolding swap_atom by simp_all

lemma expand_perm_eq:
  fixes p q :: "perm"
  shows "p = q \<longleftrightarrow> (\<forall>a::atom. p \<bullet> a = q \<bullet> a)"
  unfolding permute_atom_def
  by (metis Rep_perm_ext ext)


subsection {* Permutations for permutations *}

instantiation perm :: pt
begin

definition
  "p \<bullet> q = p + q - p"

instance
apply default
apply (simp add: permute_perm_def)
apply (simp add: permute_perm_def diff_minus minus_add add_assoc)
done

end

lemma permute_self: 
  shows "p \<bullet> p = p"
  unfolding permute_perm_def 
  by (simp add: diff_minus add_assoc)

lemma permute_eqvt:
  shows "p \<bullet> (q \<bullet> x) = (p \<bullet> q) \<bullet> (p \<bullet> x)"
  unfolding permute_perm_def by simp

lemma zero_perm_eqvt:
  shows "p \<bullet> (0::perm) = 0"
  unfolding permute_perm_def by simp

lemma add_perm_eqvt:
  fixes p p1 p2 :: perm
  shows "p \<bullet> (p1 + p2) = p \<bullet> p1 + p \<bullet> p2"
  unfolding permute_perm_def
  by (simp add: expand_perm_eq)

lemma swap_eqvt:
  shows "p \<bullet> (a \<rightleftharpoons> b) = (p \<bullet> a \<rightleftharpoons> p \<bullet> b)"
  unfolding permute_perm_def
  by (auto simp add: swap_atom expand_perm_eq)

lemma uminus_eqvt:
  fixes p q::"perm"
  shows "p \<bullet> (- q) = - (p \<bullet> q)"
  unfolding permute_perm_def
  by (simp add: diff_minus minus_add add_assoc)

subsection {* Permutations for functions *}

instantiation "fun" :: (pt, pt) pt
begin

definition
  "p \<bullet> f = (\<lambda>x. p \<bullet> (f (- p \<bullet> x)))"

instance
apply default
apply (simp add: permute_fun_def)
apply (simp add: permute_fun_def minus_add)
done

end

lemma permute_fun_app_eq:
  shows "p \<bullet> (f x) = (p \<bullet> f) (p \<bullet> x)"
  unfolding permute_fun_def by simp


subsection {* Permutations for booleans *}

instantiation bool :: pt
begin

definition "p \<bullet> (b::bool) = b"

instance
apply(default) 
apply(simp_all add: permute_bool_def)
done

end

lemma Not_eqvt:
  shows "p \<bullet> (\<not> A) = (\<not> (p \<bullet> A))"
by (simp add: permute_bool_def)

lemma conj_eqvt:
  shows "p \<bullet> (A \<and> B) = ((p \<bullet> A) \<and> (p \<bullet> B))"
  by (simp add: permute_bool_def)

lemma imp_eqvt:
  shows "p \<bullet> (A \<longrightarrow> B) = ((p \<bullet> A) \<longrightarrow> (p \<bullet> B))"
  by (simp add: permute_bool_def)

lemma ex_eqvt:
  shows "p \<bullet> (\<exists>x. P x) = (\<exists>x. (p \<bullet> P) x)"
  unfolding permute_fun_def permute_bool_def
  by (auto, rule_tac x="p \<bullet> x" in exI, simp)

lemma all_eqvt:
  shows "p \<bullet> (\<forall>x. P x) = (\<forall>x. (p \<bullet> P) x)"
  unfolding permute_fun_def permute_bool_def
  by (auto, drule_tac x="p \<bullet> x" in spec, simp)

lemma permute_boolE:
  fixes P::"bool"
  shows "p \<bullet> P \<Longrightarrow> P"
  by (simp add: permute_bool_def)

lemma permute_boolI:
  fixes P::"bool"
  shows "P \<Longrightarrow> p \<bullet> P"
  by(simp add: permute_bool_def)

subsection {* Permutations for sets *}

lemma permute_set_eq:
  fixes x::"'a::pt"
  and   p::"perm"
  shows "(p \<bullet> X) = {p \<bullet> x | x. x \<in> X}"
  unfolding permute_fun_def
  unfolding permute_bool_def
  apply(auto simp add: mem_def)
  apply(rule_tac x="- p \<bullet> x" in exI)
  apply(simp)
  done

lemma permute_set_eq_image:
  shows "p \<bullet> X = permute p ` X"
  unfolding permute_set_eq by auto

lemma permute_set_eq_vimage:
  shows "p \<bullet> X = permute (- p) -` X"
  unfolding permute_fun_def permute_bool_def
  unfolding vimage_def Collect_def mem_def ..

lemma swap_set_not_in:
  assumes a: "a \<notin> S" "b \<notin> S"
  shows "(a \<rightleftharpoons> b) \<bullet> S = S"
  unfolding permute_set_eq
  using a by (auto simp add: swap_atom)

lemma swap_set_in:
  assumes a: "a \<in> S" "b \<notin> S" "sort_of a = sort_of b"
  shows "(a \<rightleftharpoons> b) \<bullet> S \<noteq> S"
  unfolding permute_set_eq
  using a by (auto simp add: swap_atom)

lemma mem_permute_iff:
  shows "(p \<bullet> x) \<in> (p \<bullet> X) \<longleftrightarrow> x \<in> X"
  unfolding mem_def permute_fun_def permute_bool_def
  by simp

lemma mem_eqvt:
  shows "p \<bullet> (x \<in> A) \<longleftrightarrow> (p \<bullet> x) \<in> (p \<bullet> A)"
  unfolding mem_def
  by (simp add: permute_fun_app_eq)

lemma insert_eqvt:
  shows "p \<bullet> (insert x A) = insert (p \<bullet> x) (p \<bullet> A)"
  unfolding permute_set_eq_image image_insert ..


subsection {* Permutations for units *}

instantiation unit :: pt
begin

definition "p \<bullet> (u::unit) = u"

instance 
by (default) (simp_all add: permute_unit_def)

end


subsection {* Permutations for products *}

instantiation prod :: (pt, pt) pt
begin

primrec 
  permute_prod 
where
  Pair_eqvt: "p \<bullet> (x, y) = (p \<bullet> x, p \<bullet> y)"

instance
by default auto

end

subsection {* Permutations for sums *}

instantiation sum :: (pt, pt) pt
begin

primrec 
  permute_sum 
where
  "p \<bullet> (Inl x) = Inl (p \<bullet> x)"
| "p \<bullet> (Inr y) = Inr (p \<bullet> y)"

instance 
by (default) (case_tac [!] x, simp_all)

end

subsection {* Permutations for lists *}

instantiation list :: (pt) pt
begin

primrec 
  permute_list 
where
  "p \<bullet> [] = []"
| "p \<bullet> (x # xs) = p \<bullet> x # p \<bullet> xs"

instance 
by (default) (induct_tac [!] x, simp_all)

end

subsection {* Permutations for options *}

instantiation option :: (pt) pt
begin

primrec 
  permute_option 
where
  "p \<bullet> None = None"
| "p \<bullet> (Some x) = Some (p \<bullet> x)"

instance 
by (default) (induct_tac [!] x, simp_all)

end

subsection {* Permutations for @{typ char}, @{typ nat}, and @{typ int} *}

instantiation char :: pt
begin

definition "p \<bullet> (c::char) = c"

instance 
by (default) (simp_all add: permute_char_def)

end

instantiation nat :: pt
begin

definition "p \<bullet> (n::nat) = n"

instance 
by (default) (simp_all add: permute_nat_def)

end

instantiation int :: pt
begin

definition "p \<bullet> (i::int) = i"

instance 
by (default) (simp_all add: permute_int_def)

end


section {* Pure types *}

text {* Pure types will have always empty support. *}

class pure = pt +
  assumes permute_pure: "p \<bullet> x = x"

text {* Types @{typ unit} and @{typ bool} are pure. *}

instance unit :: pure
proof qed (rule permute_unit_def)

instance bool :: pure
proof qed (rule permute_bool_def)

text {* Other type constructors preserve purity. *}

instance "fun" :: (pure, pure) pure
by default (simp add: permute_fun_def permute_pure)

instance prod :: (pure, pure) pure
by default (induct_tac x, simp add: permute_pure)

instance sum :: (pure, pure) pure
by default (induct_tac x, simp_all add: permute_pure)

instance list :: (pure) pure
by default (induct_tac x, simp_all add: permute_pure)

instance option :: (pure) pure
by default (induct_tac x, simp_all add: permute_pure)


subsection {* Types @{typ char}, @{typ nat}, and @{typ int} *}

instance char :: pure
proof qed (rule permute_char_def)

instance nat :: pure
proof qed (rule permute_nat_def)

instance int :: pure
proof qed (rule permute_int_def)


subsection {* Supp, Freshness and Supports *}

context pt
begin

definition
  supp :: "'a \<Rightarrow> atom set"
where
  "supp x = {a. infinite {b. (a \<rightleftharpoons> b) \<bullet> x \<noteq> x}}"

end

definition
  fresh :: "atom \<Rightarrow> 'a::pt \<Rightarrow> bool" ("_ \<sharp> _" [55, 55] 55)
where   
  "a \<sharp> x \<equiv> a \<notin> supp x"

lemma supp_conv_fresh: 
  shows "supp x = {a. \<not> a \<sharp> x}"
  unfolding fresh_def by simp

lemma swap_rel_trans:
  assumes "sort_of a = sort_of b"
  assumes "sort_of b = sort_of c"
  assumes "(a \<rightleftharpoons> c) \<bullet> x = x"
  assumes "(b \<rightleftharpoons> c) \<bullet> x = x"
  shows "(a \<rightleftharpoons> b) \<bullet> x = x"
proof (cases)
  assume "a = b \<or> c = b"
  with assms show "(a \<rightleftharpoons> b) \<bullet> x = x" by auto
next
  assume *: "\<not> (a = b \<or> c = b)"
  have "((a \<rightleftharpoons> c) + (b \<rightleftharpoons> c) + (a \<rightleftharpoons> c)) \<bullet> x = x"
    using assms by simp
  also have "(a \<rightleftharpoons> c) + (b \<rightleftharpoons> c) + (a \<rightleftharpoons> c) = (a \<rightleftharpoons> b)"
    using assms * by (simp add: swap_triple)
  finally show "(a \<rightleftharpoons> b) \<bullet> x = x" .
qed

lemma swap_fresh_fresh:
  assumes a: "a \<sharp> x" 
  and     b: "b \<sharp> x"
  shows "(a \<rightleftharpoons> b) \<bullet> x = x"
proof (cases)
  assume asm: "sort_of a = sort_of b" 
  have "finite {c. (a \<rightleftharpoons> c) \<bullet> x \<noteq> x}" "finite {c. (b \<rightleftharpoons> c) \<bullet> x \<noteq> x}" 
    using a b unfolding fresh_def supp_def by simp_all
  then have "finite ({c. (a \<rightleftharpoons> c) \<bullet> x \<noteq> x} \<union> {c. (b \<rightleftharpoons> c) \<bullet> x \<noteq> x})" by simp
  then obtain c 
    where "(a \<rightleftharpoons> c) \<bullet> x = x" "(b \<rightleftharpoons> c) \<bullet> x = x" "sort_of c = sort_of b"
    by (rule obtain_atom) (auto)
  then show "(a \<rightleftharpoons> b) \<bullet> x = x" using asm by (rule_tac swap_rel_trans) (simp_all)
next
  assume "sort_of a \<noteq> sort_of b"
  then show "(a \<rightleftharpoons> b) \<bullet> x = x" by simp
qed


subsection {* supp and fresh are equivariant *}

lemma finite_Collect_bij:
  assumes a: "bij f"
  shows "finite {x. P (f x)} = finite {x. P x}"
by (metis a finite_vimage_iff vimage_Collect_eq)

lemma fresh_permute_iff:
  shows "(p \<bullet> a) \<sharp> (p \<bullet> x) \<longleftrightarrow> a \<sharp> x"
proof -
  have "(p \<bullet> a) \<sharp> (p \<bullet> x) \<longleftrightarrow> finite {b. (p \<bullet> a \<rightleftharpoons> b) \<bullet> p \<bullet> x \<noteq> p \<bullet> x}"
    unfolding fresh_def supp_def by simp
  also have "\<dots> \<longleftrightarrow> finite {b. (p \<bullet> a \<rightleftharpoons> p \<bullet> b) \<bullet> p \<bullet> x \<noteq> p \<bullet> x}"
    using bij_permute by (rule finite_Collect_bij[symmetric])
  also have "\<dots> \<longleftrightarrow> finite {b. p \<bullet> (a \<rightleftharpoons> b) \<bullet> x \<noteq> p \<bullet> x}"
    by (simp only: permute_eqvt [of p] swap_eqvt)
  also have "\<dots> \<longleftrightarrow> finite {b. (a \<rightleftharpoons> b) \<bullet> x \<noteq> x}"
    by (simp only: permute_eq_iff)
  also have "\<dots> \<longleftrightarrow> a \<sharp> x"
    unfolding fresh_def supp_def by simp
  finally show "(p \<bullet> a) \<sharp> (p \<bullet> x) \<longleftrightarrow> a \<sharp> x" .
qed

lemma fresh_eqvt:
  shows "p \<bullet> (a \<sharp> x) = (p \<bullet> a) \<sharp> (p \<bullet> x)"
  unfolding permute_bool_def
  by (simp add: fresh_permute_iff)

lemma supp_eqvt:
  fixes  p  :: "perm"
  and    x   :: "'a::pt"
  shows "p \<bullet> (supp x) = supp (p \<bullet> x)"
  unfolding supp_conv_fresh
  unfolding Collect_def
  unfolding permute_fun_def
  by (simp add: Not_eqvt fresh_eqvt)

subsection {* supports *}

definition
  supports :: "atom set \<Rightarrow> 'a::pt \<Rightarrow> bool" (infixl "supports" 80)
where  
  "S supports x \<equiv> \<forall>a b. (a \<notin> S \<and> b \<notin> S \<longrightarrow> (a \<rightleftharpoons> b) \<bullet> x = x)"

lemma supp_is_subset:
  fixes S :: "atom set"
  and   x :: "'a::pt"
  assumes a1: "S supports x"
  and     a2: "finite S"
  shows "(supp x) \<subseteq> S"
proof (rule ccontr)
  assume "\<not> (supp x \<subseteq> S)"
  then obtain a where b1: "a \<in> supp x" and b2: "a \<notin> S" by auto
  from a1 b2 have "\<forall>b. b \<notin> S \<longrightarrow> (a \<rightleftharpoons> b) \<bullet> x = x" unfolding supports_def by auto
  then have "{b. (a \<rightleftharpoons> b) \<bullet> x \<noteq> x} \<subseteq> S" by auto
  with a2 have "finite {b. (a \<rightleftharpoons> b)\<bullet>x \<noteq> x}" by (simp add: finite_subset)
  then have "a \<notin> (supp x)" unfolding supp_def by simp
  with b1 show False by simp
qed

lemma supports_finite:
  fixes S :: "atom set"
  and   x :: "'a::pt"
  assumes a1: "S supports x"
  and     a2: "finite S"
  shows "finite (supp x)"
proof -
  have "(supp x) \<subseteq> S" using a1 a2 by (rule supp_is_subset)
  then show "finite (supp x)" using a2 by (simp add: finite_subset)
qed

lemma supp_supports:
  fixes x :: "'a::pt"
  shows "(supp x) supports x"
unfolding supports_def
proof (intro strip)
  fix a b
  assume "a \<notin> (supp x) \<and> b \<notin> (supp x)"
  then have "a \<sharp> x" and "b \<sharp> x" by (simp_all add: fresh_def)
  then show "(a \<rightleftharpoons> b) \<bullet> x = x" by (simp add: swap_fresh_fresh)
qed

lemma supp_is_least_supports:
  fixes S :: "atom set"
  and   x :: "'a::pt"
  assumes  a1: "S supports x"
  and      a2: "finite S"
  and      a3: "\<And>S'. finite S' \<Longrightarrow> (S' supports x) \<Longrightarrow> S \<subseteq> S'"
  shows "(supp x) = S"
proof (rule equalityI)
  show "(supp x) \<subseteq> S" using a1 a2 by (rule supp_is_subset)
  with a2 have fin: "finite (supp x)" by (rule rev_finite_subset)
  have "(supp x) supports x" by (rule supp_supports)
  with fin a3 show "S \<subseteq> supp x" by blast
qed

lemma subsetCI: 
  shows "(\<And>x. x \<in> A \<Longrightarrow> x \<notin> B \<Longrightarrow> False) \<Longrightarrow> A \<subseteq> B"
  by auto

lemma finite_supp_unique:
  assumes a1: "S supports x"
  assumes a2: "finite S"
  assumes a3: "\<And>a b. \<lbrakk>a \<in> S; b \<notin> S; sort_of a = sort_of b\<rbrakk> \<Longrightarrow> (a \<rightleftharpoons> b) \<bullet> x \<noteq> x"
  shows "(supp x) = S"
  using a1 a2
proof (rule supp_is_least_supports)
  fix S'
  assume "finite S'" and "S' supports x"
  show "S \<subseteq> S'"
  proof (rule subsetCI)
    fix a
    assume "a \<in> S" and "a \<notin> S'"
    have "finite (S \<union> S')"
      using `finite S` `finite S'` by simp
    then obtain b where "b \<notin> S \<union> S'" and "sort_of b = sort_of a"
      by (rule obtain_atom)
    then have "b \<notin> S" and "b \<notin> S'"  and "sort_of a = sort_of b"
      by simp_all
    then have "(a \<rightleftharpoons> b) \<bullet> x = x"
      using `a \<notin> S'` `S' supports x` by (simp add: supports_def)
    moreover have "(a \<rightleftharpoons> b) \<bullet> x \<noteq> x"
      using `a \<in> S` `b \<notin> S` `sort_of a = sort_of b`
      by (rule a3)
    ultimately show "False" by simp
  qed
qed

section {* Support w.r.t. relations *}

text {* 
  This definition is used for unquotient types, where
  alpha-equivalence does not coincide with equality.
*}

definition 
  "supp_rel R x = {a. infinite {b. \<not>(R ((a \<rightleftharpoons> b) \<bullet> x) x)}}"



section {* Finitely-supported types *}

class fs = pt +
  assumes finite_supp: "finite (supp x)"

lemma pure_supp: 
  shows "supp (x::'a::pure) = {}"
  unfolding supp_def by (simp add: permute_pure)

lemma pure_fresh:
  fixes x::"'a::pure"
  shows "a \<sharp> x"
  unfolding fresh_def by (simp add: pure_supp)

instance pure < fs
by default (simp add: pure_supp)


subsection  {* Type @{typ atom} is finitely-supported. *}

lemma supp_atom:
  shows "supp a = {a}"
apply (rule finite_supp_unique)
apply (clarsimp simp add: supports_def)
apply simp
apply simp
done

lemma fresh_atom: 
  shows "a \<sharp> b \<longleftrightarrow> a \<noteq> b"
  unfolding fresh_def supp_atom by simp

instance atom :: fs
by default (simp add: supp_atom)

section {* Support for finite sets of atoms *}


lemma supp_finite_atom_set:
  fixes S::"atom set"
  assumes "finite S"
  shows "supp S = S"
  apply(rule finite_supp_unique)
  apply(simp add: supports_def)
  apply(simp add: swap_set_not_in)
  apply(rule assms)
  apply(simp add: swap_set_in)
done

section {* Type @{typ perm} is finitely-supported. *}

lemma perm_swap_eq:
  shows "(a \<rightleftharpoons> b) \<bullet> p = p \<longleftrightarrow> (p \<bullet> (a \<rightleftharpoons> b)) = (a \<rightleftharpoons> b)"
unfolding permute_perm_def
by (metis add_diff_cancel minus_perm_def)

lemma supports_perm: 
  shows "{a. p \<bullet> a \<noteq> a} supports p"
  unfolding supports_def
  unfolding perm_swap_eq
  by (simp add: swap_eqvt)

lemma finite_perm_lemma: 
  shows "finite {a::atom. p \<bullet> a \<noteq> a}"
  using finite_Rep_perm [of p]
  unfolding permute_atom_def .

lemma supp_perm:
  shows "supp p = {a. p \<bullet> a \<noteq> a}"
apply (rule finite_supp_unique)
apply (rule supports_perm)
apply (rule finite_perm_lemma)
apply (simp add: perm_swap_eq swap_eqvt)
apply (auto simp add: expand_perm_eq swap_atom)
done

lemma fresh_perm: 
  shows "a \<sharp> p \<longleftrightarrow> p \<bullet> a = a"
  unfolding fresh_def 
  by (simp add: supp_perm)

lemma supp_swap:
  shows "supp (a \<rightleftharpoons> b) = (if a = b \<or> sort_of a \<noteq> sort_of b then {} else {a, b})"
  by (auto simp add: supp_perm swap_atom)

lemma fresh_zero_perm: 
  shows "a \<sharp> (0::perm)"
  unfolding fresh_perm by simp

lemma supp_zero_perm: 
  shows "supp (0::perm) = {}"
  unfolding supp_perm by simp

lemma fresh_plus_perm:
  fixes p q::perm
  assumes "a \<sharp> p" "a \<sharp> q"
  shows "a \<sharp> (p + q)"
  using assms
  unfolding fresh_def
  by (auto simp add: supp_perm)

lemma supp_plus_perm:
  fixes p q::perm
  shows "supp (p + q) \<subseteq> supp p \<union> supp q"
  by (auto simp add: supp_perm)

lemma fresh_minus_perm:
  fixes p::perm
  shows "a \<sharp> (- p) \<longleftrightarrow> a \<sharp> p"
  unfolding fresh_def
  unfolding supp_perm
  apply(simp)
  apply(metis permute_minus_cancel)
  done

lemma supp_minus_perm:
  fixes p::perm
  shows "supp (- p) = supp p"
  unfolding supp_conv_fresh
  by (simp add: fresh_minus_perm)

instance perm :: fs
by default (simp add: supp_perm finite_perm_lemma)

lemma plus_perm_eq:
  fixes p q::"perm"
  assumes asm: "supp p \<inter> supp q = {}"
  shows "p + q  = q + p"
unfolding expand_perm_eq
proof
  fix a::"atom"
  show "(p + q) \<bullet> a = (q + p) \<bullet> a"
  proof -
    { assume "a \<notin> supp p" "a \<notin> supp q"
      then have "(p + q) \<bullet> a = (q + p) \<bullet> a" 
	by (simp add: supp_perm)
    }
    moreover
    { assume a: "a \<in> supp p" "a \<notin> supp q"
      then have "p \<bullet> a \<in> supp p" by (simp add: supp_perm)
      then have "p \<bullet> a \<notin> supp q" using asm by auto
      with a have "(p + q) \<bullet> a = (q + p) \<bullet> a" 
	by (simp add: supp_perm)
    }
    moreover
    { assume a: "a \<notin> supp p" "a \<in> supp q"
      then have "q \<bullet> a \<in> supp q" by (simp add: supp_perm)
      then have "q \<bullet> a \<notin> supp p" using asm by auto 
      with a have "(p + q) \<bullet> a = (q + p) \<bullet> a" 
	by (simp add: supp_perm)
    }
    ultimately show "(p + q) \<bullet> a = (q + p) \<bullet> a" 
      using asm by blast
  qed
qed

section {* Finite Support instances for other types *}

subsection {* Type @{typ "'a \<times> 'b"} is finitely-supported. *}

lemma supp_Pair: 
  shows "supp (x, y) = supp x \<union> supp y"
  by (simp add: supp_def Collect_imp_eq Collect_neg_eq)

lemma fresh_Pair: 
  shows "a \<sharp> (x, y) \<longleftrightarrow> a \<sharp> x \<and> a \<sharp> y"
  by (simp add: fresh_def supp_Pair)

lemma supp_Unit:
  shows "supp () = {}"
  by (simp add: supp_def)

lemma fresh_Unit:
  shows "a \<sharp> ()"
  by (simp add: fresh_def supp_Unit)



instance prod :: (fs, fs) fs
apply default
apply (induct_tac x)
apply (simp add: supp_Pair finite_supp)
done

subsection {* Type @{typ "'a + 'b"} is finitely supported *}

lemma supp_Inl: 
  shows "supp (Inl x) = supp x"
  by (simp add: supp_def)

lemma supp_Inr: 
  shows "supp (Inr x) = supp x"
  by (simp add: supp_def)

lemma fresh_Inl: 
  shows "a \<sharp> Inl x \<longleftrightarrow> a \<sharp> x"
  by (simp add: fresh_def supp_Inl)

lemma fresh_Inr: 
  shows "a \<sharp> Inr y \<longleftrightarrow> a \<sharp> y"
  by (simp add: fresh_def supp_Inr)

instance sum :: (fs, fs) fs
apply default
apply (induct_tac x)
apply (simp_all add: supp_Inl supp_Inr finite_supp)
done

subsection {* Type @{typ "'a option"} is finitely supported *}

lemma supp_None: 
  shows "supp None = {}"
by (simp add: supp_def)

lemma supp_Some: 
  shows "supp (Some x) = supp x"
  by (simp add: supp_def)

lemma fresh_None: 
  shows "a \<sharp> None"
  by (simp add: fresh_def supp_None)

lemma fresh_Some: 
  shows "a \<sharp> Some x \<longleftrightarrow> a \<sharp> x"
  by (simp add: fresh_def supp_Some)

instance option :: (fs) fs
apply default
apply (induct_tac x)
apply (simp_all add: supp_None supp_Some finite_supp)
done

subsubsection {* Type @{typ "'a list"} is finitely supported *}

lemma supp_Nil: 
  shows "supp [] = {}"
  by (simp add: supp_def)

lemma supp_Cons: 
  shows "supp (x # xs) = supp x \<union> supp xs"
by (simp add: supp_def Collect_imp_eq Collect_neg_eq)

lemma fresh_Nil: 
  shows "a \<sharp> []"
  by (simp add: fresh_def supp_Nil)

lemma fresh_Cons: 
  shows "a \<sharp> (x # xs) \<longleftrightarrow> a \<sharp> x \<and> a \<sharp> xs"
  by (simp add: fresh_def supp_Cons)

instance list :: (fs) fs
apply default
apply (induct_tac x)
apply (simp_all add: supp_Nil supp_Cons finite_supp)
done


section {* Support and Freshness for Applications *}

lemma fresh_conv_MOST: 
  shows "a \<sharp> x \<longleftrightarrow> (MOST b. (a \<rightleftharpoons> b) \<bullet> x = x)"
  unfolding fresh_def supp_def 
  unfolding MOST_iff_cofinite by simp

lemma supp_subset_fresh:
  assumes a: "\<And>a. a \<sharp> x \<Longrightarrow> a \<sharp> y"
  shows "supp y \<subseteq> supp x"
  using a
  unfolding fresh_def
  by blast

lemma fresh_fun_app:
  assumes "a \<sharp> f" and "a \<sharp> x" 
  shows "a \<sharp> f x"
  using assms
  unfolding fresh_conv_MOST
  unfolding permute_fun_app_eq 
  by (elim MOST_rev_mp, simp)

lemma supp_fun_app:
  shows "supp (f x) \<subseteq> (supp f) \<union> (supp x)"
  using fresh_fun_app
  unfolding fresh_def
  by auto

text {* Support of Equivariant Functions *}

lemma supp_fun_eqvt:
  assumes a: "\<And>p. p \<bullet> f = f"
  shows "supp f = {}"
  unfolding supp_def 
  using a by simp

lemma fresh_fun_eqvt_app:
  assumes a: "\<And>p. p \<bullet> f = f"
  shows "a \<sharp> x \<Longrightarrow> a \<sharp> f x"
proof -
  from a have "supp f = {}" by (simp add: supp_fun_eqvt)
  then show "a \<sharp> x \<Longrightarrow> a \<sharp> f x"
    unfolding fresh_def
    using supp_fun_app by auto
qed


section {* Support of Finite Sets of Finitely Supported Elements *}

lemma Union_fresh:
  shows "a \<sharp> S \<Longrightarrow> a \<sharp> (\<Union>x \<in> S. supp x)"
  unfolding Union_image_eq[symmetric]
  apply(rule_tac f="\<lambda>S. \<Union> supp ` S" in fresh_fun_eqvt_app)
  apply(simp add: permute_fun_def UNION_def Collect_def Bex_def ex_eqvt mem_def conj_eqvt)
  apply(subst permute_fun_app_eq)
  back
  apply(simp add: supp_eqvt)
  apply(assumption)
  done

lemma Union_supports_set:
  shows "(\<Union>x \<in> S. supp x) supports S"
proof -
  { fix a b
    have "\<forall>x \<in> S. (a \<rightleftharpoons> b) \<bullet> x = x \<Longrightarrow> (a \<rightleftharpoons> b) \<bullet> S = S"
      unfolding permute_set_eq by force
  }
  then show "(\<Union>x \<in> S. supp x) supports S"
    unfolding supports_def 
    by (simp add: fresh_def[symmetric] swap_fresh_fresh)
qed

lemma Union_of_fin_supp_sets:
  fixes S::"('a::fs set)"
  assumes fin: "finite S"   
  shows "finite (\<Union>x\<in>S. supp x)"
  using fin by (induct) (auto simp add: finite_supp)

lemma Union_included_in_supp:
  fixes S::"('a::fs set)"
  assumes fin: "finite S"
  shows "(\<Union>x\<in>S. supp x) \<subseteq> supp S"
proof -
  have "(\<Union>x\<in>S. supp x) = supp (\<Union>x\<in>S. supp x)"
    by (rule supp_finite_atom_set[symmetric])
       (rule Union_of_fin_supp_sets[OF fin])
  also have "\<dots> \<subseteq> supp S"
    by (rule supp_subset_fresh)
       (simp add: Union_fresh)
  finally show "(\<Union>x\<in>S. supp x) \<subseteq> supp S" .
qed

lemma supp_of_fin_sets:
  fixes S::"('a::fs set)"
  assumes fin: "finite S"
  shows "(supp S) = (\<Union>x\<in>S. supp x)"
apply(rule subset_antisym)
apply(rule supp_is_subset)
apply(rule Union_supports_set)
apply(rule Union_of_fin_supp_sets[OF fin])
apply(rule Union_included_in_supp[OF fin])
done

lemma supp_of_fin_union:
  fixes S T::"('a::fs) set"
  assumes fin1: "finite S"
  and     fin2: "finite T"
  shows "supp (S \<union> T) = supp S \<union> supp T"
  using fin1 fin2
  by (simp add: supp_of_fin_sets)

lemma supp_of_fin_insert:
  fixes S::"('a::fs) set"
  assumes fin:  "finite S"
  shows "supp (insert x S) = supp x \<union> supp S"
  using fin
  by (simp add: supp_of_fin_sets)


section {* Fresh-Star *}


text {* The fresh-star generalisation of fresh is used in strong
  induction principles. *}

definition 
  fresh_star :: "atom set \<Rightarrow> 'a::pt \<Rightarrow> bool" ("_ \<sharp>* _" [80,80] 80)
where 
  "as \<sharp>* x \<equiv> \<forall>a \<in> as. a \<sharp> x"

lemma fresh_star_supp_conv:
  shows "supp x \<sharp>* y \<Longrightarrow> supp y \<sharp>* x"
by (auto simp add: fresh_star_def fresh_def)

lemma fresh_star_prod:
  fixes as::"atom set"
  shows "as \<sharp>* (x, y) = (as \<sharp>* x \<and> as \<sharp>* y)" 
  by (auto simp add: fresh_star_def fresh_Pair)

lemma fresh_star_union:
  shows "(as \<union> bs) \<sharp>* x = (as \<sharp>* x \<and> bs \<sharp>* x)"
  by (auto simp add: fresh_star_def)

lemma fresh_star_insert:
  shows "(insert a as) \<sharp>* x = (a \<sharp> x \<and> as \<sharp>* x)"
  by (auto simp add: fresh_star_def)

lemma fresh_star_Un_elim:
  "((as \<union> bs) \<sharp>* x \<Longrightarrow> PROP C) \<equiv> (as \<sharp>* x \<Longrightarrow> bs \<sharp>* x \<Longrightarrow> PROP C)"
  unfolding fresh_star_def
  apply(rule)
  apply(erule meta_mp)
  apply(auto)
  done

lemma fresh_star_insert_elim:
  "(insert a as \<sharp>* x \<Longrightarrow> PROP C) \<equiv> (a \<sharp> x \<Longrightarrow> as \<sharp>* x \<Longrightarrow> PROP C)"
  unfolding fresh_star_def
  by rule (simp_all add: fresh_star_def)

lemma fresh_star_empty_elim:
  "({} \<sharp>* x \<Longrightarrow> PROP C) \<equiv> PROP C"
  by (simp add: fresh_star_def)

lemma fresh_star_unit_elim: 
  shows "(a \<sharp>* () \<Longrightarrow> PROP C) \<equiv> PROP C"
  by (simp add: fresh_star_def fresh_Unit) 

lemma fresh_star_prod_elim: 
  shows "(a \<sharp>* (x, y) \<Longrightarrow> PROP C) \<equiv> (a \<sharp>* x \<Longrightarrow> a \<sharp>* y \<Longrightarrow> PROP C)"
  by (rule, simp_all add: fresh_star_prod)

lemma fresh_star_zero:
  shows "as \<sharp>* (0::perm)"
  unfolding fresh_star_def
  by (simp add: fresh_zero_perm)

lemma fresh_star_plus:
  fixes p q::perm
  shows "\<lbrakk>a \<sharp>* p;  a \<sharp>* q\<rbrakk> \<Longrightarrow> a \<sharp>* (p + q)"
  unfolding fresh_star_def
  by (simp add: fresh_plus_perm)

lemma fresh_star_permute_iff:
  shows "(p \<bullet> a) \<sharp>* (p \<bullet> x) \<longleftrightarrow> a \<sharp>* x"
  unfolding fresh_star_def
  by (metis mem_permute_iff permute_minus_cancel(1) fresh_permute_iff)

lemma fresh_star_eqvt:
  shows "(p \<bullet> (as \<sharp>* x)) = (p \<bullet> as) \<sharp>* (p \<bullet> x)"
unfolding fresh_star_def
unfolding Ball_def
apply(simp add: all_eqvt)
apply(subst permute_fun_def)
apply(simp add: imp_eqvt fresh_eqvt mem_eqvt)
done


section {* Induction principle for permutations *}


lemma perm_struct_induct[consumes 1, case_names zero swap]:
  assumes S: "supp p \<subseteq> S"
  and zero: "P 0"
  and swap: "\<And>p a b. \<lbrakk>P p; supp p \<subseteq> S; a \<in> S; b \<in> S; a \<noteq> b; sort_of a = sort_of b\<rbrakk> \<Longrightarrow> P ((a \<rightleftharpoons> b) + p)"
  shows "P p"
proof -
  have "finite (supp p)" by (simp add: finite_supp)
  then show "P p" using S
  proof(induct A\<equiv>"supp p" arbitrary: p rule: finite_psubset_induct)
    case (psubset p)
    then have ih: "\<And>q. supp q \<subset> supp p \<Longrightarrow> P q" by auto
    have as: "supp p \<subseteq> S" by fact
    { assume "supp p = {}"
      then have "p = 0" by (simp add: supp_perm expand_perm_eq)
      then have "P p" using zero by simp
    }
    moreover
    { assume "supp p \<noteq> {}"
      then obtain a where a0: "a \<in> supp p" by blast
      then have a1: "p \<bullet> a \<in> S" "a \<in> S" "sort_of (p \<bullet> a) = sort_of a" "p \<bullet> a \<noteq> a"
        using as by (auto simp add: supp_atom supp_perm swap_atom)
      let ?q = "(p \<bullet> a \<rightleftharpoons> a) + p"
      have a2: "supp ?q \<subseteq> supp p" unfolding supp_perm by (auto simp add: swap_atom)
      moreover
      have "a \<notin> supp ?q" by (simp add: supp_perm)
      then have "supp ?q \<noteq> supp p" using a0 by auto
      ultimately have "supp ?q \<subset> supp p" using a2 by auto
      then have "P ?q" using ih by simp
      moreover
      have "supp ?q \<subseteq> S" using as a2 by simp
      ultimately  have "P ((p \<bullet> a \<rightleftharpoons> a) + ?q)" using as a1 swap by simp 
      moreover 
      have "p = (p \<bullet> a \<rightleftharpoons> a) + ?q" by (simp add: expand_perm_eq)
      ultimately have "P p" by simp
    }
    ultimately show "P p" by blast
  qed
qed

lemma perm_simple_struct_induct[case_names zero swap]:
  assumes zero: "P 0"
  and     swap: "\<And>p a b. \<lbrakk>P p; a \<noteq> b; sort_of a = sort_of b\<rbrakk> \<Longrightarrow> P ((a \<rightleftharpoons> b) + p)"
  shows "P p"
by (rule_tac S="supp p" in perm_struct_induct)
   (auto intro: zero swap)

lemma perm_subset_induct[consumes 1, case_names zero swap plus]:
  assumes S: "supp p \<subseteq> S"
  assumes zero: "P 0"
  assumes swap: "\<And>a b. \<lbrakk>sort_of a = sort_of b; a \<noteq> b; a \<in> S; b \<in> S\<rbrakk> \<Longrightarrow> P (a \<rightleftharpoons> b)"
  assumes plus: "\<And>p1 p2. \<lbrakk>P p1; P p2; supp p1 \<subseteq> S; supp p2 \<subseteq> S\<rbrakk> \<Longrightarrow> P (p1 + p2)"
  shows "P p"
using S
by (induct p rule: perm_struct_induct)
   (auto intro: zero plus swap simp add: supp_swap)

lemma supp_perm_eq:
  assumes "(supp x) \<sharp>* p"
  shows "p \<bullet> x = x"
proof -
  from assms have "supp p \<subseteq> {a. a \<sharp> x}"
    unfolding supp_perm fresh_star_def fresh_def by auto
  then show "p \<bullet> x = x"
  proof (induct p rule: perm_struct_induct)
    case zero
    show "0 \<bullet> x = x" by simp
  next
    case (swap p a b)
    then have "a \<sharp> x" "b \<sharp> x" "p \<bullet> x = x" by simp_all
    then show "((a \<rightleftharpoons> b) + p) \<bullet> x = x" by (simp add: swap_fresh_fresh)
  qed
qed

lemma supp_perm_eq_test:
  assumes "(supp x) \<sharp>* p"
  shows "p \<bullet> x = x"
proof -
  from assms have "supp p \<subseteq> {a. a \<sharp> x}"
    unfolding supp_perm fresh_star_def fresh_def by auto
  then show "p \<bullet> x = x"
  proof (induct p rule: perm_subset_induct)
    case zero
    show "0 \<bullet> x = x" by simp
  next
    case (swap a b)
    then have "a \<sharp> x" "b \<sharp> x" by simp_all
    then show "(a \<rightleftharpoons> b) \<bullet> x = x" by (simp add: swap_fresh_fresh)
  next
    case (plus p1 p2)
    have "p1 \<bullet> x = x" "p2 \<bullet> x = x" by fact+
    then show "(p1 + p2) \<bullet> x = x" by simp
  qed
qed


section {* Avoiding of atom sets *}

text {* 
  For every set of atoms, there is another set of atoms
  avoiding a finitely supported c and there is a permutation
  which 'translates' between both sets.
*}

lemma at_set_avoiding_aux:
  fixes Xs::"atom set"
  and   As::"atom set"
  assumes b: "Xs \<subseteq> As"
  and     c: "finite As"
  shows "\<exists>p. (p \<bullet> Xs) \<inter> As = {} \<and> (supp p) \<subseteq> (Xs \<union> (p \<bullet> Xs))"
proof -
  from b c have "finite Xs" by (rule finite_subset)
  then show ?thesis using b
  proof (induct rule: finite_subset_induct)
    case empty
    have "0 \<bullet> {} \<inter> As = {}" by simp
    moreover
    have "supp (0::perm) \<subseteq> {} \<union> 0 \<bullet> {}" by (simp add: supp_zero_perm)
    ultimately show ?case by blast
  next
    case (insert x Xs)
    then obtain p where
      p1: "(p \<bullet> Xs) \<inter> As = {}" and 
      p2: "supp p \<subseteq> (Xs \<union> (p \<bullet> Xs))" by blast
    from `x \<in> As` p1 have "x \<notin> p \<bullet> Xs" by fast
    with `x \<notin> Xs` p2 have "x \<notin> supp p" by fast
    hence px: "p \<bullet> x = x" unfolding supp_perm by simp
    have "finite (As \<union> p \<bullet> Xs)"
      using `finite As` `finite Xs`
      by (simp add: permute_set_eq_image)
    then obtain y where "y \<notin> (As \<union> p \<bullet> Xs)" "sort_of y = sort_of x"
      by (rule obtain_atom)
    hence y: "y \<notin> As" "y \<notin> p \<bullet> Xs" "sort_of y = sort_of x"
      by simp_all
    let ?q = "(x \<rightleftharpoons> y) + p"
    have q: "?q \<bullet> insert x Xs = insert y (p \<bullet> Xs)"
      unfolding insert_eqvt
      using `p \<bullet> x = x` `sort_of y = sort_of x`
      using `x \<notin> p \<bullet> Xs` `y \<notin> p \<bullet> Xs`
      by (simp add: swap_atom swap_set_not_in)
    have "?q \<bullet> insert x Xs \<inter> As = {}"
      using `y \<notin> As` `p \<bullet> Xs \<inter> As = {}`
      unfolding q by simp
    moreover
    have "supp ?q \<subseteq> insert x Xs \<union> ?q \<bullet> insert x Xs"
      using p2 unfolding q
      by (intro subset_trans [OF supp_plus_perm])
         (auto simp add: supp_swap)
    ultimately show ?case by blast
  qed
qed

lemma at_set_avoiding:
  assumes a: "finite Xs"
  and     b: "finite (supp c)"
  obtains p::"perm" where "(p \<bullet> Xs)\<sharp>*c" and "(supp p) \<subseteq> (Xs \<union> (p \<bullet> Xs))"
  using a b at_set_avoiding_aux [where Xs="Xs" and As="Xs \<union> supp c"]
  unfolding fresh_star_def fresh_def by blast

lemma at_set_avoiding2:
  assumes "finite xs"
  and     "finite (supp c)" "finite (supp x)"
  and     "xs \<sharp>* x"
  shows "\<exists>p. (p \<bullet> xs) \<sharp>* c \<and> supp x \<sharp>* p"
using assms
apply(erule_tac c="(c, x)" in at_set_avoiding)
apply(simp add: supp_Pair)
apply(rule_tac x="p" in exI)
apply(simp add: fresh_star_prod)
apply(rule fresh_star_supp_conv)
apply(auto simp add: fresh_star_def)
done

lemma at_set_avoiding2_atom:
  assumes "finite (supp c)" "finite (supp x)"
  and     b: "a \<sharp> x"
  shows "\<exists>p. (p \<bullet> a) \<sharp> c \<and> supp x \<sharp>* p"
proof -
  have a: "{a} \<sharp>* x" unfolding fresh_star_def by (simp add: b)
  obtain p where p1: "(p \<bullet> {a}) \<sharp>* c" and p2: "supp x \<sharp>* p"
    using at_set_avoiding2[of "{a}" "c" "x"] assms a by blast
  have c: "(p \<bullet> a) \<sharp> c" using p1
    unfolding fresh_star_def Ball_def 
    by(erule_tac x="p \<bullet> a" in allE) (simp add: permute_set_eq)
  hence "p \<bullet> a \<sharp> c \<and> supp x \<sharp>* p" using p2 by blast
  then show "\<exists>p. (p \<bullet> a) \<sharp> c \<and> supp x \<sharp>* p" by blast
qed


section {* Concrete Atoms Types *}

text {*
  Class @{text at_base} allows types containing multiple sorts of atoms.
  Class @{text at} only allows types with a single sort.
*}

class at_base = pt +
  fixes atom :: "'a \<Rightarrow> atom"
  assumes atom_eq_iff [simp]: "atom a = atom b \<longleftrightarrow> a = b"
  assumes atom_eqvt: "p \<bullet> (atom a) = atom (p \<bullet> a)"

class at = at_base +
  assumes sort_of_atom_eq [simp]: "sort_of (atom a) = sort_of (atom b)"

lemma supp_at_base: 
  fixes a::"'a::at_base"
  shows "supp a = {atom a}"
  by (simp add: supp_atom [symmetric] supp_def atom_eqvt)

lemma fresh_at_base: 
  shows "a \<sharp> b \<longleftrightarrow> a \<noteq> atom b"
  unfolding fresh_def by (simp add: supp_at_base)

instance at_base < fs
proof qed (simp add: supp_at_base)

lemma at_base_infinite [simp]:
  shows "infinite (UNIV :: 'a::at_base set)" (is "infinite ?U")
proof
  obtain a :: 'a where "True" by auto
  assume "finite ?U"
  hence "finite (atom ` ?U)"
    by (rule finite_imageI)
  then obtain b where b: "b \<notin> atom ` ?U" "sort_of b = sort_of (atom a)"
    by (rule obtain_atom)
  from b(2) have "b = atom ((atom a \<rightleftharpoons> b) \<bullet> a)"
    unfolding atom_eqvt [symmetric]
    by (simp add: swap_atom)
  hence "b \<in> atom ` ?U" by simp
  with b(1) show "False" by simp
qed

lemma swap_at_base_simps [simp]:
  fixes x y::"'a::at_base"
  shows "sort_of (atom x) = sort_of (atom y) \<Longrightarrow> (atom x \<rightleftharpoons> atom y) \<bullet> x = y"
  and   "sort_of (atom x) = sort_of (atom y) \<Longrightarrow> (atom x \<rightleftharpoons> atom y) \<bullet> y = x"
  and   "atom x \<noteq> a \<Longrightarrow> atom x \<noteq> b \<Longrightarrow> (a \<rightleftharpoons> b) \<bullet> x = x"
  unfolding atom_eq_iff [symmetric]
  unfolding atom_eqvt [symmetric]
  by simp_all

lemma obtain_at_base:
  assumes X: "finite X"
  obtains a::"'a::at_base" where "atom a \<notin> X"
proof -
  have "inj (atom :: 'a \<Rightarrow> atom)"
    by (simp add: inj_on_def)
  with X have "finite (atom -` X :: 'a set)"
    by (rule finite_vimageI)
  with at_base_infinite have "atom -` X \<noteq> (UNIV :: 'a set)"
    by auto
  then obtain a :: 'a where "atom a \<notin> X"
    by auto
  thus ?thesis ..
qed

lemma supp_finite_set_at_base:
  assumes a: "finite S"
  shows "supp S = atom ` S"
apply(simp add: supp_of_fin_sets[OF a])
apply(simp add: supp_at_base)
apply(auto)
done

section {* Infrastructure for concrete atom types *}

section {* A swapping operation for concrete atoms *}
  
definition
  flip :: "'a::at_base \<Rightarrow> 'a \<Rightarrow> perm" ("'(_ \<leftrightarrow> _')")
where
  "(a \<leftrightarrow> b) = (atom a \<rightleftharpoons> atom b)"

lemma flip_self [simp]: "(a \<leftrightarrow> a) = 0"
  unfolding flip_def by (rule swap_self)

lemma flip_commute: "(a \<leftrightarrow> b) = (b \<leftrightarrow> a)"
  unfolding flip_def by (rule swap_commute)

lemma minus_flip [simp]: "- (a \<leftrightarrow> b) = (a \<leftrightarrow> b)"
  unfolding flip_def by (rule minus_swap)

lemma add_flip_cancel: "(a \<leftrightarrow> b) + (a \<leftrightarrow> b) = 0"
  unfolding flip_def by (rule swap_cancel)

lemma permute_flip_cancel [simp]: "(a \<leftrightarrow> b) \<bullet> (a \<leftrightarrow> b) \<bullet> x = x"
  unfolding permute_plus [symmetric] add_flip_cancel by simp

lemma permute_flip_cancel2 [simp]: "(a \<leftrightarrow> b) \<bullet> (b \<leftrightarrow> a) \<bullet> x = x"
  by (simp add: flip_commute)

lemma flip_eqvt: 
  fixes a b c::"'a::at_base"
  shows "p \<bullet> (a \<leftrightarrow> b) = (p \<bullet> a \<leftrightarrow> p \<bullet> b)"
  unfolding flip_def
  by (simp add: swap_eqvt atom_eqvt)

lemma flip_at_base_simps [simp]:
  shows "sort_of (atom a) = sort_of (atom b) \<Longrightarrow> (a \<leftrightarrow> b) \<bullet> a = b"
  and   "sort_of (atom a) = sort_of (atom b) \<Longrightarrow> (a \<leftrightarrow> b) \<bullet> b = a"
  and   "\<lbrakk>a \<noteq> c; b \<noteq> c\<rbrakk> \<Longrightarrow> (a \<leftrightarrow> b) \<bullet> c = c"
  and   "sort_of (atom a) \<noteq> sort_of (atom b) \<Longrightarrow> (a \<leftrightarrow> b) \<bullet> x = x"
  unfolding flip_def
  unfolding atom_eq_iff [symmetric]
  unfolding atom_eqvt [symmetric]
  by simp_all

text {* the following two lemmas do not hold for at_base, 
  only for single sort atoms from at *}

lemma permute_flip_at:
  fixes a b c::"'a::at"
  shows "(a \<leftrightarrow> b) \<bullet> c = (if c = a then b else if c = b then a else c)"
  unfolding flip_def
  apply (rule atom_eq_iff [THEN iffD1])
  apply (subst atom_eqvt [symmetric])
  apply (simp add: swap_atom)
  done

lemma flip_at_simps [simp]:
  fixes a b::"'a::at"
  shows "(a \<leftrightarrow> b) \<bullet> a = b" 
  and   "(a \<leftrightarrow> b) \<bullet> b = a"
  unfolding permute_flip_at by simp_all

lemma flip_fresh_fresh:
  fixes a b::"'a::at_base"
  assumes "atom a \<sharp> x" "atom b \<sharp> x"
  shows "(a \<leftrightarrow> b) \<bullet> x = x"
using assms
by (simp add: flip_def swap_fresh_fresh)

subsection {* Syntax for coercing at-elements to the atom-type *}

syntax
  "_atom_constrain" :: "logic \<Rightarrow> type \<Rightarrow> logic" ("_:::_" [4, 0] 3)

translations
  "_atom_constrain a t" => "CONST atom (_constrain a t)"


subsection {* A lemma for proving instances of class @{text at}. *}

setup {* Sign.add_const_constraint (@{const_name "permute"}, NONE) *}
setup {* Sign.add_const_constraint (@{const_name "atom"}, NONE) *}

text {*
  New atom types are defined as subtypes of @{typ atom}.
*}

lemma exists_eq_simple_sort: 
  shows "\<exists>a. a \<in> {a. sort_of a = s}"
  by (rule_tac x="Atom s 0" in exI, simp)

lemma exists_eq_sort: 
  shows "\<exists>a. a \<in> {a. sort_of a \<in> range sort_fun}"
  by (rule_tac x="Atom (sort_fun x) y" in exI, simp)

lemma at_base_class:
  fixes sort_fun :: "'b \<Rightarrow>atom_sort"
  fixes Rep :: "'a \<Rightarrow> atom" and Abs :: "atom \<Rightarrow> 'a"
  assumes type: "type_definition Rep Abs {a. sort_of a \<in> range sort_fun}"
  assumes atom_def: "\<And>a. atom a = Rep a"
  assumes permute_def: "\<And>p a. p \<bullet> a = Abs (p \<bullet> Rep a)"
  shows "OFCLASS('a, at_base_class)"
proof
  interpret type_definition Rep Abs "{a. sort_of a \<in> range sort_fun}" by (rule type)
  have sort_of_Rep: "\<And>a. sort_of (Rep a) \<in> range sort_fun" using Rep by simp
  fix a b :: 'a and p p1 p2 :: perm
  show "0 \<bullet> a = a"
    unfolding permute_def by (simp add: Rep_inverse)
  show "(p1 + p2) \<bullet> a = p1 \<bullet> p2 \<bullet> a"
    unfolding permute_def by (simp add: Abs_inverse sort_of_Rep)
  show "atom a = atom b \<longleftrightarrow> a = b"
    unfolding atom_def by (simp add: Rep_inject)
  show "p \<bullet> atom a = atom (p \<bullet> a)"
    unfolding permute_def atom_def by (simp add: Abs_inverse sort_of_Rep)
qed

(*
lemma at_class:
  fixes s :: atom_sort
  fixes Rep :: "'a \<Rightarrow> atom" and Abs :: "atom \<Rightarrow> 'a"
  assumes type: "type_definition Rep Abs {a. sort_of a \<in> range (\<lambda>x::unit. s)}"
  assumes atom_def: "\<And>a. atom a = Rep a"
  assumes permute_def: "\<And>p a. p \<bullet> a = Abs (p \<bullet> Rep a)"
  shows "OFCLASS('a, at_class)"
proof
  interpret type_definition Rep Abs "{a. sort_of a \<in> range (\<lambda>x::unit. s)}" by (rule type)
  have sort_of_Rep: "\<And>a. sort_of (Rep a) = s" using Rep by (simp add: image_def)
  fix a b :: 'a and p p1 p2 :: perm
  show "0 \<bullet> a = a"
    unfolding permute_def by (simp add: Rep_inverse)
  show "(p1 + p2) \<bullet> a = p1 \<bullet> p2 \<bullet> a"
    unfolding permute_def by (simp add: Abs_inverse sort_of_Rep)
  show "sort_of (atom a) = sort_of (atom b)"
    unfolding atom_def by (simp add: sort_of_Rep)
  show "atom a = atom b \<longleftrightarrow> a = b"
    unfolding atom_def by (simp add: Rep_inject)
  show "p \<bullet> atom a = atom (p \<bullet> a)"
    unfolding permute_def atom_def by (simp add: Abs_inverse sort_of_Rep)
qed
*)

lemma at_class:
  fixes s :: atom_sort
  fixes Rep :: "'a \<Rightarrow> atom" and Abs :: "atom \<Rightarrow> 'a"
  assumes type: "type_definition Rep Abs {a. sort_of a = s}"
  assumes atom_def: "\<And>a. atom a = Rep a"
  assumes permute_def: "\<And>p a. p \<bullet> a = Abs (p \<bullet> Rep a)"
  shows "OFCLASS('a, at_class)"
proof
  interpret type_definition Rep Abs "{a. sort_of a = s}" by (rule type)
  have sort_of_Rep: "\<And>a. sort_of (Rep a) = s" using Rep by (simp add: image_def)
  fix a b :: 'a and p p1 p2 :: perm
  show "0 \<bullet> a = a"
    unfolding permute_def by (simp add: Rep_inverse)
  show "(p1 + p2) \<bullet> a = p1 \<bullet> p2 \<bullet> a"
    unfolding permute_def by (simp add: Abs_inverse sort_of_Rep)
  show "sort_of (atom a) = sort_of (atom b)"
    unfolding atom_def by (simp add: sort_of_Rep)
  show "atom a = atom b \<longleftrightarrow> a = b"
    unfolding atom_def by (simp add: Rep_inject)
  show "p \<bullet> atom a = atom (p \<bullet> a)"
    unfolding permute_def atom_def by (simp add: Abs_inverse sort_of_Rep)
qed

setup {* Sign.add_const_constraint
  (@{const_name "permute"}, SOME @{typ "perm \<Rightarrow> 'a::pt \<Rightarrow> 'a"}) *}
setup {* Sign.add_const_constraint
  (@{const_name "atom"}, SOME @{typ "'a::at_base \<Rightarrow> atom"}) *}



section {* The freshness lemma according to Andy Pitts *}

lemma freshness_lemma:
  fixes h :: "'a::at \<Rightarrow> 'b::pt"
  assumes a: "\<exists>a. atom a \<sharp> (h, h a)"
  shows  "\<exists>x. \<forall>a. atom a \<sharp> h \<longrightarrow> h a = x"
proof -
  from a obtain b where a1: "atom b \<sharp> h" and a2: "atom b \<sharp> h b"
    by (auto simp add: fresh_Pair)
  show "\<exists>x. \<forall>a. atom a \<sharp> h \<longrightarrow> h a = x"
  proof (intro exI allI impI)
    fix a :: 'a
    assume a3: "atom a \<sharp> h"
    show "h a = h b"
    proof (cases "a = b")
      assume "a = b"
      thus "h a = h b" by simp
    next
      assume "a \<noteq> b"
      hence "atom a \<sharp> b" by (simp add: fresh_at_base)
      with a3 have "atom a \<sharp> h b" 
        by (rule fresh_fun_app)
      with a2 have d1: "(atom b \<rightleftharpoons> atom a) \<bullet> (h b) = (h b)"
        by (rule swap_fresh_fresh)
      from a1 a3 have d2: "(atom b \<rightleftharpoons> atom a) \<bullet> h = h"
        by (rule swap_fresh_fresh)
      from d1 have "h b = (atom b \<rightleftharpoons> atom a) \<bullet> (h b)" by simp
      also have "\<dots> = ((atom b \<rightleftharpoons> atom a) \<bullet> h) ((atom b \<rightleftharpoons> atom a) \<bullet> b)"
        by (rule permute_fun_app_eq)
      also have "\<dots> = h a"
        using d2 by simp
      finally show "h a = h b"  by simp
    qed
  qed
qed

lemma freshness_lemma_unique:
  fixes h :: "'a::at \<Rightarrow> 'b::pt"
  assumes a: "\<exists>a. atom a \<sharp> (h, h a)"
  shows "\<exists>!x. \<forall>a. atom a \<sharp> h \<longrightarrow> h a = x"
proof (rule ex_ex1I)
  from a show "\<exists>x. \<forall>a. atom a \<sharp> h \<longrightarrow> h a = x"
    by (rule freshness_lemma)
next
  fix x y
  assume x: "\<forall>a. atom a \<sharp> h \<longrightarrow> h a = x"
  assume y: "\<forall>a. atom a \<sharp> h \<longrightarrow> h a = y"
  from a x y show "x = y"
    by (auto simp add: fresh_Pair)
qed

text {* packaging the freshness lemma into a function *}

definition
  fresh_fun :: "('a::at \<Rightarrow> 'b::pt) \<Rightarrow> 'b"
where
  "fresh_fun h = (THE x. \<forall>a. atom a \<sharp> h \<longrightarrow> h a = x)"

lemma fresh_fun_apply:
  fixes h :: "'a::at \<Rightarrow> 'b::pt"
  assumes a: "\<exists>a. atom a \<sharp> (h, h a)"
  assumes b: "atom a \<sharp> h"
  shows "fresh_fun h = h a"
unfolding fresh_fun_def
proof (rule the_equality)
  show "\<forall>a'. atom a' \<sharp> h \<longrightarrow> h a' = h a"
  proof (intro strip)
    fix a':: 'a
    assume c: "atom a' \<sharp> h"
    from a have "\<exists>x. \<forall>a. atom a \<sharp> h \<longrightarrow> h a = x" by (rule freshness_lemma)
    with b c show "h a' = h a" by auto
  qed
next
  fix fr :: 'b
  assume "\<forall>a. atom a \<sharp> h \<longrightarrow> h a = fr"
  with b show "fr = h a" by auto
qed

lemma fresh_fun_apply':
  fixes h :: "'a::at \<Rightarrow> 'b::pt"
  assumes a: "atom a \<sharp> h" "atom a \<sharp> h a"
  shows "fresh_fun h = h a"
  apply (rule fresh_fun_apply)
  apply (auto simp add: fresh_Pair intro: a)
  done

lemma fresh_fun_eqvt:
  fixes h :: "'a::at \<Rightarrow> 'b::pt"
  assumes a: "\<exists>a. atom a \<sharp> (h, h a)"
  shows "p \<bullet> (fresh_fun h) = fresh_fun (p \<bullet> h)"
  using a
  apply (clarsimp simp add: fresh_Pair)
  apply (subst fresh_fun_apply', assumption+)
  apply (drule fresh_permute_iff [where p=p, THEN iffD2])
  apply (drule fresh_permute_iff [where p=p, THEN iffD2])
  apply (simp add: atom_eqvt permute_fun_app_eq [where f=h])
  apply (erule (1) fresh_fun_apply' [symmetric])
  done

lemma fresh_fun_supports:
  fixes h :: "'a::at \<Rightarrow> 'b::pt"
  assumes a: "\<exists>a. atom a \<sharp> (h, h a)"
  shows "(supp h) supports (fresh_fun h)"
  apply (simp add: supports_def fresh_def [symmetric])
  apply (simp add: fresh_fun_eqvt [OF a] swap_fresh_fresh)
  done

notation fresh_fun (binder "FRESH " 10)

lemma FRESH_f_iff:
  fixes P :: "'a::at \<Rightarrow> 'b::pure"
  fixes f :: "'b \<Rightarrow> 'c::pure"
  assumes P: "finite (supp P)"
  shows "(FRESH x. f (P x)) = f (FRESH x. P x)"
proof -
  obtain a::'a where "atom a \<notin> supp P"
    using P by (rule obtain_at_base)
  hence "atom a \<sharp> P"
    by (simp add: fresh_def)
  show "(FRESH x. f (P x)) = f (FRESH x. P x)"
    apply (subst fresh_fun_apply' [where a=a, OF _ pure_fresh])
    apply (cut_tac `atom a \<sharp> P`)
    apply (simp add: fresh_conv_MOST)
    apply (elim MOST_rev_mp, rule MOST_I, clarify)
    apply (simp add: permute_fun_def permute_pure fun_eq_iff)
    apply (subst fresh_fun_apply' [where a=a, OF `atom a \<sharp> P` pure_fresh])
    apply (rule refl)
    done
qed

lemma FRESH_binop_iff:
  fixes P :: "'a::at \<Rightarrow> 'b::pure"
  fixes Q :: "'a::at \<Rightarrow> 'c::pure"
  fixes binop :: "'b \<Rightarrow> 'c \<Rightarrow> 'd::pure"
  assumes P: "finite (supp P)" 
  and     Q: "finite (supp Q)"
  shows "(FRESH x. binop (P x) (Q x)) = binop (FRESH x. P x) (FRESH x. Q x)"
proof -
  from assms have "finite (supp P \<union> supp Q)" by simp
  then obtain a::'a where "atom a \<notin> (supp P \<union> supp Q)"
    by (rule obtain_at_base)
  hence "atom a \<sharp> P" and "atom a \<sharp> Q"
    by (simp_all add: fresh_def)
  show ?thesis
    apply (subst fresh_fun_apply' [where a=a, OF _ pure_fresh])
    apply (cut_tac `atom a \<sharp> P` `atom a \<sharp> Q`)
    apply (simp add: fresh_conv_MOST)
    apply (elim MOST_rev_mp, rule MOST_I, clarify)
    apply (simp add: permute_fun_def permute_pure fun_eq_iff)
    apply (subst fresh_fun_apply' [where a=a, OF `atom a \<sharp> P` pure_fresh])
    apply (subst fresh_fun_apply' [where a=a, OF `atom a \<sharp> Q` pure_fresh])
    apply (rule refl)
    done
qed

lemma FRESH_conj_iff:
  fixes P Q :: "'a::at \<Rightarrow> bool"
  assumes P: "finite (supp P)" and Q: "finite (supp Q)"
  shows "(FRESH x. P x \<and> Q x) \<longleftrightarrow> (FRESH x. P x) \<and> (FRESH x. Q x)"
using P Q by (rule FRESH_binop_iff)

lemma FRESH_disj_iff:
  fixes P Q :: "'a::at \<Rightarrow> bool"
  assumes P: "finite (supp P)" and Q: "finite (supp Q)"
  shows "(FRESH x. P x \<or> Q x) \<longleftrightarrow> (FRESH x. P x) \<or> (FRESH x. Q x)"
using P Q by (rule FRESH_binop_iff)


section {* Library functions for the nominal infrastructure *}

use "nominal_library.ML"


section {* Automation for creating concrete atom types *}

text {* at the moment only single-sort concrete atoms are supported *}

use "nominal_atoms.ML"



end
