theory Fv
imports "Nominal2_Atoms" "Abs" "Perm" "Rsp" "Nominal2_FSet"
begin

(* The bindings data structure:

  Bindings are a list of lists of lists of triples.

   The first list represents the datatypes defined.
   The second list represents the constructors.
   The internal list is a list of all the bndings that
   concern the constructor.

   Every triple consists of a function, the binding and
   the body.

  Eg:
nominal_datatype

   C1
 | C2 x y z bind x in z
 | C3 x y z bind f x in z bind g y in z

yields:
[
 [],
 [(NONE, 0, 2)],
 [(SOME (Const f), 0, 2), (Some (Const g), 1, 2)]]

A SOME binding has to have a function which takes an appropriate
argument and returns an atom set. A NONE binding has to be on an
argument that is an atom or an atom set.
*)

(*
An overview of the generation of free variables:

1) fv_bn functions are generated only for the non-recursive binds.

   An fv_bn for a constructor is a union of values for the arguments:

   For an argument x that is in the bn function
   - if it is a recursive argument bn' we return: fv_bn' x
   - otherwise empty

   For an argument x that is not in the bn function
   - for atom we return: {atom x}
   - for atom set we return: atom ` x
   - for a recursive call to type ty' we return: fv_ty' x
     with fv of the appropriate type
   - otherwise empty

2) fv_ty functions generated for all types being defined:

   fv_ty for a constructor is a union of values for the arguments.

   For an argument that is bound in a shallow binding we return empty.

   For an argument x that bound in a non-recursive deep binding
   we return: fv_bn x.

   Otherwise we return the free variables of the argument minus the
   bound variables of the argument.

   The free variables for an argument x are:
   - for an atom: {atom x}
   - for atom set: atom ` x
   - for recursive call to type ty' return: fv_ty' x
   - for nominal datatype ty' return: fv_ty' x

   The bound variables are a union of results of all bindings that
   involve the given argument. For a paricular binding:

   - for a binding function bn: bn x
   - for a recursive argument of type ty': fv_fy' x
   - for nominal datatype ty' return: fv_ty' x
*)

(*
An overview of the generation of alpha-equivalence:

1) alpha_bn relations are generated for binding functions.

   An alpha_bn for a constructor is true if a conjunction of
   propositions for each argument holds.

   For an argument a proposition is build as follows from
   th:

   - for a recursive argument in the bn function, we return: alpha_bn argl argr
   - for a recursive argument for type ty not in bn, we return: alpha_ty argl argr
   - for other arguments in the bn function we return: True
   - for other arguments not in the bn function we return: argl = argr

2) alpha_ty relations are generated for all the types being defined:

   For each constructor we gather all the arguments that are bound,
   and for each of those we add a permutation. We associate those
   permutations with the bindings. Note that two bindings can have
   the same permutation if the arguments being bound are the same.

   An alpha_ty for a constructor is true if there exist permutations
   as above such that a conjunction of propositions for all arguments holds.

   For an argument we allow bindings where only one of the following
   holds:

   - Argument is bound in some shallow bindings: We return true
   - Argument of type ty is bound recursively in some other
     arguments [i1, .. in] with one binding function bn.
     We return:

     (bn argl, (argl, argl_i1, ..., argl_in)) \<approx>gen
     \<lambda>(argl,argl1,..,argln) (argr,argr1,..,argrn). 
         (alpha_ty argl argr) \<and> (alpha_i1 argl1 argr1) \<and> .. \<and> (alpha_in argln argrn)
     \<lambda>(arg,arg1,..,argn). (fv_ty arg) \<union> (fv_i1 arg1) \<union> .. \<union> (fv_in argn)
     pi
     (bn argr, (argr, argr_i1, ..., argr_in))

   - Argument is bound in some deep non-recursive bindings.
     We return: alpha_bn argl argr
   - Argument of type ty has some shallow bindings [b1..bn] and/or
     non-recursive bindings [f1 a1, .., fm am], where the bindings
     have the permutations p1..pl. We return:

     (b1l \<union>..\<union> bnl \<union> f1 a1l \<union>..\<union> fn anl, argl) \<approx>gen
     alpha_ty fv_ty (p1 +..+ pl)
     (b1r \<union>..\<union> bnr \<union> f1 a1r \<union>..\<union> fn anr, argr)

   - Argument has some recursive bindings. The bindings were
     already treated in 2nd case so we return: True
   - Argument has no bindings and is not bound.
     If it is recursive for type ty, we return: alpha_ty argl argr
     Otherwise we return: argl = argr

*)

ML {*
datatype alpha_type = AlphaGen | AlphaRes | AlphaLst;
*}

ML {*
fun atyp_const AlphaGen = @{const_name alpha_gen}
  | atyp_const AlphaRes = @{const_name alpha_res}
  | atyp_const AlphaLst = @{const_name alpha_lst}
*}

(* TODO: make sure that parser checks that bindings are compatible *)
ML {*
fun alpha_const_for_binds [] = atyp_const AlphaGen
  | alpha_const_for_binds ((NONE, _, _, at) :: t) = atyp_const at
  | alpha_const_for_binds ((SOME (_, _), _, _, at) :: _) = atyp_const at
*}

ML {*
fun is_atom thy typ =
  Sign.of_sort thy (typ, @{sort at})

fun is_atom_set thy (Type ("fun", [t, @{typ bool}])) = is_atom thy t
  | is_atom_set _ _ = false;

fun is_atom_fset thy (Type ("FSet.fset", [t])) = is_atom thy t
  | is_atom_fset _ _ = false;
*}


(* Like map2, only if the second list is empty passes empty lists insted of error *)
ML {*
fun map2i _ [] [] = []
  | map2i f (x :: xs) (y :: ys) = f x y :: map2i f xs ys
  | map2i f (x :: xs) [] = f x [] :: map2i f xs []
  | map2i _ _ _ = raise UnequalLengths;
*}

(* Finds bindings with the same function and binding, and gathers all
   bodys for such pairs
 *)
ML {*
fun gather_binds binds =
let
  fun gather_binds_cons binds =
    let
      val common = map (fn (f, bi, _, aty) => (f, bi, aty)) binds
      val nodups = distinct (op =) common
      fun find_bodys (sf, sbi, sty) =
        filter (fn (f, bi, _, aty) => f = sf andalso bi = sbi andalso aty = sty) binds
      val bodys = map ((map (fn (_, _, bo, _) => bo)) o find_bodys) nodups
    in
      nodups ~~ bodys
    end
in
  map (map gather_binds_cons) binds
end
*}

ML {*
fun un_gather_binds_cons binds =
  flat (map (fn (((f, bi, aty), bos), pi) => map (fn bo => ((f, bi, bo, aty), pi)) bos) binds)
*}

ML {*
  open Datatype_Aux; (* typ_of_dtyp, DtRec, ... *);
*}
ML {*
  (* TODO: It is the same as one in 'nominal_atoms' *)
  fun mk_atom ty = Const (@{const_name atom}, ty --> @{typ atom});
  val noatoms = @{term "{} :: atom set"};
  fun mk_single_atom x = HOLogic.mk_set @{typ atom} [mk_atom (type_of x) $ x];
  fun mk_union sets =
    fold (fn a => fn b =>
      if a = noatoms then b else
      if b = noatoms then a else
      if a = b then a else
      HOLogic.mk_binop @{const_name sup} (a, b)) (rev sets) noatoms;
  val mk_inter = foldr1 (HOLogic.mk_binop @{const_name inf})
  fun mk_diff a b =
    if b = noatoms then a else
    if b = a then noatoms else
    HOLogic.mk_binop @{const_name minus} (a, b);
  fun mk_atom_set t =
    let
      val ty = fastype_of t;
      val atom_ty = HOLogic.dest_setT ty --> @{typ atom};
      val img_ty = atom_ty --> ty --> @{typ "atom set"};
    in
      (Const (@{const_name image}, img_ty) $ Const (@{const_name atom}, atom_ty) $ t)
    end;
  fun mk_atom_fset t =
    let
      val ty = fastype_of t;
      val atom_ty = dest_fsetT ty --> @{typ atom};
      val fmap_ty = atom_ty --> ty --> @{typ "atom fset"};
      val fset_to_set = @{term "fset_to_set :: atom fset \<Rightarrow> atom set"}
    in
      fset_to_set $ ((Const (@{const_name fmap}, fmap_ty) $ Const (@{const_name atom}, atom_ty) $ t))
    end;
  (* Similar to one in USyntax *)
  fun mk_pair (fst, snd) =
    let val ty1 = fastype_of fst
      val ty2 = fastype_of snd
      val c = HOLogic.pair_const ty1 ty2
    in c $ fst $ snd
    end;
*}

(* Given [fv1, fv2, fv3] creates %(x, y, z). fv1 x u fv2 y u fv3 z *)
ML {*
fun mk_compound_fv fvs =
let
  val nos = (length fvs - 1) downto 0;
  val fvs_applied = map (fn (fv, no) => fv $ Bound no) (fvs ~~ nos);
  val fvs_union = mk_union fvs_applied;
  val (tyh :: tys) = rev (map (domain_type o fastype_of) fvs);
  fun fold_fun ty t = HOLogic.mk_split (Abs ("", ty, t))
in
  fold fold_fun tys (Abs ("", tyh, fvs_union))
end;
*}

(* Given [R1, R2, R3] creates %(x,x'). %(y,y'). %(z,z'). R x x' \<and> R y y' \<and> R z z' *)
ML {*
fun mk_compound_alpha Rs =
let
  val nos = (length Rs - 1) downto 0;
  val nos2 = (2 * length Rs - 1) downto length Rs;
  val Rs_applied = map (fn (R, (no2, no)) => R $ Bound no2 $ Bound no) (Rs ~~ (nos2 ~~ nos));
  val Rs_conj = mk_conjl Rs_applied;
  val (tyh :: tys) = rev (map (domain_type o fastype_of) Rs);
  fun fold_fun ty t = HOLogic.mk_split (Abs ("", ty, t))
  val abs_rhs = fold fold_fun tys (Abs ("", tyh, Rs_conj))
in
  fold fold_fun tys (Abs ("", tyh, abs_rhs))
end;
*}

ML {* fun add_perm (p1, p2) = Const(@{const_name plus}, @{typ "perm \<Rightarrow> perm \<Rightarrow> perm"}) $ p1 $ p2 *}

ML {*
fun non_rec_binds l =
let
  fun is_non_rec (SOME (f, false), _, _, _) = SOME f
    | is_non_rec _ = NONE
in
  distinct (op =) (map_filter is_non_rec (flat (flat l)))
end
*}

(* We assume no bindings in the type on which bn is defined *)
(* TODO: currently works only with current fv_bn function *)
ML {*
fun fv_bn thy (dt_info : Datatype_Aux.info) fv_frees bn_fvbn (fvbn, (bn, ith_dtyp, args_in_bns)) =
let
  val {descr, sorts, ...} = dt_info;
  fun nth_dtyp i = typ_of_dtyp descr sorts (DtRec i);
  fun fv_bn_constr (cname, dts) args_in_bn =
  let
    val Ts = map (typ_of_dtyp descr sorts) dts;
    val names = Datatype_Prop.make_tnames Ts;
    val args = map Free (names ~~ Ts);
    val c = Const (cname, Ts ---> (nth_dtyp ith_dtyp));
    fun fv_arg ((dt, x), arg_no) =
      let
        val ty = fastype_of x
(*        val _ = tracing ("B 1" ^ PolyML.makestring args_in_bn);*)
(*        val _ = tracing ("B 2" ^ PolyML.makestring bn_fvbn);*)
      in
        case AList.lookup (op=) args_in_bn arg_no of
          SOME NONE => @{term "{} :: atom set"}
        | SOME (SOME (f : term)) => (the (AList.lookup (op=) bn_fvbn f)) $ x
        | NONE =>
            if is_atom thy ty then mk_single_atom x else
            if is_atom_set thy ty then mk_atom_set x else
            if is_atom_fset thy ty then mk_atom_fset x else
            if is_rec_type dt then nth fv_frees (body_index dt) $ x else
            @{term "{} :: atom set"}
      end;
    val arg_nos = 0 upto (length dts - 1)
  in
    HOLogic.mk_Trueprop (HOLogic.mk_eq
      (fvbn $ list_comb (c, args), mk_union (map fv_arg (dts ~~ args ~~ arg_nos))))
  end;
  val (_, (_, _, constrs)) = nth descr ith_dtyp;
  val eqs = map2i fv_bn_constr constrs args_in_bns
in
  ((bn, fvbn), eqs)
end
*}

ML {* print_depth 100 *}
ML {*
fun fv_bns thy dt_info fv_frees rel_bns =
let
  fun mk_fvbn_free (bn, ith, _) =
    let
      val fvbn_name = "fv_" ^ (Long_Name.base_name (fst (dest_Const bn)));
    in
      (fvbn_name, Free (fvbn_name, fastype_of (nth fv_frees ith)))
    end;
  val (fvbn_names, fvbn_frees) = split_list (map mk_fvbn_free rel_bns);
  val bn_fvbn = (map (fn (bn, _, _) => bn) rel_bns) ~~ fvbn_frees
  val (l1, l2) = split_list (map (fv_bn thy dt_info fv_frees bn_fvbn) (fvbn_frees ~~ rel_bns));
in
  (l1, (fvbn_names ~~ l2))
end
*}


ML {*
fun alpha_bn (dt_info : Datatype_Aux.info) alpha_frees bn_alphabn ((bn, ith_dtyp, args_in_bns), (alpha_bn_free, _ (*is_rec*) )) =
let
  val {descr, sorts, ...} = dt_info;
  fun nth_dtyp i = typ_of_dtyp descr sorts (DtRec i);
  fun alpha_bn_constr (cname, dts) args_in_bn =
  let
    val Ts = map (typ_of_dtyp descr sorts) dts;
    val names = Name.variant_list ["pi"] (Datatype_Prop.make_tnames Ts);
    val names2 = Name.variant_list ("pi" :: names) (Datatype_Prop.make_tnames Ts);
    val args = map Free (names ~~ Ts);
    val args2 = map Free (names2 ~~ Ts);
    val c = Const (cname, Ts ---> (nth_dtyp ith_dtyp));
    val rhs = HOLogic.mk_Trueprop
      (alpha_bn_free $ (list_comb (c, args)) $ (list_comb (c, args2)));
    fun lhs_arg ((dt, arg_no), (arg, arg2)) =
      case AList.lookup (op=) args_in_bn arg_no of
        SOME NONE => @{term True}
      | SOME (SOME f) => (the (AList.lookup (op=) bn_alphabn f)) $ arg $ arg2
      | NONE =>
          if is_rec_type dt then (nth alpha_frees (body_index dt)) $ arg $ arg2
          else HOLogic.mk_eq (arg, arg2)
    val arg_nos = 0 upto (length dts - 1)
    val lhss = mk_conjl (map lhs_arg (dts ~~ arg_nos ~~ (args ~~ args2)))
    val eq = Logic.mk_implies (HOLogic.mk_Trueprop lhss, rhs)
  in
    eq
  end
  val (_, (_, _, constrs)) = nth descr ith_dtyp;
  val eqs = map2i alpha_bn_constr constrs args_in_bns
in
  ((bn, alpha_bn_free), eqs)
end
*}

ML {*
fun alpha_bns dt_info alpha_frees rel_bns bns_rec =
let
  val {descr, sorts, ...} = dt_info;
  fun nth_dtyp i = typ_of_dtyp descr sorts (DtRec i);
  fun mk_alphabn_free (bn, ith, _) =
    let
      val alphabn_name = "alpha_" ^ (Long_Name.base_name (fst (dest_Const bn)));
      val alphabn_type = nth_dtyp ith --> nth_dtyp ith --> @{typ bool};
      val alphabn_free = Free(alphabn_name, alphabn_type);
    in
      (alphabn_name, alphabn_free)
    end;
  val (alphabn_names, alphabn_frees) = split_list (map mk_alphabn_free rel_bns);
  val bn_alphabn = (map (fn (bn, _, _) => bn) rel_bns) ~~ alphabn_frees;
  val pair = split_list (map (alpha_bn dt_info alpha_frees bn_alphabn)
    (rel_bns ~~ (alphabn_frees ~~ bns_rec)))
in
  (alphabn_names, pair)
end
*}


(* Checks that a list of bindings contains only compatible ones *)
ML {*
fun bns_same l =
  length (distinct (op =) (map (fn ((b, _, _, atyp), _) => (b, atyp)) l)) = 1
*}

ML {*
fun setify x =
  if fastype_of x = @{typ "atom list"} then
  Const (@{const_name set}, @{typ "atom list \<Rightarrow> atom set"}) $ x else x
*}

(* TODO: Notice datatypes without bindings and replace alpha with equality *)
ML {*
fun define_fv_alpha (dt_info : Datatype_Aux.info) bindsall bns lthy =
let
  val thy = ProofContext.theory_of lthy;
  val {descr, sorts, ...} = dt_info;
  fun nth_dtyp i = typ_of_dtyp descr sorts (DtRec i);
  val fv_names = Datatype_Prop.indexify_names (map (fn (i, _) =>
    "fv_" ^ name_of_typ (nth_dtyp i)) descr);
  val fv_types = map (fn (i, _) => nth_dtyp i --> @{typ "atom set"}) descr;
  val fv_frees = map Free (fv_names ~~ fv_types);
(* TODO: We need a transitive closure, but instead we do this hack considering
   all binding functions as recursive or not *)
  val nr_bns =
    if (non_rec_binds bindsall) = [] then []
    else map (fn (bn, _, _) => bn) bns;
  val rel_bns = filter (fn (bn, _, _) => bn mem nr_bns) bns;
  val (bn_fv_bns, fv_bn_names_eqs) = fv_bns thy dt_info fv_frees rel_bns;
  val fvbns = map snd bn_fv_bns;
  val (fv_bn_names, fv_bn_eqs) = split_list fv_bn_names_eqs;
  val alpha_names = Datatype_Prop.indexify_names (map (fn (i, _) =>
    "alpha_" ^ name_of_typ (nth_dtyp i)) descr);
  val alpha_types = map (fn (i, _) => nth_dtyp i --> nth_dtyp i --> @{typ bool}) descr;
  val alpha_frees = map Free (alpha_names ~~ alpha_types);
  (* We assume that a bn is either recursive or not *)
  val bns_rec = map (fn (bn, _, _) => not (bn mem nr_bns)) bns;
  val (alpha_bn_names, (bn_alpha_bns, alpha_bn_eqs)) =
    alpha_bns dt_info alpha_frees bns bns_rec
  val alpha_bn_frees = map snd bn_alpha_bns;
  val alpha_bn_types = map fastype_of alpha_bn_frees;

  fun fv_alpha_constr ith_dtyp (cname, dts) bindcs =
    let
      val Ts = map (typ_of_dtyp descr sorts) dts;
      val bindslen = length bindcs
      val pi_strs_same = replicate bindslen "pi"
      val pi_strs = Name.variant_list [] pi_strs_same;
      val pis = map (fn ps => Free (ps, @{typ perm})) pi_strs;
      val bind_pis_gath = bindcs ~~ pis;
      val bind_pis = un_gather_binds_cons bind_pis_gath;
      val bindcs = map fst bind_pis;
      val names = Name.variant_list pi_strs (Datatype_Prop.make_tnames Ts);
      val args = map Free (names ~~ Ts);
      val names2 = Name.variant_list (pi_strs @ names) (Datatype_Prop.make_tnames Ts);
      val args2 = map Free (names2 ~~ Ts);
      val c = Const (cname, Ts ---> (nth_dtyp ith_dtyp));
      val fv_c = nth fv_frees ith_dtyp;
      val alpha = nth alpha_frees ith_dtyp;
      val arg_nos = 0 upto (length dts - 1)
      fun fv_bind args (NONE, i, _, _) =
            if is_rec_type (nth dts i) then (nth fv_frees (body_index (nth dts i))) $ (nth args i) else
            if ((is_atom thy) o fastype_of) (nth args i) then mk_single_atom (nth args i) else
            if ((is_atom_set thy) o fastype_of) (nth args i) then mk_atom_set (nth args i) else
            if ((is_atom_fset thy) o fastype_of) (nth args i) then mk_atom_fset (nth args i) else
            (* TODO we do not know what to do with non-atomizable things *)
            @{term "{} :: atom set"}
        | fv_bind args (SOME (f, _), i, _, _) = f $ (nth args i)
      fun fv_binds args relevant = mk_union (map (fv_bind args) relevant)
      fun fv_binds_as_set args relevant = mk_union (map (setify o fv_bind args) relevant)
      fun find_nonrec_binder j (SOME (f, false), i, _, _) = if i = j then SOME f else NONE
        | find_nonrec_binder _ _ = NONE
      fun fv_arg ((dt, x), arg_no) =
        case get_first (find_nonrec_binder arg_no) bindcs of
          SOME f =>
            (case get_first (fn (x, y) => if x = f then SOME y else NONE) bn_fv_bns of
                SOME fv_bn => fv_bn $ x
              | NONE => error "bn specified in a non-rec binding but not in bn list")
        | NONE =>
            let
              val arg =
                if is_rec_type dt then nth fv_frees (body_index dt) $ x else
                if ((is_atom thy) o fastype_of) x then mk_single_atom x else
                if ((is_atom_set thy) o fastype_of) x then mk_atom_set x else
                if ((is_atom_fset thy) o fastype_of) x then mk_atom_fset x else
                (* TODO we do not know what to do with non-atomizable things *)
                @{term "{} :: atom set"};
              (* If i = j then we generate it only once *)
              val relevant = filter (fn (_, i, j, _) => ((i = arg_no) orelse (j = arg_no))) bindcs;
              val sub = fv_binds_as_set args relevant
            in
              mk_diff arg sub
            end;
      val fv_eq = HOLogic.mk_Trueprop (HOLogic.mk_eq
        (fv_c $ list_comb (c, args), mk_union (map fv_arg  (dts ~~ args ~~ arg_nos))))
      val alpha_rhs =
        HOLogic.mk_Trueprop (alpha $ (list_comb (c, args)) $ (list_comb (c, args2)));
      fun alpha_arg ((dt, arg_no), (arg, arg2)) =
        let
          val rel_in_simp_binds = filter (fn ((NONE, i, _, _), _) => i = arg_no | _ => false) bind_pis;
          val rel_in_comp_binds = filter (fn ((SOME _, i, _, _), _) => i = arg_no | _ => false) bind_pis;
          val rel_has_binds = filter (fn ((NONE, _, j, _), _) => j = arg_no
                                       | ((SOME (_, false), _, j, _), _) => j = arg_no
                                       | _ => false) bind_pis;
          val rel_has_rec_binds = filter
            (fn ((SOME (_, true), _, j, _), _) => j = arg_no | _ => false) bind_pis;
        in
          case (rel_in_simp_binds, rel_in_comp_binds, rel_has_binds, rel_has_rec_binds) of
            ([], [], [], []) =>
              if is_rec_type dt then (nth alpha_frees (body_index dt) $ arg $ arg2)
              else (HOLogic.mk_eq (arg, arg2))
          | (_, [], [], []) => @{term True}
          | ([], [], [], _) => @{term True}
          | ([], ((((SOME (bn, is_rec)), _, _, atyp), _) :: _), [], []) =>
            if not (bns_same rel_in_comp_binds) then error "incompatible bindings for an argument" else
            if is_rec then
              let
                val (rbinds, rpis) = split_list rel_in_comp_binds
                val bound_in_nos = map (fn (_, _, i, _) => i) rbinds
                val bound_in_ty_nos = map (fn i => body_index (nth dts i)) bound_in_nos;
                val bound_args = arg :: map (nth args) bound_in_nos;
                val bound_args2 = arg2 :: map (nth args2) bound_in_nos;
                val lhs_binds = fv_binds args rbinds
                val lhs_arg = foldr1 HOLogic.mk_prod bound_args
                val lhs = mk_pair (lhs_binds, lhs_arg);
                val rhs_binds = fv_binds args2 rbinds;
                val rhs_arg = foldr1 HOLogic.mk_prod bound_args2;
                val rhs = mk_pair (rhs_binds, rhs_arg);
                val fvs = map (nth fv_frees) ((body_index dt) :: bound_in_ty_nos);
                val fv = mk_compound_fv fvs;
                val alphas = map (nth alpha_frees) ((body_index dt) :: bound_in_ty_nos);
                val alpha = mk_compound_alpha alphas;
                val pi = foldr1 add_perm (distinct (op =) rpis);
                val alpha_gen_pre = Const (atyp_const atyp, dummyT) $ lhs $ alpha $ fv $ pi $ rhs;
                val alpha_gen = Syntax.check_term lthy alpha_gen_pre
              in
                alpha_gen
              end
            else
              let
                val alpha_bn_const =
                  nth alpha_bn_frees (find_index (fn (b, _, _) => b = bn) bns)
              in
                alpha_bn_const $ arg $ arg2
              end
          | ([], [], relevant, []) =>
            let
              val (rbinds, rpis) = split_list relevant
              val lhs_binds = fv_binds args rbinds
              val lhs = mk_pair (lhs_binds, arg);
              val rhs_binds = fv_binds args2 rbinds;
              val rhs = mk_pair (rhs_binds, arg2);
              val alpha = nth alpha_frees (body_index dt);
              val fv = nth fv_frees (body_index dt);
              val pi = foldr1 add_perm (distinct (op =) rpis);
              val alpha_const = alpha_const_for_binds rbinds;
              val alpha_gen_pre = Const (alpha_const, dummyT) $ lhs $ alpha $ fv $ pi $ rhs;
              val alpha_gen = Syntax.check_term lthy alpha_gen_pre
            in
              alpha_gen
            end
          | _ => error "Fv.alpha: not supported binding structure"
        end
      val alphas = map alpha_arg (dts ~~ arg_nos ~~ (args ~~ args2))
      val alpha_lhss = mk_conjl alphas
      val alpha_lhss_ex =
        fold (fn pi_str => fn t => HOLogic.mk_exists (pi_str, @{typ perm}, t)) pi_strs alpha_lhss
      val alpha_eq = Logic.mk_implies (HOLogic.mk_Trueprop alpha_lhss_ex, alpha_rhs)
    in
      (fv_eq, alpha_eq)
    end;
  fun fv_alpha_eq (i, (_, _, constrs)) binds = map2i (fv_alpha_constr i) constrs binds;
  val fveqs_alphaeqs = map2i fv_alpha_eq descr (gather_binds bindsall)
  val (fv_eqs_perfv, alpha_eqs) = apsnd flat (split_list (map split_list fveqs_alphaeqs))
  val rel_bns_nos = map (fn (_, i, _) => i) rel_bns;
  fun filter_fun (_, b) = b mem rel_bns_nos;
  val all_fvs = (fv_names ~~ fv_eqs_perfv) ~~ (0 upto (length fv_names - 1))
  val (fv_names_fst, fv_eqs_fst) = apsnd flat (split_list (map fst (filter_out filter_fun all_fvs)))
  val (fv_names_snd, fv_eqs_snd) = apsnd flat (split_list (map fst (filter filter_fun all_fvs)))
  val fv_eqs_all = fv_eqs_fst @ (flat fv_bn_eqs);
  val fv_names_all = fv_names_fst @ fv_bn_names;
  val add_binds = map (fn x => (Attrib.empty_binding, x))
(* Function_Fun.add_fun Function_Common.default_config ... true *)
  val (fvs, lthy') = (Primrec.add_primrec
    (map (fn s => (Binding.name s, NONE, NoSyn)) fv_names_all) (add_binds fv_eqs_all) lthy)
  val (fvs2, lthy'') =
    if fv_eqs_snd = [] then (([], []), lthy') else
   (Primrec.add_primrec
    (map (fn s => (Binding.name s, NONE, NoSyn)) fv_names_snd) (add_binds fv_eqs_snd) lthy')
  val (alphas, lthy''') = (Inductive.add_inductive_i
     {quiet_mode = true, verbose = false, alt_name = Binding.empty,
      coind = false, no_elim = false, no_ind = false, skip_mono = true, fork_mono = false}
     (map2 (fn x => fn y => ((Binding.name x, y), NoSyn)) (alpha_names @ alpha_bn_names)
     (alpha_types @ alpha_bn_types)) []
     (add_binds (alpha_eqs @ flat alpha_bn_eqs)) [] lthy'')
  val ordered_fvs = fv_frees @ fvbns;
  val all_fvs = (fst fvs @ fst fvs2, snd fvs @ snd fvs2)
in
  (((all_fvs, ordered_fvs), alphas), lthy''')
end
*}



ML {*
fun build_alpha_sym_trans_gl alphas (x, y, z) =
let
  fun build_alpha alpha =
    let
      val ty = domain_type (fastype_of alpha);
      val var = Free(x, ty);
      val var2 = Free(y, ty);
      val var3 = Free(z, ty);
      val symp = HOLogic.mk_imp (alpha $ var $ var2, alpha $ var2 $ var);
      val transp = HOLogic.mk_imp (alpha $ var $ var2,
        HOLogic.mk_all (z, ty,
          HOLogic.mk_imp (alpha $ var2 $ var3, alpha $ var $ var3)))
    in
      (symp, transp)
    end;
  val eqs = map build_alpha alphas
  val (sym_eqs, trans_eqs) = split_list eqs
  fun conj l = @{term Trueprop} $ foldr1 HOLogic.mk_conj l
in
  (conj sym_eqs, conj trans_eqs)
end
*}

ML {*
fun build_alpha_refl_gl fv_alphas_lst alphas =
let
  val (fvs_alphas, _) = split_list fv_alphas_lst;
  val (_, alpha_ts) = split_list fvs_alphas;
  val tys = map (domain_type o fastype_of) alpha_ts;
  val names = Datatype_Prop.make_tnames tys;
  val args = map Free (names ~~ tys);
  fun find_alphas ty x =
    domain_type (fastype_of x) = ty;
  fun refl_eq_arg (ty, arg) =
    let
      val rel_alphas = filter (find_alphas ty) alphas;
    in
      map (fn x => x $ arg $ arg) rel_alphas
    end;
  (* Flattening loses the induction structure *)
  val eqs = map (foldr1 HOLogic.mk_conj) (map refl_eq_arg (tys ~~ args))
in
  (names, HOLogic.mk_Trueprop (foldr1 HOLogic.mk_conj eqs))
end
*}

ML {*
fun reflp_tac induct eq_iff =
  rtac induct THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps eq_iff) THEN_ALL_NEW
  split_conj_tac THEN_ALL_NEW REPEAT o rtac @{thm exI[of _ "0 :: perm"]}
  THEN_ALL_NEW split_conj_tac THEN_ALL_NEW asm_full_simp_tac (HOL_ss addsimps
     @{thms alphas fresh_star_def fresh_zero_perm permute_zero ball_triv
       add_0_left supp_zero_perm Int_empty_left split_conv})
*}

ML {*
fun build_alpha_refl fv_alphas_lst alphas induct eq_iff ctxt =
let
  val (names, gl) = build_alpha_refl_gl fv_alphas_lst alphas;
  val refl_conj = Goal.prove ctxt names [] gl (fn _ => reflp_tac induct eq_iff 1);
in
  HOLogic.conj_elims refl_conj
end
*}

lemma exi_neg: "\<exists>(pi :: perm). P pi \<Longrightarrow> (\<And>(p :: perm). P p \<Longrightarrow> Q (- p)) \<Longrightarrow> \<exists>pi. Q pi"
apply (erule exE)
apply (rule_tac x="-pi" in exI)
by auto

ML {*
fun symp_tac induct inj eqvt ctxt =
  rel_indtac induct THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps inj) THEN_ALL_NEW split_conj_tac
  THEN_ALL_NEW
  REPEAT o etac @{thm exi_neg}
  THEN_ALL_NEW
  split_conj_tac THEN_ALL_NEW
  asm_full_simp_tac (HOL_ss addsimps @{thms supp_minus_perm minus_add[symmetric]}) THEN_ALL_NEW
  TRY o (resolve_tac @{thms alphas_compose_sym2} ORELSE' resolve_tac @{thms alphas_compose_sym}) THEN_ALL_NEW
  (asm_full_simp_tac (HOL_ss addsimps (eqvt @ all_eqvts ctxt)))
*}


lemma exi_sum: "\<exists>(pi :: perm). P pi \<Longrightarrow> \<exists>(pi :: perm). Q pi \<Longrightarrow> (\<And>(p :: perm) (pi :: perm). P p \<Longrightarrow> Q pi \<Longrightarrow> R (pi + p)) \<Longrightarrow> \<exists>pi. R pi"
apply (erule exE)+
apply (rule_tac x="pia + pi" in exI)
by auto


ML {*
fun eetac rule = 
  Subgoal.FOCUS_PARAMS (fn focus =>
    let
      val concl = #concl focus
      val prems = Logic.strip_imp_prems (term_of concl)
      val exs = filter (fn x => is_ex (HOLogic.dest_Trueprop x)) prems
      val cexs = map (SOME o (cterm_of (ProofContext.theory_of (#context focus)))) exs
      val thins = map (fn cex => Drule.instantiate' [] [cex] Drule.thin_rl) cexs
    in
      (etac rule THEN' RANGE[atac, eresolve_tac thins]) 1
    end
  )
*}

ML {*
fun transp_tac ctxt induct alpha_inj term_inj distinct cases eqvt =
  rel_indtac induct THEN_ALL_NEW
  (TRY o rtac allI THEN' imp_elim_tac cases ctxt) THEN_ALL_NEW
  asm_full_simp_tac (HOL_basic_ss addsimps alpha_inj) THEN_ALL_NEW
  split_conj_tac THEN_ALL_NEW REPEAT o (eetac @{thm exi_sum} ctxt) THEN_ALL_NEW split_conj_tac
  THEN_ALL_NEW (asm_full_simp_tac (HOL_ss addsimps (term_inj @ distinct)))
  THEN_ALL_NEW split_conj_tac THEN_ALL_NEW
  TRY o (eresolve_tac @{thms alphas_compose_trans2} ORELSE' eresolve_tac @{thms alphas_compose_trans}) THEN_ALL_NEW
  (asm_full_simp_tac (HOL_ss addsimps (all_eqvts ctxt @ eqvt @ term_inj @ distinct)))
*}

lemma transpI:
  "(\<And>xa ya. R xa ya \<longrightarrow> (\<forall>z. R ya z \<longrightarrow> R xa z)) \<Longrightarrow> transp R"
  unfolding transp_def
  by blast

ML {*
fun equivp_tac reflps symps transps =
  (*let val _ = tracing (PolyML.makestring (reflps, symps, transps)) in *)
  simp_tac (HOL_ss addsimps @{thms equivp_reflp_symp_transp reflp_def symp_def})
  THEN' rtac conjI THEN' rtac allI THEN'
  resolve_tac reflps THEN'
  rtac conjI THEN' rtac allI THEN' rtac allI THEN'
  resolve_tac symps THEN'
  rtac @{thm transpI} THEN' resolve_tac transps
*}

ML {*
fun build_equivps alphas reflps alpha_induct term_inj alpha_inj distinct cases eqvt ctxt =
let
  val ([x, y, z], ctxt') = Variable.variant_fixes ["x","y","z"] ctxt;
  val (symg, transg) = build_alpha_sym_trans_gl alphas (x, y, z)
  fun symp_tac' _ = symp_tac alpha_induct alpha_inj eqvt ctxt 1;
  fun transp_tac' _ = transp_tac ctxt alpha_induct alpha_inj term_inj distinct cases eqvt 1;
  val symp_loc = Goal.prove ctxt' [] [] symg symp_tac';
  val transp_loc = Goal.prove ctxt' [] [] transg transp_tac';
  val [symp, transp] = Variable.export ctxt' ctxt [symp_loc, transp_loc]
  val symps = HOLogic.conj_elims symp
  val transps = HOLogic.conj_elims transp
  fun equivp alpha =
    let
      val equivp = Const (@{const_name equivp}, fastype_of alpha --> @{typ bool})
      val goal = @{term Trueprop} $ (equivp $ alpha)
      fun tac _ = equivp_tac reflps symps transps 1
    in
      Goal.prove ctxt [] [] goal tac
    end
in
  map equivp alphas
end
*}

lemma not_in_union: "c \<notin> a \<union> b \<equiv> (c \<notin> a \<and> c \<notin> b)"
by auto

ML {*
fun supports_tac perm =
  simp_tac (HOL_ss addsimps @{thms supports_def not_in_union} @ perm) THEN_ALL_NEW (
    REPEAT o rtac allI THEN' REPEAT o rtac impI THEN' split_conj_tac THEN'
    asm_full_simp_tac (HOL_ss addsimps @{thms fresh_def[symmetric]
      swap_fresh_fresh fresh_atom swap_at_base_simps(3) swap_atom_image_fresh
      supp_fset_to_set supp_fmap_atom}))
*}

ML {*
fun mk_supp ty x =
  Const (@{const_name supp}, ty --> @{typ "atom set"}) $ x
*}

ML {*
fun mk_supports_eq thy cnstr =
let
  val (tys, ty) = (strip_type o fastype_of) cnstr
  val names = Datatype_Prop.make_tnames tys
  val frees = map Free (names ~~ tys)
  val rhs = list_comb (cnstr, frees)

  fun mk_supp_arg (x, ty) =
    if is_atom thy ty then mk_supp @{typ atom} (mk_atom ty $ x) else
    if is_atom_set thy ty then mk_supp @{typ "atom set"} (mk_atom_set x) else
    if is_atom_fset thy ty then mk_supp @{typ "atom set"} (mk_atom_fset x)
    else mk_supp ty x
  val lhss = map mk_supp_arg (frees ~~ tys)
  val supports = Const(@{const_name "supports"}, @{typ "atom set"} --> ty --> @{typ bool})
  val eq = HOLogic.mk_Trueprop (supports $ mk_union lhss $ rhs)
in
  (names, eq)
end
*}

ML {*
fun prove_supports ctxt perms cnst =
let
  val (names, eq) = mk_supports_eq (ProofContext.theory_of ctxt) cnst
in
  Goal.prove ctxt names [] eq (fn _ => supports_tac perms 1)
end
*}

ML {*
fun mk_fs tys =
let
  val names = Datatype_Prop.make_tnames tys
  val frees = map Free (names ~~ tys)
  val supps = map2 mk_supp tys frees
  val fin_supps = map (fn x => @{term "finite :: atom set \<Rightarrow> bool"} $ x) supps
in
  (names, HOLogic.mk_Trueprop (mk_conjl fin_supps))
end
*}

ML {*
fun fs_tac induct supports = rel_indtac induct THEN_ALL_NEW (
  rtac @{thm supports_finite} THEN' resolve_tac supports) THEN_ALL_NEW
  asm_full_simp_tac (HOL_ss addsimps @{thms supp_atom supp_atom_image supp_fset_to_set
    supp_fmap_atom finite_insert finite.emptyI finite_Un finite_supp})
*}

ML {*
fun prove_fs ctxt induct supports tys =
let
  val (names, eq) = mk_fs tys
in
  Goal.prove ctxt names [] eq (fn _ => fs_tac induct supports 1)
end
*}

ML {*
fun mk_supp x = Const (@{const_name supp}, fastype_of x --> @{typ "atom set"}) $ x;

fun mk_supp_neq arg (fv, alpha) =
let
  val collect = Const ("Collect", @{typ "(atom \<Rightarrow> bool) \<Rightarrow> atom \<Rightarrow> bool"});
  val ty = fastype_of arg;
  val perm = Const ("Nominal2_Base.pt_class.permute", @{typ perm} --> ty --> ty);
  val finite = @{term "finite :: atom set \<Rightarrow> bool"}
  val rhs = collect $ Abs ("a", @{typ atom},
    HOLogic.mk_not (finite $
      (collect $ Abs ("b", @{typ atom},
        HOLogic.mk_not (alpha $ (perm $ (@{term swap} $ Bound 1 $ Bound 0) $ arg) $ arg)))))
in
  HOLogic.mk_eq (fv $ arg, rhs)
end;

fun supp_eq fv_alphas_lst =
let
  val (fvs_alphas, ls) = split_list fv_alphas_lst;
  val (fv_ts, _) = split_list fvs_alphas;
  val tys = map (domain_type o fastype_of) fv_ts;
  val names = Datatype_Prop.make_tnames tys;
  val args = map Free (names ~~ tys);
  fun supp_eq_arg ((fv, arg), l) =
    mk_conjl
      ((HOLogic.mk_eq (fv $ arg, mk_supp arg)) ::
       (map (mk_supp_neq arg) l))
  val eqs = mk_conjl (map supp_eq_arg ((fv_ts ~~ args) ~~ ls))
in
  (names, HOLogic.mk_Trueprop eqs)
end
*}

ML {*
fun combine_fv_alpha_bns (fv_ts_nobn, fv_ts_bn) (alpha_ts_nobn, alpha_ts_bn) bn_nos =
if length fv_ts_bn < length alpha_ts_bn then
  (fv_ts_nobn ~~ alpha_ts_nobn) ~~ (replicate (length fv_ts_nobn) [])
else let
  val fv_alpha_nos = 0 upto (length fv_ts_nobn - 1);
  fun filter_fn i (x, j) = if j = i then SOME x else NONE;
  val fv_alpha_bn_nos = (fv_ts_bn ~~ alpha_ts_bn) ~~ bn_nos;
  val fv_alpha_bn_all = map (fn i => map_filter (filter_fn i) fv_alpha_bn_nos) fv_alpha_nos;
in
  (fv_ts_nobn ~~ alpha_ts_nobn) ~~ fv_alpha_bn_all
end
*}

(* TODO: this is a hack, it assumes that only one type of Abs's is present
   in the type and chooses this supp_abs. Additionally single atoms are
   treated properly. *)
ML {*
fun choose_alpha_abs eqiff =
let
  fun exists_subterms f ts = true mem (map (exists_subterm f) ts);
  val terms = map prop_of eqiff;
  fun check cname = exists_subterms (fn x => fst(dest_Const x) = cname handle _ => false) terms
  val no =
    if check @{const_name alpha_lst} then 2 else
    if check @{const_name alpha_res} then 1 else
    if check @{const_name alpha_gen} then 0 else
    error "Failure choosing supp_abs"
in
  nth @{thms supp_abs[symmetric]} no
end
*}
lemma supp_abs_atom: "supp (Abs {atom a} (x :: 'a :: fs)) = supp x - {atom a}"
by (rule supp_abs(1))

lemma supp_abs_sum:
  "supp (Abs x (a :: 'a :: fs)) \<union> supp (Abs x (b :: 'b :: fs)) = supp (Abs x (a, b))"
  "supp (Abs_res x (a :: 'a :: fs)) \<union> supp (Abs_res x (b :: 'b :: fs)) = supp (Abs_res x (a, b))"
  "supp (Abs_lst y (a :: 'a :: fs)) \<union> supp (Abs_lst y (b :: 'b :: fs)) = supp (Abs_lst y (a, b))"
  apply (simp_all add: supp_abs supp_Pair)
  apply blast+
  done


ML {*
fun supp_eq_tac ind fv perm eqiff ctxt =
  rel_indtac ind THEN_ALL_NEW
  asm_full_simp_tac (HOL_basic_ss addsimps fv) THEN_ALL_NEW
  asm_full_simp_tac (HOL_basic_ss addsimps @{thms supp_abs_atom[symmetric]}) THEN_ALL_NEW
  asm_full_simp_tac (HOL_basic_ss addsimps [choose_alpha_abs eqiff]) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps @{thms supp_abs_sum}) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps @{thms supp_def}) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps (@{thms permute_abs} @ perm)) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps (@{thms Abs_eq_iff} @ eqiff)) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps @{thms alphas3 alphas2}) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps @{thms alphas}) THEN_ALL_NEW
  asm_full_simp_tac (HOL_basic_ss addsimps (@{thm supp_Pair} :: sym_eqvts ctxt)) THEN_ALL_NEW
  asm_full_simp_tac (HOL_basic_ss addsimps (@{thm Pair_eq} :: all_eqvts ctxt)) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps @{thms supp_at_base[symmetric,simplified supp_def]}) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps @{thms Collect_disj_eq[symmetric]}) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps @{thms infinite_Un[symmetric]}) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps @{thms Collect_disj_eq[symmetric]}) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps @{thms de_Morgan_conj[symmetric]}) THEN_ALL_NEW
  simp_tac (HOL_basic_ss addsimps @{thms ex_simps(1,2)[symmetric]}) THEN_ALL_NEW
  simp_tac (HOL_ss addsimps @{thms Collect_const finite.emptyI})
*}



ML {*
fun build_eqvt_gl pi frees fnctn ctxt =
let
  val typ = domain_type (fastype_of fnctn);
  val arg = the (AList.lookup (op=) frees typ);
in
  ([HOLogic.mk_eq ((perm_arg (fnctn $ arg) $ pi $ (fnctn $ arg)), (fnctn $ (perm_arg arg $ pi $ arg)))], ctxt)
end
*}

ML {*
fun prove_eqvt tys ind simps funs ctxt =
let
  val ([pi], ctxt') = Variable.variant_fixes ["p"] ctxt;
  val pi = Free (pi, @{typ perm});
  val tac = asm_full_simp_tac (HOL_ss addsimps (@{thms atom_eqvt permute_list.simps} @ simps @ all_eqvts ctxt'))
  val ths_loc = prove_by_induct tys (build_eqvt_gl pi) ind tac funs ctxt'
  val ths = Variable.export ctxt' ctxt ths_loc
  val add_eqvt = Attrib.internal (fn _ => Nominal_ThmDecls.eqvt_add)
in
  (ths, snd (Local_Theory.note ((Binding.empty, [add_eqvt]), ths) ctxt))
end
*}

end
