(*  Title:      Nominal2_Base
    Authors:    Brian Huffman, Christian Urban

    Basic definitions and lemma infrastructure for 
    Nominal Isabelle. 
*)
theory Nominal2_Base
imports Main Infinite_Set
uses ("nominal_library.ML")
begin

section {* Atoms and Sorts *}

text {* A simple implementation for atom_sorts is strings. *}
(* types atom_sort = string *)

text {* To deal with Church-like binding we use trees of  
  strings as sorts. *}

datatype atom_sort = Sort "string" "atom_sort list"

datatype atom = Atom atom_sort nat


text {* Basic projection function. *}

primrec
  sort_of :: "atom \<Rightarrow> atom_sort"
where
  "sort_of (Atom s i) = s"

primrec
  nat_of :: "atom \<Rightarrow> nat"
where
  "nat_of (Atom s n) = n"


text {* There are infinitely many atoms of each sort. *}
lemma INFM_sort_of_eq: 
  shows "INFM a. sort_of a = s"
proof -
  have "INFM i. sort_of (Atom s i) = s" by simp
  moreover have "inj (Atom s)" by (simp add: inj_on_def)
  ultimately show "INFM a. sort_of a = s" by (rule INFM_inj)
qed

lemma infinite_sort_of_eq:
  shows "infinite {a. sort_of a = s}"
  using INFM_sort_of_eq unfolding INFM_iff_infinite .

lemma atom_infinite [simp]: 
  shows "infinite (UNIV :: atom set)"
  using subset_UNIV infinite_sort_of_eq
  by (rule infinite_super)

lemma obtain_atom:
  fixes X :: "atom set"
  assumes X: "finite X"
  obtains a where "a \<notin> X" "sort_of a = s"
proof -
  from X have "MOST a. a \<notin> X"
    unfolding MOST_iff_cofinite by simp
  with INFM_sort_of_eq
  have "INFM a. sort_of a = s \<and> a \<notin> X"
    by (rule INFM_conjI)
  then obtain a where "a \<notin> X" "sort_of a = s"
    by (auto elim: INFM_E)
  then show ?thesis ..
qed

lemma atom_components_eq_iff:
  fixes a b :: atom
  shows "a = b \<longleftrightarrow> sort_of a = sort_of b \<and> nat_of a = nat_of b"
  by (induct a, induct b, simp)

section {* Sort-Respecting Permutations *}

typedef perm =
  "{f. bij f \<and> finite {a. f a \<noteq> a} \<and> (\<forall>a. sort_of (f a) = sort_of a)}"
proof
  show "id \<in> ?perm" by simp
qed

lemma permI:
  assumes "bij f" and "MOST x. f x = x" and "\<And>a. sort_of (f a) = sort_of a"
  shows "f \<in> perm"
  using assms unfolding perm_def MOST_iff_cofinite by simp

lemma perm_is_bij: "f \<in> perm \<Longrightarrow> bij f"
  unfolding perm_def by simp

lemma perm_is_finite: "f \<in> perm \<Longrightarrow> finite {a. f a \<noteq> a}"
  unfolding perm_def by simp

lemma perm_is_sort_respecting: "f \<in> perm \<Longrightarrow> sort_of (f a) = sort_of a"
  unfolding perm_def by simp

lemma perm_MOST: "f \<in> perm \<Longrightarrow> MOST x. f x = x"
  unfolding perm_def MOST_iff_cofinite by simp

lemma perm_id: "id \<in> perm"
  unfolding perm_def by simp

lemma perm_comp:
  assumes f: "f \<in> perm" and g: "g \<in> perm"
  shows "(f \<circ> g) \<in> perm"
apply (rule permI)
apply (rule bij_comp)
apply (rule perm_is_bij [OF g])
apply (rule perm_is_bij [OF f])
apply (rule MOST_rev_mp [OF perm_MOST [OF g]])
apply (rule MOST_rev_mp [OF perm_MOST [OF f]])
apply (simp)
apply (simp add: perm_is_sort_respecting [OF f])
apply (simp add: perm_is_sort_respecting [OF g])
done

lemma perm_inv:
  assumes f: "f \<in> perm"
  shows "(inv f) \<in> perm"
apply (rule permI)
apply (rule bij_imp_bij_inv)
apply (rule perm_is_bij [OF f])
apply (rule MOST_mono [OF perm_MOST [OF f]])
apply (erule subst, rule inv_f_f)
apply (rule bij_is_inj [OF perm_is_bij [OF f]])
apply (rule perm_is_sort_respecting [OF f, THEN sym, THEN trans])
apply (simp add: surj_f_inv_f [OF bij_is_surj [OF perm_is_bij [OF f]]])
done

lemma bij_Rep_perm: "bij (Rep_perm p)"
  using Rep_perm [of p] unfolding perm_def by simp

lemma finite_Rep_perm: "finite {a. Rep_perm p a \<noteq> a}"
  using Rep_perm [of p] unfolding perm_def by simp

lemma sort_of_Rep_perm: "sort_of (Rep_perm p a) = sort_of a"
  using Rep_perm [of p] unfolding perm_def by simp

lemma Rep_perm_ext:
  "Rep_perm p1 = Rep_perm p2 \<Longrightarrow> p1 = p2"
  by (simp add: expand_fun_eq Rep_perm_inject [symmetric])


subsection {* Permutations form a group *}

instantiation perm :: group_add
begin

definition
  "0 = Abs_perm id"

definition
  "- p = Abs_perm (inv (Rep_perm p))"

definition
  "p + q = Abs_perm (Rep_perm p \<circ> Rep_perm q)"

definition
  "(p1::perm) - p2 = p1 + - p2"

lemma Rep_perm_0: "Rep_perm 0 = id"
  unfolding zero_perm_def
  by (simp add: Abs_perm_inverse perm_id)

lemma Rep_perm_add:
  "Rep_perm (p1 + p2) = Rep_perm p1 \<circ> Rep_perm p2"
  unfolding plus_perm_def
  by (simp add: Abs_perm_inverse perm_comp Rep_perm)

lemma Rep_perm_uminus:
  "Rep_perm (- p) = inv (Rep_perm p)"
  unfolding uminus_perm_def
  by (simp add: Abs_perm_inverse perm_inv Rep_perm)

instance
apply default
unfolding Rep_perm_inject [symmetric]
unfolding minus_perm_def
unfolding Rep_perm_add
unfolding Rep_perm_uminus
unfolding Rep_perm_0
by (simp_all add: o_assoc inv_o_cancel [OF bij_is_inj [OF bij_Rep_perm]])

end


section {* Implementation of swappings *}

definition
  swap :: "atom \<Rightarrow> atom \<Rightarrow> perm" ("'(_ \<rightleftharpoons> _')")
where
  "(a \<rightleftharpoons> b) =
    Abs_perm (if sort_of a = sort_of b 
              then (\<lambda>c. if a = c then b else if b = c then a else c) 
              else id)"

lemma Rep_perm_swap:
  "Rep_perm (a \<rightleftharpoons> b) =
    (if sort_of a = sort_of b 
     then (\<lambda>c. if a = c then b else if b = c then a else c)
     else id)"
unfolding swap_def
apply (rule Abs_perm_inverse)
apply (rule permI)
apply (auto simp add: bij_def inj_on_def surj_def)[1]
apply (rule MOST_rev_mp [OF MOST_neq(1) [of a]])
apply (rule MOST_rev_mp [OF MOST_neq(1) [of b]])
apply (simp)
apply (simp)
done

lemmas Rep_perm_simps =
  Rep_perm_0
  Rep_perm_add
  Rep_perm_uminus
  Rep_perm_swap

lemma swap_different_sorts [simp]:
  "sort_of a \<noteq> sort_of b \<Longrightarrow> (a \<rightleftharpoons> b) = 0"
  by (rule Rep_perm_ext) (simp add: Rep_perm_simps)

lemma swap_cancel:
  "(a \<rightleftharpoons> b) + (a \<rightleftharpoons> b) = 0"
  by (rule Rep_perm_ext) 
     (simp add: Rep_perm_simps expand_fun_eq)

lemma swap_self [simp]:
  "(a \<rightleftharpoons> a) = 0"
  by (rule Rep_perm_ext, simp add: Rep_perm_simps expand_fun_eq)

lemma minus_swap [simp]:
  "- (a \<rightleftharpoons> b) = (a \<rightleftharpoons> b)"
  by (rule minus_unique [OF swap_cancel])

lemma swap_commute:
  "(a \<rightleftharpoons> b) = (b \<rightleftharpoons> a)"
  by (rule Rep_perm_ext)
     (simp add: Rep_perm_swap expand_fun_eq)

lemma swap_triple:
  assumes "a \<noteq> b" and "c \<noteq> b"
  assumes "sort_of a = sort_of b" "sort_of b = sort_of c"
  shows "(a \<rightleftharpoons> c) + (b \<rightleftharpoons> c) + (a \<rightleftharpoons> c) = (a \<rightleftharpoons> b)"
  using assms
  by (rule_tac Rep_perm_ext)
     (auto simp add: Rep_perm_simps expand_fun_eq)


section {* Permutation Types *}

text {*
  Infix syntax for @{text permute} has higher precedence than
  addition, but lower than unary minus.
*}

class pt =
  fixes permute :: "perm \<Rightarrow> 'a \<Rightarrow> 'a" ("_ \<bullet> _" [76, 75] 75)
  assumes permute_zero [simp]: "0 \<bullet> x = x"
  assumes permute_plus [simp]: "(p + q) \<bullet> x = p \<bullet> (q \<bullet> x)"
begin

lemma permute_diff [simp]:
  shows "(p - q) \<bullet> x = p \<bullet> - q \<bullet> x"
  unfolding diff_minus by simp

lemma permute_minus_cancel [simp]:
  shows "p \<bullet> - p \<bullet> x = x"
  and   "- p \<bullet> p \<bullet> x = x"
  unfolding permute_plus [symmetric] by simp_all

lemma permute_swap_cancel [simp]:
  shows "(a \<rightleftharpoons> b) \<bullet> (a \<rightleftharpoons> b) \<bullet> x = x"
  unfolding permute_plus [symmetric]
  by (simp add: swap_cancel)

lemma permute_swap_cancel2 [simp]:
  shows "(a \<rightleftharpoons> b) \<bullet> (b \<rightleftharpoons> a) \<bullet> x = x"
  unfolding permute_plus [symmetric]
  by (simp add: swap_commute)

lemma inj_permute [simp]: 
  shows "inj (permute p)"
  by (rule inj_on_inverseI)
     (rule permute_minus_cancel)

lemma surj_permute [simp]: 
  shows "surj (permute p)"
  by (rule surjI, rule permute_minus_cancel)

lemma bij_permute [simp]: 
  shows "bij (permute p)"
  by (rule bijI [OF inj_permute surj_permute])

lemma inv_permute: 
  shows "inv (permute p) = permute (- p)"
  by (rule inv_equality) (simp_all)

lemma permute_minus: 
  shows "permute (- p) = inv (permute p)"
  by (simp add: inv_permute)

lemma permute_eq_iff [simp]: 
  shows "p \<bullet> x = p \<bullet> y \<longleftrightarrow> x = y"
  by (rule inj_permute [THEN inj_eq])

end

subsection {* Permutations for atoms *}

instantiation atom :: pt
begin

definition
  "p \<bullet> a = (Rep_perm p) a"

instance 
apply(default)
apply(simp_all add: permute_atom_def Rep_perm_simps)
done

end

lemma sort_of_permute [simp]:
  shows "sort_of (p \<bullet> a) = sort_of a"
  unfolding permute_atom_def by (rule sort_of_Rep_perm)

lemma swap_atom:
  shows "(a \<rightleftharpoons> b) \<bullet> c =
           (if sort_of a = sort_of b
            then (if c = a then b else if c = b then a else c) else c)"
  unfolding permute_atom_def
  by (simp add: Rep_perm_swap)

lemma swap_atom_simps [simp]:
  "sort_of a = sort_of b \<Longrightarrow> (a \<rightleftharpoons> b) \<bullet> a = b"
  "sort_of a = sort_of b \<Longrightarrow> (a \<rightleftharpoons> b) \<bullet> b = a"
  "c \<noteq> a \<Longrightarrow> c \<noteq> b \<Longrightarrow> (a \<rightleftharpoons> b) \<bullet> c = c"
  unfolding swap_atom by simp_all

lemma expand_perm_eq:
  fixes p q :: "perm"
  shows "p = q \<longleftrightarrow> (\<forall>a::atom. p \<bullet> a = q \<bullet> a)"
  unfolding permute_atom_def
  by (metis Rep_perm_ext ext)


subsection {* Permutations for permutations *}

instantiation perm :: pt
begin

definition
  "p \<bullet> q = p + q - p"

instance
apply default
apply (simp add: permute_perm_def)
apply (simp add: permute_perm_def diff_minus minus_add add_assoc)
done

end

lemma permute_self: 
  shows "p \<bullet> p = p"
  unfolding permute_perm_def 
  by (simp add: diff_minus add_assoc)

lemma permute_eqvt:
  shows "p \<bullet> (q \<bullet> x) = (p \<bullet> q) \<bullet> (p \<bullet> x)"
  unfolding permute_perm_def by simp

lemma zero_perm_eqvt:
  shows "p \<bullet> (0::perm) = 0"
  unfolding permute_perm_def by simp

lemma add_perm_eqvt:
  fixes p p1 p2 :: perm
  shows "p \<bullet> (p1 + p2) = p \<bullet> p1 + p \<bullet> p2"
  unfolding permute_perm_def
  by (simp add: expand_perm_eq)

lemma swap_eqvt:
  shows "p \<bullet> (a \<rightleftharpoons> b) = (p \<bullet> a \<rightleftharpoons> p \<bullet> b)"
  unfolding permute_perm_def
  by (auto simp add: swap_atom expand_perm_eq)


subsection {* Permutations for functions *}

instantiation "fun" :: (pt, pt) pt
begin

definition
  "p \<bullet> f = (\<lambda>x. p \<bullet> (f (- p \<bullet> x)))"

instance
apply default
apply (simp add: permute_fun_def)
apply (simp add: permute_fun_def minus_add)
done

end

lemma permute_fun_app_eq:
  shows "p \<bullet> (f x) = (p \<bullet> f) (p \<bullet> x)"
  unfolding permute_fun_def by simp


subsection {* Permutations for booleans *}

instantiation bool :: pt
begin

definition "p \<bullet> (b::bool) = b"

instance
apply(default) 
apply(simp_all add: permute_bool_def)
done

end

lemma Not_eqvt:
  shows "p \<bullet> (\<not> A) = (\<not> (p \<bullet> A))"
by (simp add: permute_bool_def)

lemma permute_boolE:
  fixes P::"bool"
  shows "p \<bullet> P \<Longrightarrow> P"
  by (simp add: permute_bool_def)

lemma permute_boolI:
  fixes P::"bool"
  shows "P \<Longrightarrow> p \<bullet> P"
  by(simp add: permute_bool_def)

subsection {* Permutations for sets *}

lemma permute_set_eq:
  fixes x::"'a::pt"
  and   p::"perm"
  shows "(p \<bullet> X) = {p \<bullet> x | x. x \<in> X}"
  unfolding permute_fun_def
  unfolding permute_bool_def
  apply(auto simp add: mem_def)
  apply(rule_tac x="- p \<bullet> x" in exI)
  apply(simp)
  done

lemma permute_set_eq_image:
  shows "p \<bullet> X = permute p ` X"
  unfolding permute_set_eq by auto

lemma permute_set_eq_vimage:
  shows "p \<bullet> X = permute (- p) -` X"
  unfolding permute_fun_def permute_bool_def
  unfolding vimage_def Collect_def mem_def ..

lemma swap_set_not_in:
  assumes a: "a \<notin> S" "b \<notin> S"
  shows "(a \<rightleftharpoons> b) \<bullet> S = S"
  unfolding permute_set_eq
  using a by (auto simp add: swap_atom)

lemma swap_set_in:
  assumes a: "a \<in> S" "b \<notin> S" "sort_of a = sort_of b"
  shows "(a \<rightleftharpoons> b) \<bullet> S \<noteq> S"
  unfolding permute_set_eq
  using a by (auto simp add: swap_atom)

subsection {* Permutations for units *}

instantiation unit :: pt
begin

definition "p \<bullet> (u::unit) = u"

instance 
by (default) (simp_all add: permute_unit_def)

end


subsection {* Permutations for products *}

instantiation "*" :: (pt, pt) pt
begin

primrec 
  permute_prod 
where
  Pair_eqvt: "p \<bullet> (x, y) = (p \<bullet> x, p \<bullet> y)"

instance
by default auto

end

subsection {* Permutations for sums *}

instantiation "+" :: (pt, pt) pt
begin

primrec 
  permute_sum 
where
  "p \<bullet> (Inl x) = Inl (p \<bullet> x)"
| "p \<bullet> (Inr y) = Inr (p \<bullet> y)"

instance 
by (default) (case_tac [!] x, simp_all)

end

subsection {* Permutations for lists *}

instantiation list :: (pt) pt
begin

primrec 
  permute_list 
where
  "p \<bullet> [] = []"
| "p \<bullet> (x # xs) = p \<bullet> x # p \<bullet> xs"

instance 
by (default) (induct_tac [!] x, simp_all)

end

subsection {* Permutations for options *}

instantiation option :: (pt) pt
begin

primrec 
  permute_option 
where
  "p \<bullet> None = None"
| "p \<bullet> (Some x) = Some (p \<bullet> x)"

instance 
by (default) (induct_tac [!] x, simp_all)

end

subsection {* Permutations for @{typ char}, @{typ nat}, and @{typ int} *}

instantiation char :: pt
begin

definition "p \<bullet> (c::char) = c"

instance 
by (default) (simp_all add: permute_char_def)

end

instantiation nat :: pt
begin

definition "p \<bullet> (n::nat) = n"

instance 
by (default) (simp_all add: permute_nat_def)

end

instantiation int :: pt
begin

definition "p \<bullet> (i::int) = i"

instance 
by (default) (simp_all add: permute_int_def)

end


section {* Pure types *}

text {* Pure types will have always empty support. *}

class pure = pt +
  assumes permute_pure: "p \<bullet> x = x"

text {* Types @{typ unit} and @{typ bool} are pure. *}

instance unit :: pure
proof qed (rule permute_unit_def)

instance bool :: pure
proof qed (rule permute_bool_def)

text {* Other type constructors preserve purity. *}

instance "fun" :: (pure, pure) pure
by default (simp add: permute_fun_def permute_pure)

instance "*" :: (pure, pure) pure
by default (induct_tac x, simp add: permute_pure)

instance "+" :: (pure, pure) pure
by default (induct_tac x, simp_all add: permute_pure)

instance list :: (pure) pure
by default (induct_tac x, simp_all add: permute_pure)

instance option :: (pure) pure
by default (induct_tac x, simp_all add: permute_pure)


subsection {* Types @{typ char}, @{typ nat}, and @{typ int} *}

instance char :: pure
proof qed (rule permute_char_def)

instance nat :: pure
proof qed (rule permute_nat_def)

instance int :: pure
proof qed (rule permute_int_def)


subsection {* Supp, Freshness and Supports *}

context pt
begin

definition
  supp :: "'a \<Rightarrow> atom set"
where
  "supp x = {a. infinite {b. (a \<rightleftharpoons> b) \<bullet> x \<noteq> x}}"

end

definition
  fresh :: "atom \<Rightarrow> 'a::pt \<Rightarrow> bool" ("_ \<sharp> _" [55, 55] 55)
where   
  "a \<sharp> x \<equiv> a \<notin> supp x"

lemma supp_conv_fresh: 
  shows "supp x = {a. \<not> a \<sharp> x}"
  unfolding fresh_def by simp

lemma swap_rel_trans:
  assumes "sort_of a = sort_of b"
  assumes "sort_of b = sort_of c"
  assumes "(a \<rightleftharpoons> c) \<bullet> x = x"
  assumes "(b \<rightleftharpoons> c) \<bullet> x = x"
  shows "(a \<rightleftharpoons> b) \<bullet> x = x"
proof (cases)
  assume "a = b \<or> c = b"
  with assms show "(a \<rightleftharpoons> b) \<bullet> x = x" by auto
next
  assume *: "\<not> (a = b \<or> c = b)"
  have "((a \<rightleftharpoons> c) + (b \<rightleftharpoons> c) + (a \<rightleftharpoons> c)) \<bullet> x = x"
    using assms by simp
  also have "(a \<rightleftharpoons> c) + (b \<rightleftharpoons> c) + (a \<rightleftharpoons> c) = (a \<rightleftharpoons> b)"
    using assms * by (simp add: swap_triple)
  finally show "(a \<rightleftharpoons> b) \<bullet> x = x" .
qed

lemma swap_fresh_fresh:
  assumes a: "a \<sharp> x" 
  and     b: "b \<sharp> x"
  shows "(a \<rightleftharpoons> b) \<bullet> x = x"
proof (cases)
  assume asm: "sort_of a = sort_of b" 
  have "finite {c. (a \<rightleftharpoons> c) \<bullet> x \<noteq> x}" "finite {c. (b \<rightleftharpoons> c) \<bullet> x \<noteq> x}" 
    using a b unfolding fresh_def supp_def by simp_all
  then have "finite ({c. (a \<rightleftharpoons> c) \<bullet> x \<noteq> x} \<union> {c. (b \<rightleftharpoons> c) \<bullet> x \<noteq> x})" by simp
  then obtain c 
    where "(a \<rightleftharpoons> c) \<bullet> x = x" "(b \<rightleftharpoons> c) \<bullet> x = x" "sort_of c = sort_of b"
    by (rule obtain_atom) (auto)
  then show "(a \<rightleftharpoons> b) \<bullet> x = x" using asm by (rule_tac swap_rel_trans) (simp_all)
next
  assume "sort_of a \<noteq> sort_of b"
  then show "(a \<rightleftharpoons> b) \<bullet> x = x" by simp
qed


subsection {* supp and fresh are equivariant *}

lemma finite_Collect_bij:
  assumes a: "bij f"
  shows "finite {x. P (f x)} = finite {x. P x}"
by (metis a finite_vimage_iff vimage_Collect_eq)

lemma fresh_permute_iff:
  shows "(p \<bullet> a) \<sharp> (p \<bullet> x) \<longleftrightarrow> a \<sharp> x"
proof -
  have "(p \<bullet> a) \<sharp> (p \<bullet> x) \<longleftrightarrow> finite {b. (p \<bullet> a \<rightleftharpoons> b) \<bullet> p \<bullet> x \<noteq> p \<bullet> x}"
    unfolding fresh_def supp_def by simp
  also have "\<dots> \<longleftrightarrow> finite {b. (p \<bullet> a \<rightleftharpoons> p \<bullet> b) \<bullet> p \<bullet> x \<noteq> p \<bullet> x}"
    using bij_permute by (rule finite_Collect_bij[symmetric])
  also have "\<dots> \<longleftrightarrow> finite {b. p \<bullet> (a \<rightleftharpoons> b) \<bullet> x \<noteq> p \<bullet> x}"
    by (simp only: permute_eqvt [of p] swap_eqvt)
  also have "\<dots> \<longleftrightarrow> finite {b. (a \<rightleftharpoons> b) \<bullet> x \<noteq> x}"
    by (simp only: permute_eq_iff)
  also have "\<dots> \<longleftrightarrow> a \<sharp> x"
    unfolding fresh_def supp_def by simp
  finally show "(p \<bullet> a) \<sharp> (p \<bullet> x) \<longleftrightarrow> a \<sharp> x" .
qed

lemma fresh_eqvt:
  shows "p \<bullet> (a \<sharp> x) = (p \<bullet> a) \<sharp> (p \<bullet> x)"
  unfolding permute_bool_def
  by (simp add: fresh_permute_iff)

lemma supp_eqvt:
  fixes  p  :: "perm"
  and    x   :: "'a::pt"
  shows "p \<bullet> (supp x) = supp (p \<bullet> x)"
  unfolding supp_conv_fresh
  unfolding Collect_def
  unfolding permute_fun_def
  by (simp add: Not_eqvt fresh_eqvt)

subsection {* supports *}

definition
  supports :: "atom set \<Rightarrow> 'a::pt \<Rightarrow> bool" (infixl "supports" 80)
where  
  "S supports x \<equiv> \<forall>a b. (a \<notin> S \<and> b \<notin> S \<longrightarrow> (a \<rightleftharpoons> b) \<bullet> x = x)"

lemma supp_is_subset:
  fixes S :: "atom set"
  and   x :: "'a::pt"
  assumes a1: "S supports x"
  and     a2: "finite S"
  shows "(supp x) \<subseteq> S"
proof (rule ccontr)
  assume "\<not> (supp x \<subseteq> S)"
  then obtain a where b1: "a \<in> supp x" and b2: "a \<notin> S" by auto
  from a1 b2 have "\<forall>b. b \<notin> S \<longrightarrow> (a \<rightleftharpoons> b) \<bullet> x = x" unfolding supports_def by auto
  then have "{b. (a \<rightleftharpoons> b) \<bullet> x \<noteq> x} \<subseteq> S" by auto
  with a2 have "finite {b. (a \<rightleftharpoons> b)\<bullet>x \<noteq> x}" by (simp add: finite_subset)
  then have "a \<notin> (supp x)" unfolding supp_def by simp
  with b1 show False by simp
qed

lemma supports_finite:
  fixes S :: "atom set"
  and   x :: "'a::pt"
  assumes a1: "S supports x"
  and     a2: "finite S"
  shows "finite (supp x)"
proof -
  have "(supp x) \<subseteq> S" using a1 a2 by (rule supp_is_subset)
  then show "finite (supp x)" using a2 by (simp add: finite_subset)
qed

lemma supp_supports:
  fixes x :: "'a::pt"
  shows "(supp x) supports x"
unfolding supports_def
proof (intro strip)
  fix a b
  assume "a \<notin> (supp x) \<and> b \<notin> (supp x)"
  then have "a \<sharp> x" and "b \<sharp> x" by (simp_all add: fresh_def)
  then show "(a \<rightleftharpoons> b) \<bullet> x = x" by (simp add: swap_fresh_fresh)
qed

lemma supp_is_least_supports:
  fixes S :: "atom set"
  and   x :: "'a::pt"
  assumes  a1: "S supports x"
  and      a2: "finite S"
  and      a3: "\<And>S'. finite S' \<Longrightarrow> (S' supports x) \<Longrightarrow> S \<subseteq> S'"
  shows "(supp x) = S"
proof (rule equalityI)
  show "(supp x) \<subseteq> S" using a1 a2 by (rule supp_is_subset)
  with a2 have fin: "finite (supp x)" by (rule rev_finite_subset)
  have "(supp x) supports x" by (rule supp_supports)
  with fin a3 show "S \<subseteq> supp x" by blast
qed

lemma subsetCI: 
  shows "(\<And>x. x \<in> A \<Longrightarrow> x \<notin> B \<Longrightarrow> False) \<Longrightarrow> A \<subseteq> B"
  by auto

lemma finite_supp_unique:
  assumes a1: "S supports x"
  assumes a2: "finite S"
  assumes a3: "\<And>a b. \<lbrakk>a \<in> S; b \<notin> S; sort_of a = sort_of b\<rbrakk> \<Longrightarrow> (a \<rightleftharpoons> b) \<bullet> x \<noteq> x"
  shows "(supp x) = S"
  using a1 a2
proof (rule supp_is_least_supports)
  fix S'
  assume "finite S'" and "S' supports x"
  show "S \<subseteq> S'"
  proof (rule subsetCI)
    fix a
    assume "a \<in> S" and "a \<notin> S'"
    have "finite (S \<union> S')"
      using `finite S` `finite S'` by simp
    then obtain b where "b \<notin> S \<union> S'" and "sort_of b = sort_of a"
      by (rule obtain_atom)
    then have "b \<notin> S" and "b \<notin> S'"  and "sort_of a = sort_of b"
      by simp_all
    then have "(a \<rightleftharpoons> b) \<bullet> x = x"
      using `a \<notin> S'` `S' supports x` by (simp add: supports_def)
    moreover have "(a \<rightleftharpoons> b) \<bullet> x \<noteq> x"
      using `a \<in> S` `b \<notin> S` `sort_of a = sort_of b`
      by (rule a3)
    ultimately show "False" by simp
  qed
qed

section {* Finitely-supported types *}

class fs = pt +
  assumes finite_supp: "finite (supp x)"

lemma pure_supp: 
  shows "supp (x::'a::pure) = {}"
  unfolding supp_def by (simp add: permute_pure)

lemma pure_fresh:
  fixes x::"'a::pure"
  shows "a \<sharp> x"
  unfolding fresh_def by (simp add: pure_supp)

instance pure < fs
by default (simp add: pure_supp)


subsection  {* Type @{typ atom} is finitely-supported. *}

lemma supp_atom:
  shows "supp a = {a}"
apply (rule finite_supp_unique)
apply (clarsimp simp add: supports_def)
apply simp
apply simp
done

lemma fresh_atom: 
  shows "a \<sharp> b \<longleftrightarrow> a \<noteq> b"
  unfolding fresh_def supp_atom by simp

instance atom :: fs
by default (simp add: supp_atom)

section {* Support for finite sets of atoms *}

lemma supp_finite_atom_set:
  fixes S::"atom set"
  assumes "finite S"
  shows "supp S = S"
  apply(rule finite_supp_unique)
  apply(simp add: supports_def)
  apply(simp add: swap_set_not_in)
  apply(rule assms)
  apply(simp add: swap_set_in)
done

lemma supp_atom_insert:
  fixes S::"atom set"
  assumes a: "finite S"
  shows "supp (insert a S) = supp a \<union> supp S"
  using a by (simp add: supp_finite_atom_set supp_atom)

section {* Type @{typ perm} is finitely-supported. *}

lemma perm_swap_eq:
  shows "(a \<rightleftharpoons> b) \<bullet> p = p \<longleftrightarrow> (p \<bullet> (a \<rightleftharpoons> b)) = (a \<rightleftharpoons> b)"
unfolding permute_perm_def
by (metis add_diff_cancel minus_perm_def)

lemma supports_perm: 
  shows "{a. p \<bullet> a \<noteq> a} supports p"
  unfolding supports_def
  unfolding perm_swap_eq
  by (simp add: swap_eqvt)

lemma finite_perm_lemma: 
  shows "finite {a::atom. p \<bullet> a \<noteq> a}"
  using finite_Rep_perm [of p]
  unfolding permute_atom_def .

lemma supp_perm:
  shows "supp p = {a. p \<bullet> a \<noteq> a}"
apply (rule finite_supp_unique)
apply (rule supports_perm)
apply (rule finite_perm_lemma)
apply (simp add: perm_swap_eq swap_eqvt)
apply (auto simp add: expand_perm_eq swap_atom)
done

lemma fresh_perm: 
  shows "a \<sharp> p \<longleftrightarrow> p \<bullet> a = a"
  unfolding fresh_def 
  by (simp add: supp_perm)

lemma supp_swap:
  shows "supp (a \<rightleftharpoons> b) = (if a = b \<or> sort_of a \<noteq> sort_of b then {} else {a, b})"
  by (auto simp add: supp_perm swap_atom)

lemma fresh_zero_perm: 
  shows "a \<sharp> (0::perm)"
  unfolding fresh_perm by simp

lemma supp_zero_perm: 
  shows "supp (0::perm) = {}"
  unfolding supp_perm by simp

lemma fresh_plus_perm:
  fixes p q::perm
  assumes "a \<sharp> p" "a \<sharp> q"
  shows "a \<sharp> (p + q)"
  using assms
  unfolding fresh_def
  by (auto simp add: supp_perm)

lemma supp_plus_perm:
  fixes p q::perm
  shows "supp (p + q) \<subseteq> supp p \<union> supp q"
  by (auto simp add: supp_perm)

lemma fresh_minus_perm:
  fixes p::perm
  shows "a \<sharp> (- p) \<longleftrightarrow> a \<sharp> p"
  unfolding fresh_def
  unfolding supp_perm
  apply(simp)
  apply(metis permute_minus_cancel)
  done

lemma supp_minus_perm:
  fixes p::perm
  shows "supp (- p) = supp p"
  unfolding supp_conv_fresh
  by (simp add: fresh_minus_perm)

instance perm :: fs
by default (simp add: supp_perm finite_perm_lemma)

lemma plus_perm_eq:
  fixes p q::"perm"
  assumes asm: "supp p \<inter> supp q = {}"
  shows "p + q  = q + p"
unfolding expand_perm_eq
proof
  fix a::"atom"
  show "(p + q) \<bullet> a = (q + p) \<bullet> a"
  proof -
    { assume "a \<notin> supp p" "a \<notin> supp q"
      then have "(p + q) \<bullet> a = (q + p) \<bullet> a" 
	by (simp add: supp_perm)
    }
    moreover
    { assume a: "a \<in> supp p" "a \<notin> supp q"
      then have "p \<bullet> a \<in> supp p" by (simp add: supp_perm)
      then have "p \<bullet> a \<notin> supp q" using asm by auto
      with a have "(p + q) \<bullet> a = (q + p) \<bullet> a" 
	by (simp add: supp_perm)
    }
    moreover
    { assume a: "a \<notin> supp p" "a \<in> supp q"
      then have "q \<bullet> a \<in> supp q" by (simp add: supp_perm)
      then have "q \<bullet> a \<notin> supp p" using asm by auto 
      with a have "(p + q) \<bullet> a = (q + p) \<bullet> a" 
	by (simp add: supp_perm)
    }
    ultimately show "(p + q) \<bullet> a = (q + p) \<bullet> a" 
      using asm by blast
  qed
qed

section {* Finite Support instances for other types *}

subsection {* Type @{typ "'a \<times> 'b"} is finitely-supported. *}

lemma supp_Pair: 
  shows "supp (x, y) = supp x \<union> supp y"
  by (simp add: supp_def Collect_imp_eq Collect_neg_eq)

lemma fresh_Pair: 
  shows "a \<sharp> (x, y) \<longleftrightarrow> a \<sharp> x \<and> a \<sharp> y"
  by (simp add: fresh_def supp_Pair)

instance "*" :: (fs, fs) fs
apply default
apply (induct_tac x)
apply (simp add: supp_Pair finite_supp)
done

subsection {* Type @{typ "'a + 'b"} is finitely supported *}

lemma supp_Inl: 
  shows "supp (Inl x) = supp x"
  by (simp add: supp_def)

lemma supp_Inr: 
  shows "supp (Inr x) = supp x"
  by (simp add: supp_def)

lemma fresh_Inl: 
  shows "a \<sharp> Inl x \<longleftrightarrow> a \<sharp> x"
  by (simp add: fresh_def supp_Inl)

lemma fresh_Inr: 
  shows "a \<sharp> Inr y \<longleftrightarrow> a \<sharp> y"
  by (simp add: fresh_def supp_Inr)

instance "+" :: (fs, fs) fs
apply default
apply (induct_tac x)
apply (simp_all add: supp_Inl supp_Inr finite_supp)
done

subsection {* Type @{typ "'a option"} is finitely supported *}

lemma supp_None: 
  shows "supp None = {}"
by (simp add: supp_def)

lemma supp_Some: 
  shows "supp (Some x) = supp x"
  by (simp add: supp_def)

lemma fresh_None: 
  shows "a \<sharp> None"
  by (simp add: fresh_def supp_None)

lemma fresh_Some: 
  shows "a \<sharp> Some x \<longleftrightarrow> a \<sharp> x"
  by (simp add: fresh_def supp_Some)

instance option :: (fs) fs
apply default
apply (induct_tac x)
apply (simp_all add: supp_None supp_Some finite_supp)
done

subsubsection {* Type @{typ "'a list"} is finitely supported *}

lemma supp_Nil: 
  shows "supp [] = {}"
  by (simp add: supp_def)

lemma supp_Cons: 
  shows "supp (x # xs) = supp x \<union> supp xs"
by (simp add: supp_def Collect_imp_eq Collect_neg_eq)

lemma fresh_Nil: 
  shows "a \<sharp> []"
  by (simp add: fresh_def supp_Nil)

lemma fresh_Cons: 
  shows "a \<sharp> (x # xs) \<longleftrightarrow> a \<sharp> x \<and> a \<sharp> xs"
  by (simp add: fresh_def supp_Cons)

instance list :: (fs) fs
apply default
apply (induct_tac x)
apply (simp_all add: supp_Nil supp_Cons finite_supp)
done

section {* Support and freshness for applications *}

lemma fresh_conv_MOST: 
  shows "a \<sharp> x \<longleftrightarrow> (MOST b. (a \<rightleftharpoons> b) \<bullet> x = x)"
  unfolding fresh_def supp_def 
  unfolding MOST_iff_cofinite by simp

lemma supp_subset_fresh:
  assumes a: "\<And>a. a \<sharp> x \<Longrightarrow> a \<sharp> y"
  shows "supp y \<subseteq> supp x"
  using a
  unfolding fresh_def
  by blast

lemma fresh_fun_app:
  assumes "a \<sharp> f" and "a \<sharp> x" 
  shows "a \<sharp> f x"
  using assms
  unfolding fresh_conv_MOST
  unfolding permute_fun_app_eq 
  by (elim MOST_rev_mp, simp)

lemma supp_fun_app:
  shows "supp (f x) \<subseteq> (supp f) \<union> (supp x)"
  using fresh_fun_app
  unfolding fresh_def
  by auto

text {* support of equivariant functions *}

lemma supp_fun_eqvt:
  assumes a: "\<And>p. p \<bullet> f = f"
  shows "supp f = {}"
  unfolding supp_def 
  using a by simp

lemma fresh_fun_eqvt_app:
  assumes a: "\<And>p. p \<bullet> f = f"
  shows "a \<sharp> x \<Longrightarrow> a \<sharp> f x"
proof -
  from a have "supp f = {}" by (simp add: supp_fun_eqvt)
  then show "a \<sharp> x \<Longrightarrow> a \<sharp> f x"
    unfolding fresh_def
    using supp_fun_app by auto
qed


section {* Concrete atoms types *}

text {*
  Class @{text at_base} allows types containing multiple sorts of atoms.
  Class @{text at} only allows types with a single sort.
*}

class at_base = pt +
  fixes atom :: "'a \<Rightarrow> atom"
  assumes atom_eq_iff [simp]: "atom a = atom b \<longleftrightarrow> a = b"
  assumes atom_eqvt: "p \<bullet> (atom a) = atom (p \<bullet> a)"

class at = at_base +
  assumes sort_of_atom_eq [simp]: "sort_of (atom a) = sort_of (atom b)"

lemma supp_at_base: 
  fixes a::"'a::at_base"
  shows "supp a = {atom a}"
  by (simp add: supp_atom [symmetric] supp_def atom_eqvt)

lemma fresh_at_base: 
  shows "a \<sharp> b \<longleftrightarrow> a \<noteq> atom b"
  unfolding fresh_def by (simp add: supp_at_base)

instance at_base < fs
proof qed (simp add: supp_at_base)

lemma at_base_infinite [simp]:
  shows "infinite (UNIV :: 'a::at_base set)" (is "infinite ?U")
proof
  obtain a :: 'a where "True" by auto
  assume "finite ?U"
  hence "finite (atom ` ?U)"
    by (rule finite_imageI)
  then obtain b where b: "b \<notin> atom ` ?U" "sort_of b = sort_of (atom a)"
    by (rule obtain_atom)
  from b(2) have "b = atom ((atom a \<rightleftharpoons> b) \<bullet> a)"
    unfolding atom_eqvt [symmetric]
    by (simp add: swap_atom)
  hence "b \<in> atom ` ?U" by simp
  with b(1) show "False" by simp
qed

lemma swap_at_base_simps [simp]:
  fixes x y::"'a::at_base"
  shows "sort_of (atom x) = sort_of (atom y) \<Longrightarrow> (atom x \<rightleftharpoons> atom y) \<bullet> x = y"
  and   "sort_of (atom x) = sort_of (atom y) \<Longrightarrow> (atom x \<rightleftharpoons> atom y) \<bullet> y = x"
  and   "atom x \<noteq> a \<Longrightarrow> atom x \<noteq> b \<Longrightarrow> (a \<rightleftharpoons> b) \<bullet> x = x"
  unfolding atom_eq_iff [symmetric]
  unfolding atom_eqvt [symmetric]
  by simp_all

lemma obtain_at_base:
  assumes X: "finite X"
  obtains a::"'a::at_base" where "atom a \<notin> X"
proof -
  have "inj (atom :: 'a \<Rightarrow> atom)"
    by (simp add: inj_on_def)
  with X have "finite (atom -` X :: 'a set)"
    by (rule finite_vimageI)
  with at_base_infinite have "atom -` X \<noteq> (UNIV :: 'a set)"
    by auto
  then obtain a :: 'a where "atom a \<notin> X"
    by auto
  thus ?thesis ..
qed

lemma image_eqvt:
  shows "p \<bullet> (f ` A) = (p \<bullet> f) ` (p \<bullet> A)"
  unfolding permute_set_eq_image
  unfolding permute_fun_def [where f=f]
  by (simp add: image_image)

lemma atom_image_cong:
  shows "(atom ` X = atom ` Y) = (X = Y)"
  apply(rule inj_image_eq_iff)
  apply(simp add: inj_on_def)
  done

lemma atom_image_supp:
  shows "supp S = supp (atom ` S)"
  apply(simp add: supp_def)
  apply(simp add: image_eqvt)
  apply(subst (2) permute_fun_def)
  apply(simp add: atom_eqvt)
  apply(simp add: atom_image_cong)
  done

lemma supp_finite_set_at_base:
  assumes a: "finite S"
  shows "supp S = atom ` S"
proof -
  have fin: "finite (atom ` S)" using a by simp
  have "supp S = supp (atom ` S)" by (rule atom_image_supp)
  also have "\<dots> = atom ` S" using fin by (simp add: supp_finite_atom_set)
  finally show "supp S = atom ` S" by simp
qed

lemma supp_at_base_insert:
  fixes a::"'a::at_base"
  assumes a: "finite S"
  shows "supp (insert a S) = supp a \<union> supp S"
  using a by (simp add: supp_finite_set_at_base supp_at_base)

section {* library functions for the nominal infrastructure *}
use "nominal_library.ML"

end
