theory Test
imports "Nominal2_Atoms" "Nominal2_Eqvt" "Nominal2_Supp"
begin

atom_decl name


(* tests *)
ML {*
Datatype.datatype_cmd;
Datatype.add_datatype Datatype.default_config;

Primrec.add_primrec_cmd;
Primrec.add_primrec;
Primrec.add_primrec_simple;
*}

section {* test for setting up a primrec on the ML-level *}

section{* Interface for nominal_datatype *}

text {*

Nominal-Datatype-part:

1st Arg: string list  
         ^^^^^^^^^^^
         strings of the types to be defined

2nd Arg: (string list * binding * mixfix * (binding * typ list * mixfix) list) list
         ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^   ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
               type(s) to be defined             constructors list
               (ty args, name, syn)              (name, typs, syn)

Binder-Function-part:

3rd Arg: (binding * typ option * mixfix) list 
         ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^    
            binding function(s)           
              to be defined               
            (name, type, syn)             

4th Arg:  term list 
          ^^^^^^^^^
          the equations of the binding functions
          (Trueprop equations)
*}

text {*****************************************************}
ML {* 
(* nominal datatype parser *)
local
  structure P = OuterParse
in

val _ = OuterKeyword.keyword "bind"
val anno_typ = Scan.option (P.name --| P.$$$ "::") -- P.typ

(* binding specification *)
(* should use and_list *)
val bind_parser = 
  P.enum "," ((P.$$$ "bind" |-- P.term) -- (P.$$$ "in" |-- P.name))

val constr_parser =
  P.binding -- Scan.repeat anno_typ

(* datatype parser *)
val dt_parser =
  ((P.type_args -- P.binding -- P.opt_infix) >> P.triple1) -- 
    (P.$$$ "=" |-- P.enum1 "|" ((constr_parser -- bind_parser -- P.opt_mixfix) >> P.triple_swap))

(* function equation parser *)
val fun_parser = 
  Scan.optional (P.$$$ "binder" |-- P.fixes -- SpecParse.where_alt_specs) ([],[])

(* main parser *)
val main_parser =
  (P.and_list1 dt_parser) -- fun_parser

end
*}

(* adds "_raw" to the end of constants and types *)
ML {*
fun add_raw s = s ^ "_raw"
fun raw_bind bn = Binding.suffix_name "_raw" bn

fun raw_str [] s = s
  | raw_str (s'::ss) s = 
      if (Long_Name.base_name s) = s' 
      then add_raw s
      else raw_str ss s

fun raw_typ ty_strs (Type (a, Ts)) = Type (raw_str ty_strs a, map (raw_typ ty_strs) Ts)
  | raw_typ ty_strs T = T  

fun raw_aterm trm_strs (Const (a, T)) = Const (raw_str trm_strs a, T)
  | raw_aterm trm_strs (Free (a, T)) = Free (raw_str trm_strs a, T)
  | raw_aterm trm_strs trm = trm
 
fun raw_term trm_strs ty_strs trm =
  trm |> Term.map_aterms (raw_aterm trm_strs) |> map_types (raw_typ ty_strs) 

fun raw_dts ty_strs dts =
let
  fun raw_dts_aux1 (bind, tys, mx) =
    (raw_bind bind, map (raw_typ ty_strs) tys, mx)

  fun raw_dts_aux2 (ty_args, bind, mx, constrs) =
    (ty_args, raw_bind bind, mx, map raw_dts_aux1 constrs)
in
  map raw_dts_aux2 dts
end

fun get_constr_strs dts =
  flat (map (fn (_, _, _, constrs) => map (fn (bn, _, _) => Binding.name_of bn) constrs) dts)

fun get_bn_fun_strs bn_funs =
  map (fn (bn_fun, _, _) => Binding.name_of bn_fun) bn_funs
*}

ML {*
fun raw_dts_decl dt_names dts lthy =
let
  val conf = Datatype.default_config

  val dt_names' = map add_raw dt_names
  val dts' = raw_dts dt_names dts
in
  Local_Theory.theory_result (Datatype.add_datatype conf dt_names' dts') lthy
end 
*}

ML {*
fun raw_bn_fun_decl dt_names cnstr_names bn_funs bn_eqs lthy =
let
  val bn_fun_strs = get_bn_fun_strs bn_funs
  
  val bn_funs' = map (fn (bn, opt_ty, mx) => 
    (raw_bind bn, Option.map (raw_typ dt_names) opt_ty, mx)) bn_funs
  val bn_eqs' = map (fn trm => 
    (Attrib.empty_binding, raw_term (cnstr_names @ bn_fun_strs) dt_names trm)) bn_eqs
in
  (*Primrec.add_primrec bn_funs' bn_eqs' lthy*)
  ((), lthy)
end 
*}

ML {* 
fun nominal_datatype2 dts bn_funs bn_eqs lthy =
let
  val dts_names = map (fn (_, s, _, _) => Binding.name_of s) dts
  val ctrs_names = get_constr_strs dts
in
  lthy
  |> raw_dts_decl dts_names dts 
  ||>> raw_bn_fun_decl dts_names ctrs_names bn_funs bn_eqs
end
*}

ML {*
fun nominal_datatype2_cmd (dt_strs, (bn_fun_strs, bn_eq_strs)) lthy =
let
  val lthy_tmp = lthy
    |> (Local_Theory.theory
        (Sign.add_types (map (fn ((tvs, tname, mx), _) =>
          (tname, length tvs, mx)) dt_strs)))

  fun prep_cnstr lthy (((cname, atys), mx), binders) =
    (cname, map (Syntax.read_typ lthy o snd) atys, mx)
  
  fun prep_dt lthy ((tvs, tname, mx), cnstrs) = 
    (tvs, tname, mx, map (prep_cnstr lthy) cnstrs)

  fun prep_bn_fun lthy (bn, ty_str, mx) =
    (bn, Option.map (Syntax.read_typ lthy) ty_str, mx) 

  fun prep_bn_eq lthy (attr, trm_str) = Syntax.read_term lthy trm_str

  val dts_prep = map (prep_dt lthy_tmp) dt_strs
  val bn_fun_prep = map (prep_bn_fun lthy_tmp) bn_fun_strs
  val bn_eq_prep = map (prep_bn_eq lthy_tmp) bn_eq_strs
in
  nominal_datatype2 dts_prep bn_fun_prep [] lthy |> snd
end
*}

(* Command Keyword *)
ML {*
let
   val kind = OuterKeyword.thy_decl
in
   OuterSyntax.local_theory "nominal_datatype" "test" kind 
     (main_parser >> nominal_datatype2_cmd)
end
*}

text {* example 1 *}
nominal_datatype lam =
  VAR "name"
| APP "lam" "lam"
| LET bp::"bp" t::"lam"   bind "bi bp" in t ("Let _ in _" [100,100] 100)
and bp = 
  BP "name" "lam"  ("_ ::= _" [100,100] 100)
binder
  bi::"bp \<Rightarrow> name set"
where
  "bi (BP x t) = {x}"

typ lam_raw
term VAR_raw
term BP_raw

print_theorems

text {* examples 2 *}
nominal_datatype trm =
  Var "name"
| App "trm" "trm"
| Lam x::"name" t::"trm"        bind x in t 
| Let p::"pat" "trm" t::"trm"   bind "f p" in t
and pat =
  PN
| PS "name"
| PD "name \<times> name"
binder
  f::"pat \<Rightarrow> name set"
where 
  "f PN = {}"
| "f (PS x) = {x}"
| "f (PD (x,y)) = {x,y}"

nominal_datatype trm2 =
  Var2 "name"
| App2 "trm2" "trm2"
| Lam2 x::"name" t::"trm2"          bind x in t 
| Let2 p::"pat2" "trm2" t::"trm2"   bind "f2 p" in t
and pat2 =
  PN2
| PS2 "name"
| PD2 "pat2 \<times> pat2"
binder
  f2::"pat2 \<Rightarrow> name set"
where 
  "f2 PN2 = {}"
| "f2 (PS2 x) = {x}"
| "f2 (PD2 (p1, p2)) = (f2 p1) \<union> (f2 p2)"

text {* example type schemes *}
datatype ty = 
  Var "name" 
| Fun "ty" "ty"

nominal_datatype tyS = 
  All xs::"name list" ty::"ty" bind xs in ty




end


(* printing stuff *)

ML {*
fun print_str_option NONE = "NONE"
  | print_str_option (SOME s) = (s:bstring)

fun print_anno_typ lthy (NONE, ty) = Syntax.string_of_typ lthy ty
  | print_anno_typ lthy (SOME x, ty) = x ^ ":" ^ Syntax.string_of_typ lthy ty

fun print_bind (s1, s2) = "bind " ^ s1 ^ " in " ^ s2

fun print_constr lthy (((name, anno_tys), bds), syn) =
  (Binding.name_of name ^ " of " ^ (commas (map (print_anno_typ lthy) anno_tys)) ^ "   "
   ^ (commas (map print_bind bds)) ^ "  "
   ^ (Pretty.str_of (Syntax.pretty_mixfix syn)))

fun print_single_dt lthy (((s2, s3), syn), constrs) =
   ["constructor declaration"]
 @ ["type arguments: "] @ s2 
 @ ["datatype name: ", Binding.name_of s3] 
 @ ["typ mixfix: ", Pretty.string_of (Syntax.pretty_mixfix syn)]
 @ ["constructors: "] @ (map (print_constr lthy) constrs)
 |> separate "\n"
 |> implode

fun print_single_fun_eq lthy (at, s) = 
  ["function equations: ", (Syntax.string_of_term lthy s)] 
  |> separate "\n"
  |> implode

fun print_single_fun_fix lthy (b, _, _) = 
  ["functions: ", Binding.name_of b] 
  |> separate "\n"
  |> implode

fun dt_name (((s2, s3), syn), constrs) = Binding.name_of s3

fun print_dts (dts, (funs, feqs)) lthy =
let
  val _ = warning (implode (separate "\n" (map (print_single_dt lthy) dts)));
  val _ = warning (implode (separate "\n" (map (print_single_fun_fix lthy) funs)));
  val _ = warning (implode (separate "\n" (map (print_single_fun_eq lthy) feqs)));
in
  ()
end
*}

