   
theory BitCoded
  imports "Lexer" 
begin

section {* Bit-Encodings *}

datatype bit = Z | S

fun 
  code :: "val \<Rightarrow> bit list"
where
  "code Void = []"
| "code (Char c) = []"
| "code (Left v) = Z # (code v)"
| "code (Right v) = S # (code v)"
| "code (Seq v1 v2) = (code v1) @ (code v2)"
| "code (Stars []) = [S]"
| "code (Stars (v # vs)) =  (Z # code v) @ code (Stars vs)"


fun 
  Stars_add :: "val \<Rightarrow> val \<Rightarrow> val"
where
  "Stars_add v (Stars vs) = Stars (v # vs)"

function
  decode' :: "bit list \<Rightarrow> rexp \<Rightarrow> (val * bit list)"
where
  "decode' ds ZERO = (Void, [])"
| "decode' ds ONE = (Void, ds)"
| "decode' ds (CHAR d) = (Char d, ds)"
| "decode' [] (ALT r1 r2) = (Void, [])"
| "decode' (Z # ds) (ALT r1 r2) = (let (v, ds') = decode' ds r1 in (Left v, ds'))"
| "decode' (S # ds) (ALT r1 r2) = (let (v, ds') = decode' ds r2 in (Right v, ds'))"
| "decode' ds (SEQ r1 r2) = (let (v1, ds') = decode' ds r1 in
                             let (v2, ds'') = decode' ds' r2 in (Seq v1 v2, ds''))"
| "decode' [] (STAR r) = (Void, [])"
| "decode' (S # ds) (STAR r) = (Stars [], ds)"
| "decode' (Z # ds) (STAR r) = (let (v, ds') = decode' ds r in
                                    let (vs, ds'') = decode' ds' (STAR r) 
                                    in (Stars_add v vs, ds''))"
by pat_completeness auto

lemma decode'_smaller:
  assumes "decode'_dom (ds, r)"
  shows "length (snd (decode' ds r)) \<le> length ds"
using assms
apply(induct ds r)
apply(auto simp add: decode'.psimps split: prod.split)
using dual_order.trans apply blast
by (meson dual_order.trans le_SucI)

termination "decode'"  
apply(relation "inv_image (measure(%cs. size cs) <*lex*> measure(%s. size s)) (%(ds,r). (r,ds))") 
apply(auto dest!: decode'_smaller)
by (metis less_Suc_eq_le snd_conv)

definition
  decode :: "bit list \<Rightarrow> rexp \<Rightarrow> val option"
where
  "decode ds r \<equiv> (let (v, ds') = decode' ds r 
                  in (if ds' = [] then Some v else None))"

lemma decode'_code_Stars:
  assumes "\<forall>v\<in>set vs. \<Turnstile> v : r \<and> (\<forall>x. decode' (code v @ x) r = (v, x)) \<and> flat v \<noteq> []" 
  shows "decode' (code (Stars vs) @ ds) (STAR r) = (Stars vs, ds)"
  using assms
  apply(induct vs)
  apply(auto)
  done

lemma decode'_code:
  assumes "\<Turnstile> v : r"
  shows "decode' ((code v) @ ds) r = (v, ds)"
using assms
  apply(induct v r arbitrary: ds) 
  apply(auto)
  using decode'_code_Stars by blast

lemma decode_code:
  assumes "\<Turnstile> v : r"
  shows "decode (code v) r = Some v"
  using assms unfolding decode_def
  by (smt append_Nil2 decode'_code old.prod.case)


section {* Annotated Regular Expressions *}

datatype arexp = 
  AZERO
| AONE "bit list"
| ACHAR "bit list" char
| ASEQ "bit list" arexp arexp
| AALTs "bit list" "arexp list"
| ASTAR "bit list" arexp

abbreviation
  "AALT bs r1 r2 \<equiv> AALTs bs [r1, r2]"

fun asize :: "arexp \<Rightarrow> nat" where
  "asize AZERO = 1"
| "asize (AONE cs) = 1" 
| "asize (ACHAR cs c) = 1"
| "asize (AALTs cs rs) = Suc (sum_list (map asize rs))"
| "asize (ASEQ cs r1 r2) = Suc (asize r1 + asize r2)"
| "asize (ASTAR cs r) = Suc (asize r)"



fun fuse :: "bit list \<Rightarrow> arexp \<Rightarrow> arexp" where
  "fuse bs AZERO = AZERO"
| "fuse bs (AONE cs) = AONE (bs @ cs)" 
| "fuse bs (ACHAR cs c) = ACHAR (bs @ cs) c"
| "fuse bs (AALTs cs rs) = AALTs (bs @ cs) rs"
| "fuse bs (ASEQ cs r1 r2) = ASEQ (bs @ cs) r1 r2"
| "fuse bs (ASTAR cs r) = ASTAR (bs @ cs) r"

lemma fuse_append:
  shows "fuse (bs1 @ bs2) r = fuse bs1 (fuse bs2 r)"
  apply(induct r)
  apply(auto)
  done


fun intern :: "rexp \<Rightarrow> arexp" where
  "intern ZERO = AZERO"
| "intern ONE = AONE []"
| "intern (CHAR c) = ACHAR [] c"
| "intern (ALT r1 r2) = AALT [] (fuse [Z] (intern r1)) 
                                (fuse [S]  (intern r2))"
| "intern (SEQ r1 r2) = ASEQ [] (intern r1) (intern r2)"
| "intern (STAR r) = ASTAR [] (intern r)"


fun retrieve :: "arexp \<Rightarrow> val \<Rightarrow> bit list" where
  "retrieve (AONE bs) Void = bs"
| "retrieve (ACHAR bs c) (Char d) = bs"
| "retrieve (AALTs bs [r]) v = bs @ retrieve r v" 
| "retrieve (AALTs bs (r#rs)) (Left v) = bs @ retrieve r v"
| "retrieve (AALTs bs (r#rs)) (Right v) = bs @ retrieve (AALTs [] rs) v"
| "retrieve (ASEQ bs r1 r2) (Seq v1 v2) = bs @ retrieve r1 v1 @ retrieve r2 v2"
| "retrieve (ASTAR bs r) (Stars []) = bs @ [S]"
| "retrieve (ASTAR bs r) (Stars (v#vs)) = 
     bs @ [Z] @ retrieve r v @ retrieve (ASTAR [] r) (Stars vs)"

fun 
  erase :: "arexp \<Rightarrow> rexp"
where
  "erase AZERO = ZERO"
| "erase (AONE _) = ONE"
| "erase (ACHAR _ c) = CHAR c"
| "erase (AALTs _ []) = ZERO"
| "erase (AALTs _ [r]) = (erase r)"
| "erase (AALTs bs (r#rs)) = ALT (erase r) (erase (AALTs bs rs))"
| "erase (ASEQ _ r1 r2) = SEQ (erase r1) (erase r2)"
| "erase (ASTAR _ r) = STAR (erase r)"

fun
 bnullable :: "arexp \<Rightarrow> bool"
where
  "bnullable (AZERO) = False"
| "bnullable (AONE bs) = True"
| "bnullable (ACHAR bs c) = False"
| "bnullable (AALTs bs rs) = (\<exists>r \<in> set rs. bnullable r)"
| "bnullable (ASEQ bs r1 r2) = (bnullable r1 \<and> bnullable r2)"
| "bnullable (ASTAR bs r) = True"

fun 
  bmkeps :: "arexp \<Rightarrow> bit list"
where
  "bmkeps(AONE bs) = bs"
| "bmkeps(ASEQ bs r1 r2) = bs @ (bmkeps r1) @ (bmkeps r2)"
| "bmkeps(AALTs bs [r]) = bs @ (bmkeps r)"
| "bmkeps(AALTs bs (r#rs)) = (if bnullable(r) then bs @ (bmkeps r) else (bmkeps (AALTs bs rs)))"
| "bmkeps(ASTAR bs r) = bs @ [S]"


fun
 bder :: "char \<Rightarrow> arexp \<Rightarrow> arexp"
where
  "bder c (AZERO) = AZERO"
| "bder c (AONE bs) = AZERO"
| "bder c (ACHAR bs d) = (if c = d then AONE bs else AZERO)"
| "bder c (AALTs bs rs) = AALTs bs (map (bder c) rs)"
| "bder c (ASEQ bs r1 r2) = 
     (if bnullable r1
      then AALT bs (ASEQ [] (bder c r1) r2) (fuse (bmkeps r1) (bder c r2))
      else ASEQ bs (bder c r1) r2)"
| "bder c (ASTAR bs r) = ASEQ bs (fuse [Z] (bder c r)) (ASTAR [] r)"


fun 
  bders :: "arexp \<Rightarrow> string \<Rightarrow> arexp"
where
  "bders r [] = r"
| "bders r (c#s) = bders (bder c r) s"

lemma bders_append:
  "bders r (s1 @ s2) = bders (bders r s1) s2"
  apply(induct s1 arbitrary: r s2)
  apply(simp_all)
  done

lemma bnullable_correctness:
  shows "nullable (erase r) = bnullable r"
  apply(induct r rule: erase.induct)
  apply(simp_all)
  done

lemma erase_fuse:
  shows "erase (fuse bs r) = erase r"
  apply(induct r rule: erase.induct)
  apply(simp_all)
  done

lemma erase_intern [simp]:
  shows "erase (intern r) = r"
  apply(induct r)
  apply(simp_all add: erase_fuse)
  done

lemma erase_bder [simp]:
  shows "erase (bder a r) = der a (erase r)"
  apply(induct r rule: erase.induct)
  apply(simp_all add: erase_fuse bnullable_correctness)
  done

lemma erase_bders [simp]:
  shows "erase (bders r s) = ders s (erase r)"
  apply(induct s arbitrary: r )
  apply(simp_all)
  done

lemma retrieve_encode_STARS:
  assumes "\<forall>v\<in>set vs. \<Turnstile> v : r \<and> code v = retrieve (intern r) v"
  shows "code (Stars vs) = retrieve (ASTAR [] (intern r)) (Stars vs)"
  using assms
  apply(induct vs)
  apply(simp_all)
  done

lemma retrieve_fuse2:
  assumes "\<Turnstile> v : (erase r)"
  shows "retrieve (fuse bs r) v = bs @ retrieve r v"
  using assms
  apply(induct r arbitrary: v bs)
         apply(auto elim: Prf_elims)[4]
   defer
  using retrieve_encode_STARS
   apply(auto elim!: Prf_elims)[1]
   apply(case_tac vs)
    apply(simp)
   apply(simp)
  (* AALTs  case *)
  apply(simp)
  apply(case_tac x2a)
   apply(simp)
   apply(auto elim!: Prf_elims)[1]
  apply(simp)
   apply(case_tac list)
   apply(simp)
  apply(auto)
  apply(auto elim!: Prf_elims)[1]
  done

lemma retrieve_fuse:
  assumes "\<Turnstile> v : r"
  shows "retrieve (fuse bs (intern r)) v = bs @ retrieve (intern r) v"
  using assms 
  by (simp_all add: retrieve_fuse2)


lemma retrieve_code:
  assumes "\<Turnstile> v : r"
  shows "code v = retrieve (intern r) v"
  using assms
  apply(induct v r )
  apply(simp_all add: retrieve_fuse retrieve_encode_STARS)
  done

lemma r:
  assumes "bnullable (AALTs bs (a # rs))"
  shows "bnullable a \<or> (\<not> bnullable a \<and> bnullable (AALTs bs rs))"
  using assms
  apply(induct rs)
   apply(auto)
  done

lemma r0:
  assumes "bnullable a" 
  shows  "bmkeps (AALTs bs (a # rs)) = bs @ (bmkeps a)"
  using assms
  by (metis bmkeps.simps(3) bmkeps.simps(4) list.exhaust)

lemma r1:
  assumes "\<not> bnullable a" "bnullable (AALTs bs rs)"
  shows  "bmkeps (AALTs bs (a # rs)) = bmkeps (AALTs bs rs)"
  using assms
  apply(induct rs)
   apply(auto)
  done

lemma r2:
  assumes "x \<in> set rs" "bnullable x"
  shows "bnullable (AALTs bs rs)"
  using assms
  apply(induct rs)
   apply(auto)
  done

lemma  r3:
  assumes "\<not> bnullable r" 
          " \<exists> x \<in> set rs. bnullable x"
  shows "retrieve (AALTs bs rs) (mkeps (erase (AALTs bs rs))) =
         retrieve (AALTs bs (r # rs)) (mkeps (erase (AALTs bs (r # rs))))"
  using assms
  apply(induct rs arbitrary: r bs)
   apply(auto)[1]
  apply(auto)
  using bnullable_correctness apply blast
    apply(auto simp add: bnullable_correctness mkeps_nullable retrieve_fuse2)
   apply(subst retrieve_fuse2[symmetric])
  apply (smt bnullable.simps(4) bnullable_correctness erase.simps(5) erase.simps(6) insert_iff list.exhaust list.set(2) mkeps.simps(3) mkeps_nullable)
   apply(simp)
  apply(case_tac "bnullable a")
  apply (smt append_Nil2 bnullable.simps(4) bnullable_correctness erase.simps(5) erase.simps(6) fuse.simps(4) insert_iff list.exhaust list.set(2) mkeps.simps(3) mkeps_nullable retrieve_fuse2)
  apply(drule_tac x="a" in meta_spec)
  apply(drule_tac x="bs" in meta_spec)
  apply(drule meta_mp)
   apply(simp)
  apply(drule meta_mp)
   apply(auto)
  apply(subst retrieve_fuse2[symmetric])
  apply(case_tac rs)
    apply(simp)
   apply(auto)[1]
      apply (simp add: bnullable_correctness)
  apply (metis append_Nil2 bnullable_correctness erase_fuse fuse.simps(4) list.set_intros(1) mkeps.simps(3) mkeps_nullable nullable.simps(4) r2)
    apply (simp add: bnullable_correctness)
  apply (metis append_Nil2 bnullable_correctness erase.simps(6) erase_fuse fuse.simps(4) list.set_intros(2) mkeps.simps(3) mkeps_nullable r2)
  apply(simp)
  done


lemma t: 
  assumes "\<forall>r \<in> set rs. nullable (erase r) \<longrightarrow> bmkeps r = retrieve r (mkeps (erase r))" 
          "nullable (erase (AALTs bs rs))"
  shows " bmkeps (AALTs bs rs) = retrieve (AALTs bs rs) (mkeps (erase (AALTs bs rs)))"
  using assms
  apply(induct rs arbitrary: bs)
   apply(simp)
  apply(auto simp add: bnullable_correctness)
   apply(case_tac rs)
     apply(auto simp add: bnullable_correctness)[2]
   apply(subst r1)
     apply(simp)
    apply(rule r2)
     apply(assumption)
    apply(simp)
   apply(drule_tac x="bs" in meta_spec)
   apply(drule meta_mp)
    apply(auto)[1]
   prefer 2
  apply(case_tac "bnullable a")
    apply(subst r0)
     apply blast
    apply(subgoal_tac "nullable (erase a)")
  prefer 2
  using bnullable_correctness apply blast
  apply (metis (no_types, lifting) erase.simps(5) erase.simps(6) list.exhaust mkeps.simps(3) retrieve.simps(3) retrieve.simps(4))
  apply(subst r1)
     apply(simp)
  using r2 apply blast
  apply(drule_tac x="bs" in meta_spec)
   apply(drule meta_mp)
    apply(auto)[1]
   apply(simp)
  using r3 apply blast
  apply(auto)
  using r3 by blast

lemma bmkeps_retrieve:
  assumes "nullable (erase r)"
  shows "bmkeps r = retrieve r (mkeps (erase r))"
  using assms
  apply(induct r)
         apply(simp)
        apply(simp)
       apply(simp)
    apply(simp)
   defer
   apply(simp)
  apply(rule t)
   apply(auto)
  done

lemma bder_retrieve:
  assumes "\<Turnstile> v : der c (erase r)"
  shows "retrieve (bder c r) v = retrieve r (injval (erase r) c v)"
  using assms
  apply(induct r arbitrary: v rule: erase.induct)
  apply(auto elim!: Prf_elims simp add: retrieve_fuse2 bnullable_correctness bmkeps_retrieve)
  apply(case_tac va)
   apply(simp)
  apply(auto)
  by (smt Prf_elims(3) injval.simps(2) injval.simps(3) retrieve.simps(4) retrieve.simps(5) same_append_eq)
  

lemma MAIN_decode:
  assumes "\<Turnstile> v : ders s r"
  shows "Some (flex r id s v) = decode (retrieve (bders (intern r) s) v) r"
  using assms
proof (induct s arbitrary: v rule: rev_induct)
  case Nil
  have "\<Turnstile> v : ders [] r" by fact
  then have "\<Turnstile> v : r" by simp
  then have "Some v = decode (retrieve (intern r) v) r"
    using decode_code retrieve_code by auto
  then show "Some (flex r id [] v) = decode (retrieve (bders (intern r) []) v) r"
    by simp
next
  case (snoc c s v)
  have IH: "\<And>v. \<Turnstile> v : ders s r \<Longrightarrow> 
     Some (flex r id s v) = decode (retrieve (bders (intern r) s) v) r" by fact
  have asm: "\<Turnstile> v : ders (s @ [c]) r" by fact
  then have asm2: "\<Turnstile> injval (ders s r) c v : ders s r" 
    by(simp add: Prf_injval ders_append)
  have "Some (flex r id (s @ [c]) v) = Some (flex r id s (injval (ders s r) c v))"
    by (simp add: flex_append)
  also have "... = decode (retrieve (bders (intern r) s) (injval (ders s r) c v)) r"
    using asm2 IH by simp
  also have "... = decode (retrieve (bder c (bders (intern r) s)) v) r"
    using asm by(simp_all add: bder_retrieve ders_append)
  finally show "Some (flex r id (s @ [c]) v) = 
                 decode (retrieve (bders (intern r) (s @ [c])) v) r" by (simp add: bders_append)
qed


definition blexer where
 "blexer r s \<equiv> if bnullable (bders (intern r) s) then 
                decode (bmkeps (bders (intern r) s)) r else None"

lemma blexer_correctness:
  shows "blexer r s = lexer r s"
proof -
  { define bds where "bds \<equiv> bders (intern r) s"
    define ds  where "ds \<equiv> ders s r"
    assume asm: "nullable ds"
    have era: "erase bds = ds" 
      unfolding ds_def bds_def by simp
    have mke: "\<Turnstile> mkeps ds : ds"
      using asm by (simp add: mkeps_nullable)
    have "decode (bmkeps bds) r = decode (retrieve bds (mkeps ds)) r"
      using bmkeps_retrieve
      using asm era by (simp add: bmkeps_retrieve)
    also have "... =  Some (flex r id s (mkeps ds))"
      using mke by (simp_all add: MAIN_decode ds_def bds_def)
    finally have "decode (bmkeps bds) r = Some (flex r id s (mkeps ds))" 
      unfolding bds_def ds_def .
  }
  then show "blexer r s = lexer r s"
    unfolding blexer_def lexer_flex
    apply(subst bnullable_correctness[symmetric])
    apply(simp)
    done
qed


fun distinctBy :: "'a list \<Rightarrow> ('a \<Rightarrow> 'b) \<Rightarrow> 'b set \<Rightarrow> 'a list"
  where
  "distinctBy [] f acc = []"
| "distinctBy (x#xs) f acc = 
     (if (f x) \<in> acc then distinctBy xs f acc 
      else x # (distinctBy xs f ({f x} \<union> acc)))"

fun flts :: "arexp list \<Rightarrow> arexp list"
  where 
  "flts [] = []"
| "flts (AZERO # rs) = flts rs"
| "flts ((AALTs bs  rs1) # rs) = (map (fuse bs) rs1) @ flts rs"
| "flts (r1 # rs) = r1 # flts rs"

fun bsimp_ASEQ :: "bit list \<Rightarrow> arexp \<Rightarrow> arexp \<Rightarrow> arexp"
  where
  "bsimp_ASEQ _ AZERO _ = AZERO"
| "bsimp_ASEQ _ _ AZERO = AZERO"
| "bsimp_ASEQ bs1 (AONE bs2) r2 = fuse (bs1 @ bs2) r2"
| "bsimp_ASEQ bs1 r1 r2 = ASEQ  bs1 r1 r2"


fun bsimp_AALTs :: "bit list \<Rightarrow> arexp list \<Rightarrow> arexp"
  where
  "bsimp_AALTs _ [] = AZERO"
| "bsimp_AALTs bs1 [r] = fuse bs1 r"
| "bsimp_AALTs bs1 rs = AALTs bs1 rs"


fun bsimp :: "arexp \<Rightarrow> arexp" 
  where
  "bsimp (ASEQ bs1 r1 r2) = bsimp_ASEQ bs1 (bsimp r1) (bsimp r2)"
| "bsimp (AALTs bs1 rs) = bsimp_AALTs bs1 (flts (map bsimp rs))"
| "bsimp r = r"

fun 
  bders_simp :: "arexp \<Rightarrow> string \<Rightarrow> arexp"
where
  "bders_simp r [] = r"
| "bders_simp r (c # s) = bders_simp (bsimp (bder c r)) s"

definition blexer_simp where
 "blexer_simp r s \<equiv> if bnullable (bders_simp (intern r) s) then 
                decode (bmkeps (bders_simp (intern r) s)) r else None"


lemma asize0:
  shows "0 < asize r"
  apply(induct  r)
       apply(auto)
  done


lemma bders_simp_append:
  shows "bders_simp r (s1 @ s2) = bders_simp (bders_simp r s1) s2"
  apply(induct s1 arbitrary: r s2)
   apply(simp)
  apply(simp)
  done

lemma bsimp_ASEQ_size:
  shows "asize (bsimp_ASEQ bs r1 r2) \<le> Suc (asize r1 + asize r2)"
  apply(induct bs r1 r2 rule: bsimp_ASEQ.induct)
  apply(auto)
  done

lemma fuse_size:
  shows "asize (fuse bs r) = asize r"
  apply(induct r)
  apply(auto)
  done

lemma flts_size:
  shows "sum_list (map asize (flts rs)) \<le> sum_list (map asize rs)"
  apply(induct rs rule: flts.induct)
        apply(simp_all)
  by (metis (mono_tags, lifting) add_mono_thms_linordered_semiring(1) comp_apply fuse_size le_SucI order_refl sum_list_cong)
  

lemma bsimp_AALTs_size:
  shows "asize (bsimp_AALTs bs rs) \<le> Suc (sum_list (map asize rs))"
  apply(induct rs rule: bsimp_AALTs.induct)
  apply(auto simp add: fuse_size)
  done
  
lemma bsimp_size:
  shows "asize (bsimp r) \<le> asize r"
  apply(induct r)
       apply(simp_all)
   apply (meson Suc_le_mono add_mono_thms_linordered_semiring(1) bsimp_ASEQ_size le_trans)
  apply(rule le_trans)
   apply(rule bsimp_AALTs_size)
  apply(simp)
   apply(rule le_trans)
   apply(rule flts_size)
  by (simp add: sum_list_mono)

fun nonalt :: "arexp \<Rightarrow> bool"
  where
  "nonalt (AALTs bs2 rs) = False"
| "nonalt r = True"




lemma bsimp_AALTs_size2:
  assumes "\<forall>r \<in> set  rs. nonalt r"
  shows "asize (bsimp_AALTs bs rs) \<ge> sum_list (map asize rs)"
  using assms
  apply(induct rs rule: bsimp_AALTs.induct)
    apply(simp_all add: fuse_size)
  done

lemma qq:
  shows "map (asize \<circ> fuse bs) rs = map asize rs"
  apply(induct rs)
   apply(auto simp add: fuse_size)
  done

lemma flts_size2:
  assumes "\<exists>bs rs'. AALTs bs  rs' \<in> set rs"
  shows "sum_list (map asize (flts rs)) < sum_list (map asize rs)"
  using assms
  apply(induct rs)
   apply(auto simp add: qq)
   apply (simp add: flts_size less_Suc_eq_le)
  apply(case_tac a)
       apply(auto simp add: qq)
   prefer 2
   apply (simp add: flts_size le_imp_less_Suc)
  using less_Suc_eq by auto
  
lemma L_bsimp_ASEQ:
  "L (SEQ (erase r1) (erase r2)) = L (erase (bsimp_ASEQ bs r1 r2))"
  apply(induct bs r1 r2 rule: bsimp_ASEQ.induct)
  apply(simp_all)
  by (metis erase_fuse fuse.simps(4))

lemma L_bsimp_AALTs:
  "L (erase (AALTs bs rs)) = L (erase (bsimp_AALTs bs rs))"
  apply(induct bs rs rule: bsimp_AALTs.induct)
  apply(simp_all add: erase_fuse)
  done

lemma L_erase_AALTs:
  shows "L (erase (AALTs bs rs)) = \<Union> (L ` erase ` (set rs))"
  apply(induct rs)
   apply(simp)
  apply(simp)
  apply(case_tac rs)
   apply(simp)
  apply(simp)
  done

lemma L_erase_flts:
  shows "\<Union> (L ` erase ` (set (flts rs))) = \<Union> (L ` erase ` (set rs))"
  apply(induct rs rule: flts.induct)
        apply(simp_all)
  apply(auto)
  using L_erase_AALTs erase_fuse apply auto[1]
  by (simp add: L_erase_AALTs erase_fuse)


lemma L_bsimp_erase:
  shows "L (erase r) = L (erase (bsimp r))"
  apply(induct r)
  apply(simp)
  apply(simp)
  apply(simp)
  apply(auto simp add: Sequ_def)[1]
  apply(subst L_bsimp_ASEQ[symmetric])
  apply(auto simp add: Sequ_def)[1]
  apply(subst (asm)  L_bsimp_ASEQ[symmetric])
  apply(auto simp add: Sequ_def)[1]
   apply(simp)
   apply(subst L_bsimp_AALTs[symmetric])
   defer
   apply(simp)
  apply(subst (2)L_erase_AALTs)
  apply(subst L_erase_flts)
  apply(auto)
   apply (simp add: L_erase_AALTs)
  using L_erase_AALTs by blast

lemma bsimp_ASEQ0:
  shows "bsimp_ASEQ bs r1 AZERO = AZERO"
  apply(induct r1)
  apply(auto)
  done



lemma bsimp_ASEQ1:
  assumes "r1 \<noteq> AZERO" "r2 \<noteq> AZERO" "\<forall>bs. r1 \<noteq> AONE bs"
  shows "bsimp_ASEQ bs r1 r2 = ASEQ bs r1 r2"
  using assms
  apply(induct bs r1 r2 rule: bsimp_ASEQ.induct)
  apply(auto)
  done

lemma bsimp_ASEQ2:
  shows "bsimp_ASEQ bs (AONE bs1) r2 = fuse (bs @ bs1) r2"
  apply(induct r2)
  apply(auto)
  done


lemma L_bders_simp:
  shows "L (erase (bders_simp r s)) = L (erase (bders r s))"
  apply(induct s arbitrary: r rule: rev_induct)
   apply(simp)
  apply(simp)
  apply(simp add: ders_append)
  apply(simp add: bders_simp_append)
  apply(simp add: L_bsimp_erase[symmetric])
  by (simp add: der_correctness)

lemma b1:
  "bsimp_ASEQ bs1 (AONE bs) r =  fuse (bs1 @ bs) r" 
  apply(induct r)
       apply(auto)
  done

lemma b2:
  assumes "bnullable r"
  shows "bmkeps (fuse bs r) = bs @ bmkeps r"
  by (simp add: assms bmkeps_retrieve bnullable_correctness erase_fuse mkeps_nullable retrieve_fuse2)

lemma b3:
  shows "bnullable r = bnullable (bsimp r)"
  using L_bsimp_erase bnullable_correctness nullable_correctness by auto


lemma b4:
  shows "bnullable (bders_simp r s) = bnullable (bders r s)"
  by (metis L_bders_simp bnullable_correctness lexer.simps(1) lexer_correct_None option.distinct(1))

lemma q1:
  assumes "\<forall>r \<in> set rs. bmkeps(bsimp r) = bmkeps r"
  shows "map (\<lambda>r. bmkeps(bsimp r)) rs = map bmkeps rs"
  using assms
  apply(induct rs)
  apply(simp)
  apply(simp)
  done

lemma q3:
  assumes "\<exists>r \<in> set rs. bnullable r"
  shows "bmkeps (AALTs bs rs) = bmkeps (bsimp_AALTs bs rs)"
  using assms
  apply(induct bs rs rule: bsimp_AALTs.induct)
    apply(simp)
   apply(simp)
  apply (simp add: b2)
  apply(simp)
  done

lemma qq1:
  assumes "\<exists>r \<in> set rs. bnullable r"
  shows "bmkeps (AALTs bs (rs @ rs1)) = bmkeps (AALTs bs rs)"
  using assms
  apply(induct rs arbitrary: rs1 bs)
  apply(simp)
  apply(simp)
  by (metis Nil_is_append_conv bmkeps.simps(4) neq_Nil_conv r0 split_list_last)

lemma qq2:
  assumes "\<forall>r \<in> set rs. \<not> bnullable r" "\<exists>r \<in> set rs1. bnullable r"
  shows "bmkeps (AALTs bs (rs @ rs1)) = bmkeps (AALTs bs rs1)"
  using assms
  apply(induct rs arbitrary: rs1 bs)
  apply(simp)
  apply(simp)
  by (metis append_assoc in_set_conv_decomp r1 r2)
  
lemma qq3:
  shows "bnullable (AALTs bs rs) = (\<exists>r \<in> set rs. bnullable r)"
  apply(induct rs arbitrary: bs)
  apply(simp)
  apply(simp)
  done

lemma fuse_empty:
  shows "fuse [] r = r"
  apply(induct r)
       apply(auto)
  done

lemma flts_fuse:
  shows "map (fuse bs) (flts rs) = flts (map (fuse bs) rs)"
  apply(induct rs arbitrary: bs rule: flts.induct)
        apply(auto simp add: fuse_append)
  done

lemma bsimp_ASEQ_fuse:
  shows "fuse bs1 (bsimp_ASEQ bs2 r1 r2) = bsimp_ASEQ (bs1 @ bs2) r1 r2"
  apply(induct r1 r2 arbitrary: bs1 bs2 rule: bsimp_ASEQ.induct)
  apply(auto)
  done

lemma bsimp_AALTs_fuse:
  assumes "\<forall>r \<in> set rs. fuse bs1 (fuse bs2 r) = fuse (bs1 @ bs2) r"
  shows "fuse bs1 (bsimp_AALTs bs2 rs) = bsimp_AALTs (bs1 @ bs2) rs"
  using assms
  apply(induct bs2 rs arbitrary: bs1 rule: bsimp_AALTs.induct)
  apply(auto)
  done



lemma bsimp_fuse:
  shows "fuse bs (bsimp r) = bsimp (fuse bs r)"
apply(induct r arbitrary: bs)
       apply(simp)
      apply(simp)
     apply(simp)
    prefer 3
    apply(simp)
   apply(simp)
   apply (simp add: bsimp_ASEQ_fuse)
  apply(simp)
  by (simp add: bsimp_AALTs_fuse fuse_append)

lemma bsimp_fuse_AALTs:
  shows "fuse bs (bsimp (AALTs [] rs)) = bsimp (AALTs bs rs)"
  apply(subst bsimp_fuse) 
  apply(simp)
  done

lemma bsimp_fuse_AALTs2:
  shows "fuse bs (bsimp_AALTs [] rs) = bsimp_AALTs bs rs"
  using bsimp_AALTs_fuse fuse_append by auto
  

lemma bsimp_ASEQ_idem:
  assumes "bsimp (bsimp r1) = bsimp r1" "bsimp (bsimp r2) = bsimp r2"
  shows "bsimp (bsimp_ASEQ x1 (bsimp r1) (bsimp r2)) = bsimp_ASEQ x1 (bsimp r1) (bsimp r2)"
  using assms
  apply(case_tac "bsimp r1 = AZERO")
    apply(simp)
 apply(case_tac "bsimp r2 = AZERO")
    apply(simp)
  apply (metis bnullable.elims(2) bnullable.elims(3) bsimp.simps(3) bsimp_ASEQ.simps(2) bsimp_ASEQ.simps(3) bsimp_ASEQ.simps(4) bsimp_ASEQ.simps(5) bsimp_ASEQ.simps(6))  
  apply(case_tac "\<exists>bs. bsimp r1 = AONE bs")
    apply(auto)[1]
    apply(subst bsimp_ASEQ2)
   apply(subst bsimp_ASEQ2)
  apply (metis assms(2) bsimp_fuse)
      apply(subst bsimp_ASEQ1)
      apply(auto)
  done


fun nonnested :: "arexp \<Rightarrow> bool"
  where
  "nonnested (AALTs bs2 []) = True"
| "nonnested (AALTs bs2 ((AALTs bs1 rs1) # rs2)) = False"
| "nonnested (AALTs bs2 (r # rs2)) = nonnested (AALTs bs2 rs2)"
| "nonnested r = True"


lemma  k0:
  shows "flts (r # rs1) = flts [r] @ flts rs1"
  apply(induct r arbitrary: rs1)
   apply(auto)
  done

lemma  k00:
  shows "flts (rs1 @ rs2) = flts rs1 @ flts rs2"
  apply(induct rs1 arbitrary: rs2)
   apply(auto)
  by (metis append.assoc k0)

lemma  k0a:
  shows "flts [AALTs bs rs] = map (fuse bs)  rs"
  apply(simp)
  done

fun spill where
 "spill (AALTs bs rs) =  map (fuse bs) rs"

lemma  k0a2:
  assumes "\<not> nonalt r"  
  shows "flts [r] = spill r"
  using  assms
  apply(case_tac r)
  apply(simp_all)
  done

lemma  k0b:
  assumes "nonalt r" "r \<noteq> AZERO"
  shows "flts [r] = [r]"
  using assms
  apply(case_tac  r)
  apply(simp_all)
  done

lemma nn1:
  assumes "nonnested (AALTs bs rs)"
  shows "\<nexists>bs1 rs1. flts rs = [AALTs bs1 rs1]"
  using assms
  apply(induct rs rule: flts.induct)
  apply(auto)
  done

lemma nn1q:
  assumes "nonnested (AALTs bs rs)"
  shows "\<nexists>bs1 rs1. AALTs bs1 rs1 \<in> set (flts rs)"
  using assms
  apply(induct rs rule: flts.induct)
  apply(auto)
  done

lemma nn1qq:
  assumes "nonnested (AALTs bs rs)"
  shows "\<nexists>bs1 rs1. AALTs bs1 rs1 \<in> set rs"
  using assms
  apply(induct rs rule: flts.induct)
  apply(auto)
  done

lemma nn10:
  assumes "nonnested (AALTs cs rs)" 
  shows "nonnested (AALTs (bs @ cs) rs)"
  using assms
  apply(induct rs arbitrary: cs bs)
   apply(simp_all)
  apply(case_tac a)
       apply(simp_all)
  done

lemma nn11a:
  assumes "nonalt r"
  shows "nonalt (fuse bs r)"
  using assms
  apply(induct r)
       apply(auto)
  done


lemma nn1a:
  assumes "nonnested r"
  shows "nonnested (fuse bs r)"
  using assms
  apply(induct bs r arbitrary: rule: fuse.induct)
       apply(simp_all add: nn10)
  done  

lemma n0:
  shows "nonnested (AALTs bs rs) \<longleftrightarrow> (\<forall>r \<in> set rs. nonalt r)"
  apply(induct rs  arbitrary: bs)
   apply(auto)
    apply (metis list.set_intros(1) nn1qq nonalt.elims(3))
   apply (metis list.set_intros(2) nn1qq nonalt.elims(3))
  by (metis nonalt.elims(2) nonnested.simps(3) nonnested.simps(4) nonnested.simps(5) nonnested.simps(6) nonnested.simps(7))

  
  

lemma nn1c:
  assumes "\<forall>r \<in> set rs. nonnested r"
  shows "\<forall>r \<in> set (flts rs). nonalt r"
  using assms
  apply(induct rs rule: flts.induct)
        apply(auto)
  apply(rule nn11a)
  by (metis nn1qq nonalt.elims(3))

lemma nn1bb:
  assumes "\<forall>r \<in> set rs. nonalt r"
  shows "nonnested (bsimp_AALTs bs rs)"
  using assms
  apply(induct bs rs rule: bsimp_AALTs.induct)
    apply(auto)
   apply (metis nn11a nonalt.simps(1) nonnested.elims(3))
  using n0 by auto
    
lemma nn1b:
  shows "nonnested (bsimp r)"
  apply(induct r)
       apply(simp_all)
  apply(case_tac "bsimp r1 = AZERO")
    apply(simp)
 apply(case_tac "bsimp r2 = AZERO")
   apply(simp)
    apply(subst bsimp_ASEQ0)
  apply(simp)
  apply(case_tac "\<exists>bs. bsimp r1 = AONE bs")
    apply(auto)[1]
    apply(subst bsimp_ASEQ2)
  apply (simp add: nn1a)    
   apply(subst bsimp_ASEQ1)
      apply(auto)
  apply(rule nn1bb)
  apply(auto)
  by (metis (mono_tags, hide_lams) imageE nn1c set_map)

lemma rt:
  shows "sum_list (map asize (flts (map bsimp rs))) \<le> sum_list (map asize rs)"
  apply(induct rs)
   apply(simp)
  apply(simp)
  apply(subst  k0)
  apply(simp)
  by (smt add_le_cancel_right add_mono bsimp_size flts.simps(1) flts_size k0 le_iff_add list.simps(9) map_append sum_list.Cons sum_list.append trans_le_add1)

lemma flts_idem:
  assumes "\<forall>r \<in> set rs. bsimp (bsimp r) = bsimp r"
  shows "flts (map bsimp (flts (map bsimp rs))) = flts (map bsimp rs)"
  using assms
  apply(induct rs)
   apply(simp)
  apply(simp)
  apply(subst k0)
  apply(subst (2) k0) 
  apply(case_tac "bsimp a = AZERO")
   apply(simp)
  apply(case_tac "nonalt (bsimp a)")
  thm k0 k0a k0b
  apply(subst k0b)
     apply(simp)
    apply(simp)
   apply(simp)
  apply(subst k0b)
     apply(simp)
    apply(simp)
   apply(simp)
  apply(subst k0)
  apply(subst k0b)
     apply(simp)
    apply(simp)
   apply(simp)
  apply(simp)
  apply(simp add: k00)
  apply(subst k0a2)
   apply(simp)
  apply(subst k0a2)
   apply(simp)
  apply(case_tac a)
  apply(simp_all)
  sorry

lemma bsimp_AALTs_idem:
  (*assumes "\<forall>r \<in> set rs. bsimp (bsimp r) = bsimp r \<and> nonalt (bsimp r)" *)
  (*assumes "\<forall>r \<in> set rs. bsimp (bsimp r) = bsimp r" *)
  shows "bsimp (bsimp_AALTs bs rs) = bsimp_AALTs bs (flts (map bsimp rs))"
  apply(induct rs arbitrary: bs taking: "\<lambda>rs. sum_list (map asize rs)" rule: measure_induct)
  apply(case_tac x)
  apply(simp)
  apply(simp)
  apply(case_tac "\<exists>bs'  rs'. a = AALTs bs' rs'")
   apply(clarify)
  apply(case_tac list)
        apply(simp)
       apply(simp) 

  apply(drule_tac x="flts (rs' @ list)" in spec)
  apply(erule impE)
  prefer 2
  apply(case_tac a)
       apply(simp)
       apply(case_tac list)
        apply(simp)
       apply(simp) 
   apply(case_tac list)
        apply(simp)
      apply(simp) 
   apply(case_tac list)
        apply(simp)
     apply(simp) 
  prefer 3
   apply(case_tac list)
        apply(simp)
       apply(simp) 
   apply(case_tac list)
    apply(simp)
      

       apply(simp) 

      apply(case_tac "flts (map bsimp list)")
       apply(simp)
      apply(simp)
  
  

   prefer 2
   apply(simp)
  
   apply(subst k0)
   apply(subst (2) k0)
  
  apply(case_tac a)
       apply(simp_all)
  
  prefer 2
   apply(simp)
  apply(case_tac r)
       apply(auto)
   apply(case_tac "bsimp x42 = AZERO")
    apply(simp)
 apply(case_tac "bsimp x43 = AZERO")
   apply(simp)
   apply(subst bsimp_ASEQ0)
   apply(subst bsimp_ASEQ0)
  apply(simp)
  apply(case_tac "\<exists>bs. bsimp x42 = AONE bs")
    apply(auto)[1]
    apply(subst bsimp_ASEQ2)
   apply(subst bsimp_ASEQ2)
  prefer 2
      apply(subst bsimp_ASEQ1)
      apply(auto)
  apply(subst bsimp_ASEQ1)
      apply(auto)
  apply(subst (asm) bsimp_ASEQ2)
   apply(subst (asm) bsimp_ASEQ2)
  using flts_fuse bsimp_fuse bsimp_fuse_AALTs bsimp_fuse_AALTs2 bsimp_AALTs.simps flts.simps
    
   apply(case_tac x43)
        apply(simp_all)
    prefer 3
  oops

lemma bsimp_idem:
  shows "bsimp (bsimp r) = bsimp r"
apply(induct r taking: "asize" rule: measure_induct)
  apply(case_tac x)
  apply(simp)
      apply(simp)
     apply(simp)
    prefer 3
    apply(simp)
   apply(simp)
   apply (simp add: bsimp_ASEQ_idem)
  apply(clarify)
  apply(case_tac x52)
  apply(simp)
  apply(simp)
  apply(subst k0)
  apply(subst (2) k0)
  apply(case_tac "bsimp a = AZERO")
   apply(simp)
   apply(frule_tac x="AALTs x51 (flts (map bsimp list))" in spec)
   apply(drule mp)
    apply(simp)
  apply (meson add_le_cancel_right asize0 le_trans not_le rt trans_le_add2)
  apply(simp)
    apply(subst (asm) flts_idem)
     apply(auto)[1]
     apply(drule_tac  x="r" in spec)
  apply (metis add.commute add_lessD1 not_add_less1 not_less_eq sum_list_map_remove1)
  apply(simp)
  apply(subst flts_idem)
   apply(auto)[1]
     apply(drule_tac  x="r" in spec)
  apply (metis add.commute add_lessD1 not_add_less1 not_less_eq sum_list_map_remove1)
   apply(simp)
  apply(case_tac "nonalt (bsimp a)")
  apply(subst k0b)
     apply(simp)
    apply(simp)
  apply(subst k0b)
     apply(simp)
    apply(simp)
   apply(auto)[1]
  apply(frule_tac x="AALTs x51 (bsimp a # flts (map bsimp list))" in spec)
   apply(drule mp)
    apply(simp)
    prefer 2
    apply(simp)
  apply(subst (asm) k0)
  apply(subst (asm) flts_idem)
     apply(auto)[1]
  apply (simp add: sum_list_map_remove1)
  apply(subst (asm) k0b)
     apply(simp)
    apply(simp)
    apply(simp)
   apply(subst k0)
  apply(subst flts_idem)
     apply(auto)[1]
     apply (simp add: sum_list_map_remove1)
  apply(subst k0b)
     apply(simp)
    apply(simp)
  apply(simp)
lemma XX_bder:
  shows "bsimp (bder c (bsimp r)) = (bsimp \<circ> bder c) r"
  apply(induct r)
       apply(simp)
      apply(simp)
     apply(simp)
    prefer 3
    apply(simp)
  prefer 2
   apply(simp)
   apply(case_tac x2a)
    apply(simp)
  apply(simp)
  apply(auto)[1]


lemma q3a:
  assumes "\<exists>r \<in> set rs. bnullable r"
  shows "bmkeps (AALTs bs (map (fuse bs1) rs)) = bmkeps (AALTs (bs@bs1) rs)"
  using assms
  apply(induct rs arbitrary: bs bs1)
   apply(simp)
  apply(simp)
  apply(auto)
   apply (metis append_assoc b2 bnullable_correctness erase_fuse r0)
  apply(case_tac "bnullable a")
   apply (metis append.assoc b2 bnullable_correctness erase_fuse r0)
  apply(case_tac rs)
  apply(simp)
  apply(simp)
  apply(auto)[1]
   apply (metis bnullable_correctness erase_fuse)+
  done

lemma qq4:
  assumes "\<exists>x\<in>set list. bnullable x"
  shows "\<exists>x\<in>set (flts list). bnullable x"
  using assms
  apply(induct list rule: flts.induct)
        apply(auto)
  by (metis UnCI bnullable_correctness erase_fuse imageI)
  

lemma qs3:
  assumes "\<exists>r \<in> set rs. bnullable r"
  shows "bmkeps (AALTs bs rs) = bmkeps (AALTs bs (flts rs))"
  using assms
  apply(induct rs arbitrary: bs taking: size rule: measure_induct)
  apply(case_tac x)
  apply(simp)
  apply(simp)
  apply(case_tac a)
       apply(simp)
       apply (simp add: r1)
      apply(simp)
      apply (simp add: r0)
     apply(simp)
     apply(case_tac "flts list")
      apply(simp)
  apply (metis L_erase_AALTs L_erase_flts L_flat_Prf1 L_flat_Prf2 Prf_elims(1) bnullable_correctness erase.simps(4) mkeps_nullable r2)
     apply(simp)
     apply (simp add: r1)
    prefer 3
    apply(simp)
    apply (simp add: r0)
   prefer 2
   apply(simp)
  apply(case_tac "\<exists>x\<in>set x52. bnullable x")
  apply(case_tac "list")
    apply(simp)
    apply (metis b2 fuse.simps(4) q3a r2)
   apply(erule disjE)
    apply(subst qq1)
     apply(auto)[1]
     apply (metis bnullable_correctness erase_fuse)
    apply(simp)
     apply (metis b2 fuse.simps(4) q3a r2)
    apply(simp)
    apply(auto)[1]
     apply(subst qq1)
      apply (metis bnullable_correctness erase_fuse image_eqI set_map)
     apply (metis b2 fuse.simps(4) q3a r2)
  apply(subst qq1)
      apply (metis bnullable_correctness erase_fuse image_eqI set_map)
    apply (metis b2 fuse.simps(4) q3a r2)
   apply(simp)
   apply(subst qq2)
     apply (metis bnullable_correctness erase_fuse imageE set_map)
  prefer 2
  apply(case_tac "list")
     apply(simp)
    apply(simp)
   apply (simp add: qq4)
  apply(simp)
  apply(auto)
   apply(case_tac list)
    apply(simp)
   apply(simp)
   apply (simp add: r0)
  apply(case_tac "bnullable (ASEQ x41 x42 x43)")
   apply(case_tac list)
    apply(simp)
   apply(simp)
   apply (simp add: r0)
  apply(simp)
  using qq4 r1 r2 by auto



lemma k1:
  assumes "\<And>x2aa. \<lbrakk>x2aa \<in> set x2a; bnullable x2aa\<rbrakk> \<Longrightarrow> bmkeps x2aa = bmkeps (bsimp x2aa)"
          "\<exists>x\<in>set x2a. bnullable x"
        shows "bmkeps (AALTs x1 (flts x2a)) = bmkeps (AALTs x1 (flts (map bsimp x2a)))"
  using assms
  apply(induct x2a)
  apply fastforce
  apply(simp)
  apply(subst k0)
  apply(subst (2) k0)
  apply(auto)[1]
  apply (metis b3 k0 list.set_intros(1) qs3 r0)
  by (smt b3 imageI insert_iff k0 list.set(2) qq3 qs3 r0 r1 set_map)
  
  
  
lemma bmkeps_simp:
  assumes "bnullable r"
  shows "bmkeps r = bmkeps (bsimp r)"
  using  assms
  apply(induct r)
       apply(simp)
      apply(simp)
     apply(simp)
    apply(simp)
    prefer 3
  apply(simp)
   apply(case_tac "bsimp r1 = AZERO")
    apply(simp)
    apply(auto)[1]
  apply (metis L_bsimp_erase L_flat_Prf1 L_flat_Prf2 Prf_elims(1) bnullable_correctness erase.simps(1) mkeps_nullable)
 apply(case_tac "bsimp r2 = AZERO")
    apply(simp)  
    apply(auto)[1]
  apply (metis L_bsimp_erase L_flat_Prf1 L_flat_Prf2 Prf_elims(1) bnullable_correctness erase.simps(1) mkeps_nullable)
  apply(case_tac "\<exists>bs. bsimp r1 = AONE bs")
    apply(auto)[1]
    apply(subst b1)
    apply(subst b2)
  apply(simp add: b3[symmetric])
    apply(simp)
   apply(subgoal_tac "bsimp_ASEQ x1 (bsimp r1) (bsimp r2) = ASEQ x1 (bsimp r1) (bsimp r2)")
    prefer 2
  apply (smt b3 bnullable.elims(2) bsimp_ASEQ.simps(17) bsimp_ASEQ.simps(19) bsimp_ASEQ.simps(20) bsimp_ASEQ.simps(21) bsimp_ASEQ.simps(22) bsimp_ASEQ.simps(24) bsimp_ASEQ.simps(25) bsimp_ASEQ.simps(26) bsimp_ASEQ.simps(27) bsimp_ASEQ.simps(29) bsimp_ASEQ.simps(30) bsimp_ASEQ.simps(31))
   apply(simp)
  apply(simp)
  apply(subst q3[symmetric])
   apply simp
  using b3 qq4 apply auto[1]
  apply(subst qs3)
   apply simp
  using k1 by blast

thm bmkeps_retrieve bmkeps_simp bder_retrieve

lemma bmkeps_bder_AALTs:
  assumes "\<exists>r \<in> set rs. bnullable (bder c r)" 
  shows "bmkeps (bder c (bsimp_AALTs bs rs)) = bmkeps (bsimp_AALTs bs (map (bder c) rs))"
  using assms
  apply(induct rs)
   apply(simp)
  apply(simp)
  apply(auto)
  apply(case_tac rs)
    apply(simp)
  apply (metis (full_types) Prf_injval bder_retrieve bmkeps_retrieve bnullable_correctness erase_bder erase_fuse mkeps_nullable retrieve_fuse2)
   apply(simp)
  apply(case_tac  rs)
   apply(simp_all)
  done


fun extr :: "arexp  \<Rightarrow> (bit list) set" where
  "extr (AONE bs) = {bs}"
| "extr (ACHAR bs c) = {bs}"
| "extr (AALTs bs (r#rs)) = 
     {bs @ bs' | bs'.  bs' \<in> extr r} \<union>
     {bs @ bs' | bs'.  bs' \<in> extr (AALTs [] rs)}"
| "extr (ASEQ bs r1 r2) = 
     {bs @ bs1 @ bs2 | bs1 bs2. bs1 \<in> extr r1 \<and> bs2 \<in> extr r2}"
| "extr (ASTAR bs r) = {bs @ [S]} \<union>
     {bs @ [Z] @ bs1 @ bs2 | bs1 bs2. bs1 \<in> extr r \<and> bs2 \<in> extr (ASTAR  [] r)}"


lemma MAIN_decode:
  assumes "\<Turnstile> v : ders s r"
  shows "Some (flex r id s v) = decode (retrieve (bders_simp (intern r) s) v) r"
  using assms
proof (induct s arbitrary: v rule: rev_induct)
  case Nil
  have "\<Turnstile> v : ders [] r" by fact
  then have "\<Turnstile> v : r" by simp
  then have "Some v = decode (retrieve (intern r) v) r"
    using decode_code retrieve_code by auto
  then show "Some (flex r id [] v) = decode (retrieve (bders_simp (intern r) []) v) r"
    by simp
next
  case (snoc c s v)
  have IH: "\<And>v. \<Turnstile> v : ders s r \<Longrightarrow> 
     Some (flex r id s v) = decode (retrieve (bders_simp (intern r) s) v) r" by fact
  have asm: "\<Turnstile> v : ders (s @ [c]) r" by fact
  then have asm2: "\<Turnstile> injval (ders s r) c v : ders s r" 
    by(simp add: Prf_injval ders_append)
  have "Some (flex r id (s @ [c]) v) = Some (flex r id s (injval (ders s r) c v))"
    by (simp add: flex_append)
  also have "... = decode (retrieve (bders_simp (intern r) s) (injval (ders s r) c v)) r"
    using asm2 IH by simp
  also have "... = decode (retrieve (bder c (bders_simp (intern r) s)) v) r"
    using asm bder_retrieve ders_append 
    apply -
    apply(drule_tac x="v" in meta_spec)
    apply(drule_tac x="c" in meta_spec)
    apply(drule_tac x="bders_simp (intern r) s" in meta_spec)
    apply(drule_tac  meta_mp)
     apply(simp add: ders_append)
     defer
    apply(simp)
    oops

fun vsimp :: "arexp \<Rightarrow> val \<Rightarrow> val"
  where
  "vsimp (ASEQ _ (AONE _) r) (Seq v1 v2) = vsimp r v1"
| "vsimp _ v = v" 

lemma fuse_vsimp:
  assumes "\<Turnstile> v : (erase r)"
  shows "vsimp (fuse bs r) v = vsimp r v"
  using assms
  apply(induct r arbitrary: v bs)
       apply(simp_all)
  apply(case_tac "\<exists>bs. r1 = AONE bs")
   apply(auto)
   apply (metis Prf_elims(2) vsimp.simps(1))
  apply(erule Prf_elims)
  apply(auto)
  apply(case_tac r1)
       apply(auto)
  done  
  

lemma retrieve_XXX0:
  assumes "\<And>r v. \<lbrakk>r \<in> set rs; \<Turnstile> v : erase r\<rbrakk> \<Longrightarrow> 
              \<exists>v'. \<Turnstile> v' : erase (bsimp r) \<and> retrieve (bsimp r) v' = retrieve r v"
          "\<Turnstile> v : erase (AALTs bs rs)"
        shows "\<exists>v'. \<Turnstile> v' : erase (bsimp_AALTs bs (flts (map bsimp rs))) \<and>
                retrieve (bsimp_AALTs bs (flts (map bsimp rs))) v' = retrieve (AALTs bs rs) v"
  using assms
apply(induct rs arbitrary: bs v taking: size rule: measure_induct)
  apply(case_tac x)
   apply(simp)
  using Prf_elims(1) apply blast
  apply(simp)
  apply(case_tac list)
   apply(simp_all)
   apply(case_tac a)
  apply(simp_all)
  using Prf_elims(1) apply blast
       apply (metis erase.simps(2) fuse.simps(2) retrieve_fuse2)
  using Prf_elims(5) apply force
     apply(erule Prf_elims)
     apply(auto)[1]
  
  
  

  apply(simp)
       apply(erule Prf_elims)
  using Prf_elims(1) apply b last
       apply(auto)
       apply (metis append_Ni l2 erase_fuse fuse.simps(4) retrieve_fuse2)
  apply(case_tac rs)
       apply(auto)

  
  oops

fun get where
  "get (Some v) = v"

  
lemma retrieve_XXX:
  assumes "\<Turnstile> v : erase r" 
  shows "\<Turnstile> get (decode (code v) (erase (bsimp r))) : erase (bsimp r)"
  using assms
apply(induct r arbitrary: v)
       apply(simp)
  using Prf_elims(1) apply auto[1]  
      apply(simp)
  apply (simp add: decode_code)
     apply(simp)
  apply (simp add: decode_code)
    apply(simp)
    apply(erule Prf_elims)
  apply(simp)
   apply(case_tac  "r1 = AZERO")
     apply(simp)
  apply (meson Prf_elims(1) Prf_elims(2))
  apply(case_tac  "r2 = AZERO")
     apply(simp)
     apply (meson Prf_elims(1) Prf_elims(2))
   apply(case_tac "\<exists>bs. bsimp r1 = AONE bs")
     apply(clarify)
     apply(simp)
     apply(subst bsimp_ASEQ2)
     apply(subst bsimp_ASEQ2)
     apply(simp add: erase_fuse)
     apply(case_tac r1)
          apply(simp_all)
  using Prf_elims(4) apply fastforce
      apply(erule Prf_elims)
      apply(simp)
  
      apply(simp)
  
  
     defer
     apply(subst bsimp_ASEQ1)
  using L_bsimp_erase L_flat_Prf1 L_flat_Prf2 apply fastforce
  using L_bsimp_erase L_

lemma retrieve_XXX:
  assumes "\<Turnstile> v : erase r" 
  shows "\<Turnstile> (vsimp (bsimp r) v : erase (bsimp r)  \<and> retrieve (bsimp r) (vsimp (bsimp r) v) = retrieve r v"
  using assms
  apply(induct r arbitrary: v)
       apply(simp)
  using Prf_elims(1) apply blast
      apply(simp)
  using Prf_elims(4) apply fastforce
    apply(simp)
  apply blast
  apply simp
   apply(case_tac  "r1 = AZERO")
     apply(simp)
  apply (meson Prf_elims(1) Prf_elims(2))
  apply(case_tac  "r2 = AZERO")
     apply(simp)
     apply (meson Prf_elims(1) Prf_elims(2))
  apply(erule Prf_elims)
  apply(simp)
   apply(case_tac "\<exists>bs. bsimp r1 = AONE bs")
     apply(clarify)
     apply(simp)
    apply(subst bsimp_ASEQ2)
     defer
     apply(subst bsimp_ASEQ1)
  using L_bsimp_erase L_flat_Prf1 L_flat_Prf2 apply fastforce
  using L_bsimp_erase L_flat_Prf1 L_flat_Prf2 apply fastforce
      apply(simp)
    apply(simp)
  apply(drule_tac  x="v1" in meta_spec)
  apply(drule_tac  x="v2" in meta_spec)
     apply(simp)
  apply(clarify)
     apply(rule_tac x="Seq v' v'a" in exI)
     apply(simp)
  apply (metis Prf.intros(1) Prf_elims(1) bsimp_ASEQ1 erase.simps(1) retrieve.simps(6))
    prefer 3
  apply(drule_tac  x="v1" in meta_spec)
  apply(drule_tac  x="v2" in meta_spec)
     apply(simp)
    apply(clarify)
    apply(rule_tac x="v'a" in exI)
    apply(subst bsimp_ASEQ2)
    apply (metis Prf_elims(4) append_assoc erase_fuse retrieve.simps(1) retrieve_fuse2)
   prefer 2  
   apply(auto)
  apply(case_tac "x2a")
   apply(simp)
  using Prf_elims(1) apply blast
  apply(simp)
  apply(case_tac "list")
   apply(simp)
  sorry  


lemma retrieve_XXX:
  assumes "\<Turnstile> v : erase r" 
  shows "\<exists>v'. \<Turnstile> v' : erase (bsimp r)  \<and> retrieve (bsimp r) v' = retrieve r v"
  using assms
  apply(induct r arbitrary: v)
       apply(simp)
  using Prf_elims(1) apply blast
      apply(simp)
  using Prf_elims(4) apply fastforce
    apply(simp)
  apply blast
  apply simp
   apply(case_tac  "r1 = AZERO")
     apply(simp)
  apply (meson Prf_elims(1) Prf_elims(2))
  apply(case_tac  "r2 = AZERO")
     apply(simp)
     apply (meson Prf_elims(1) Prf_elims(2))
  apply(erule Prf_elims)
  apply(simp)
   apply(case_tac "\<exists>bs. bsimp r1 = AONE bs")
     apply(clarify)
     apply(simp)
    apply(subst bsimp_ASEQ2)
     defer
     apply(subst bsimp_ASEQ1)
  using L_bsimp_erase L_flat_Prf1 L_flat_Prf2 apply fastforce
  using L_bsimp_erase L_flat_Prf1 L_flat_Prf2 apply fastforce
      apply(simp)
    apply(simp)
  apply(drule_tac  x="v1" in meta_spec)
  apply(drule_tac  x="v2" in meta_spec)
     apply(simp)
  apply(clarify)
     apply(rule_tac x="Seq v' v'a" in exI)
     apply(simp)
  apply (metis Prf.intros(1) Prf_elims(1) bsimp_ASEQ1 erase.simps(1) retrieve.simps(6))
    prefer 3
  apply(drule_tac  x="v1" in meta_spec)
  apply(drule_tac  x="v2" in meta_spec)
     apply(simp)
    apply(clarify)
    apply(rule_tac x="v'a" in exI)
    apply(subst bsimp_ASEQ2)
    apply (metis Prf_elims(4) append_assoc erase_fuse retrieve.simps(1) retrieve_fuse2)
   prefer 2  
   apply(auto)
  apply(case_tac "x2a")
   apply(simp)
  using Prf_elims(1) apply blast
  apply(simp)
  apply(case_tac "list")
   apply(simp)
  sorry


lemma TEST:
  assumes "\<Turnstile> v : ders s r"
  shows "\<exists>v'. retrieve (bders (intern r) s) v' = retrieve (bsimp (bders (intern r) s)) v"
  using assms
  apply(induct s arbitrary: r v rule: rev_induct)
   apply(simp)
  
   defer
   apply(simp add: ders_append)
   apply(frule Prf_injval)
  apply(drule_tac x="r" in meta_spec)
   apply(drule_tac x="injval (ders xs r) x v" in meta_spec)
   apply(simp)
   apply(simp add: bders_append)
   apply(subst bder_retrieve)
    apply(simp)
   apply(simp)
  thm bder_retrieve
  thm bmkeps_retrieve


lemma bmkeps_simp2:
  assumes "bnullable (bder c r)"
  shows "bmkeps (bder c (bsimp r)) = bmkeps (bder c r)"
  using  assms
  apply(induct r)
       apply(simp)
      apply(simp)
     apply(simp)
  prefer 3
    apply(simp)
   apply(simp)
   apply(auto)[1]
     prefer 2
     apply(case_tac "r1 = AZERO")
      apply(simp)
   apply(case_tac "r2 = AZERO")
      apply(simp)
     apply(case_tac "\<exists>bs. (bsimp r1) = AONE bs")
      apply(clarify)
      apply(simp)
      apply(subst bsimp_ASEQ2)
  
   apply(simp add: bmkeps_simp)
  apply(simp add: bders_append)
  apply(drule_tac x="bder a r" in meta_spec)
       apply(simp)
      apply(simp)
     apply(simp)
    prefer 3
    apply(simp)
   prefer 2
   apply(simp)
   apply(case_tac x2a)
    apply(simp)
  apply(simp add: )
  apply(subst k0)
   apply(auto)[1]
    apply(case_tac list)
     apply(simp)
  

   apply(case_tac  "r1=AZERO")
    apply(simp)
   apply(case_tac  "r2=AZERO")
    apply(simp)
    apply(auto)[1]
   apply(case_tac  "\<exists>bs. r1=AONE bs")
  apply(simp)
    apply(auto)[1]
     apply(subst bsimp_ASEQ2)

  
   prefer 2
   apply(simp)
  apply(subst  bmkeps_bder_AALTs)
   apply(case_tac x2a)
    apply(simp)
   apply(simp)
   apply(auto)[1]
    apply(subst  bmkeps_bder_AALTs)
  
   apply(case_tac a)
  apply(simp_all)
   apply(auto)[1]
    apply(case_tac list)
         apply(simp)
        apply(simp)
  
     prefer 2
  apply(simp)


lemma bbs0:
  shows "blexer_simp r [] = blexer r []"
  apply(simp add: blexer_def blexer_simp_def)
  done

lemma bbs1:
  shows "blexer_simp r [c] = blexer r [c]"
  apply(simp add: blexer_def blexer_simp_def)
  apply(auto)
    defer
  using b3 apply auto[1]
  using b3 apply auto[1]  
  apply(subst bmkeps_simp[symmetric])
   apply(simp)
  apply(simp)
  done
  
lemma bbs1:
  shows "blexer_simp r [c1, c2] = blexer r [c1, c2]"
  apply(simp add: blexer_def blexer_simp_def)
  apply(auto)
    defer
  apply (metis L_bsimp_erase bnullable_correctness der_correctness erase_bder lexer.simps(1) lexer_correct_None option.distinct(1))
  apply (metis L_bsimp_erase bnullable_correctness der_correctness erase_bder lexer.simps(1) lexer_correct_None option.distinct(1))
  apply(subst bmkeps_simp[symmetric])
  using b3 apply auto[1]
  apply(subst bmkeps_retrieve)
  using b3 bnullable_correctness apply blast
  apply(subst bder_retrieve)
  using b3 bnullable_correctness mkeps_nullable apply fastforce
  apply(subst bmkeps_retrieve)
  using bnullable_correctness apply blast
  apply(subst bder_retrieve)
  using bnullable_correctness mkeps_nullable apply force
  
  using bder_retrieve bmkeps_simp bmkeps_retrieve

  

lemma bsimp_retrieve_bder:
  assumes "\<Turnstile> v : der c (erase r)"
  shows "retrieve (bder c r) v = retrieve (bsimp (bder c r)) v"
  thm bder_retrieve bmkeps_simp
  apply(induct  r arbitrary: c v)
       apply(simp)
      apply(simp)
     apply(simp)
    apply(auto)[1]
     apply(case_tac "bsimp (bder c r1) = AZERO")
      apply(simp)
  
    prefer 3
    apply(simp)
  apply(auto elim!: Prf_elims)[1]
    apply(case_tac "(bsimp (fuse [Z] (bder c r))) = AZERO")
     apply(simp)
     apply (metis L_bsimp_erase L_flat_Prf1 L_flat_Prf2 Prf_elims(1) erase.simps(1) erase_bder erase_fuse)
    apply(case_tac "\<exists>bs. bsimp (fuse [Z] (bder c r)) = AONE bs")
     apply(clarify)
     apply(subgoal_tac "L (der c (erase r)) = {[]}")
      prefer 2
      apply (metis L.simps(2) L_bsimp_erase erase.simps(2) erase_bder erase_fuse)
     apply(simp)
  
  
  
    apply(subst bsimp_ASEQ1)
       apply(simp)
      apply(simp)
  apply(auto)[1]
  
     prefer 2
  

lemma oo:
  shows "(case (blexer (der c r) s) of None \<Rightarrow> None | Some v \<Rightarrow> Some (injval r c v)) = blexer r (c # s)"
  apply(simp add: blexer_correctness)
  done

lemma oo2a:
  assumes "\<forall>r. bmkeps (bders_simp r s) = bmkeps (bders r s)" "c # s \<in> L r"
          "bnullable (bders_simp (bsimp (bder c (intern r))) s)"
  shows "(case (blexer_simp (der c r) s) of None \<Rightarrow> None | Some v \<Rightarrow> Some (injval r c v)) = blexer_simp r (c # s)"
  using assms
  apply(simp add: blexer_simp_def)
  apply(auto  split: option.split)
    apply (metis blexer_correctness blexer_def lexer.simps(2) lexer_correct_None option.simps(4))
   prefer 2
  apply (metis L_bders_simp L_bsimp_erase Posix1(1) Posix_mkeps bnullable_correctness ders_correctness erase_bder erase_bders erase_intern lexer.simps(1) lexer_correct_None)
  apply(subst bmkeps_retrieve)
  using L_bders_simp bnullable_correctness nullable_correctness apply blast
  
  thm bder_retrieve
  
  
  apply(subst bder_retrieve[symmetric])
  
  

  apply(drule_tac x="bsimp (bder c (intern r))" in  spec)
  apply(drule sym)
   apply(simp)
   apply(subst blexer_simp_def)
  apply(case_tac "bnullable (bders_simp (intern (der c r)) s)")
   apply(simp)
  apply(auto split: option.split)
  apply(subst oo)
  apply(simp)
  done



lemma oo3:
  assumes "\<forall>r. bders r s = bders_simp r s"
  shows "blexer_simp r (s @ [c]) = blexer r (s @ [c])"
  using assms
  apply(simp (no_asm) add: blexer_simp_def)
  apply(auto)
   prefer 2
  apply (metis L_bders_simp blexer_def bnullable_correctness lexer.simps(1) lexer_correct_None option.distinct(1))
  apply(simp add: bders_simp_append)
  apply(subst bmkeps_simp[symmetric])
  using b3 apply auto[1]
  apply(simp add: blexer_def)
  apply(auto)[1]
   prefer 2
  apply (metis (mono_tags, lifting) L_bders_simp Posix_mkeps append.right_neutral bders_simp.simps(1) bders_simp.simps(2) bders_simp_append bnullable_correctness lexer.simps(1) lexer_correct_None lexer_correctness(1) option.distinct(1))
  apply(simp add: bders_append)     
  done

lemma oo4:
  assumes "\<forall>r. bmkeps (bders r s) = bmkeps (bders_simp r s)" "bnullable (bder c (bders r s))"
  shows "bmkeps (bders_simp r (s @ [c])) = bmkeps (bders r (s @ [c]))"
  using assms
  apply(simp add: bders_simp_append)
  apply(subst bmkeps_simp[symmetric])
  apply (metis L_bders_simp bnullable_correctness der_correctness erase_bder lexer.simps(1) lexer_correct_None option.distinct(1))
  apply(simp add: bders_append)
  done

lemma oo4:
  shows "blexer_simp r s = blexer r s"
  apply(induct s arbitrary: r rule: rev_induct)
   apply(simp only: blexer_simp_def blexer_def)
   apply(simp)
  apply(simp only: blexer_simp_def blexer_def)
  apply(subgoal_tac "bnullable (bders_simp (intern r) (xs @ [x])) = bnullable (bders (intern r) (xs @ [x]))")
   prefer 2
   apply (simp add: b4)
  apply(simp)
  apply(rule impI)
  apply(simp add: bders_simp_append)
  apply(subst bmkeps_simp[symmetric])
  using b3 apply auto[1]
  apply(subst bmkeps_retrieve)
  using b3 bnullable_correctness apply blast
  apply(subst bder_retrieve)
  using b3 bnullable_correctness mkeps_nullable apply fastforce
  apply(simp add: bders_append)
  apply(subst bmkeps_retrieve)
  using bnullable_correctness apply blast
  apply(subst bder_retrieve)
  using bnullable_correctness mkeps_nullable apply fastforce
  apply(subst erase_bder)
  apply(case_tac "xs \<in> L")
  apply(subst (asm) (2) bmkeps_retrieve)
  
  
  thm bmkeps_retrieve bmkeps_retrieve
  apply(subst bmkeps_retrieve[symmetric])
  
   apply (simp add: bnullable_correctness)
  apply(simp add: bders_simp_append)
   
  
  apply(induct s arbitrary: r rule: rev_induct) 
   apply(simp add: blexer_def blexer_simp_def)
  apply(rule oo3)
  apply(simp (no_asm) add: blexer_simp_def)
  apply(auto)
   prefer 2
  apply (metis L_bders_simp blexer_def bnullable_correctness lexer.simps(1) lexer_correct_None option.distinct(1))
  apply(simp add: bders_simp_append)
  apply(subst bmkeps_simp[symmetric])
  using b3 apply auto[1]
  apply(simp add: blexer_def)
  apply(auto)[1]
   prefer 2
  apply (m etis (mono_tags, lifting) L_bders_simp Posix_mkeps append.right_neutral bders_simp.simps(1) bders_simp.simps(2) bders_simp_append bnullable_correctness lexer.simps(1) lexer_correct_None lexer_correctness(1) option.distinct(1))
  apply(simp add: bders_append)     
  oops


lemma bnullable_simp:
  assumes "s \<in> L (erase r)"
  shows "bmkeps (bders r s) = bmkeps (bders_simp r s)"
  using assms
  apply(induct s arbitrary: r rule: rev_induct)
   apply(simp)
  apply(simp add: bders_append bders_simp_append)
  apply(subst bmkeps_simp[symmetric])
  apply (metis L_bders_simp b3 bders_simp.simps(1) bders_simp.simps(2) bders_simp_append blexer_correctness blexer_def bnullable_correctness erase_bders erase_intern lexer.simps(1) lexer_correct_None lexer_correct_Some lexer_correctness(1))
  apply(subst bmkeps_retrieve)
  apply (metis bders.simps(1) bders.simps(2) bders_append blexer_correctness blexer_def bnullable_correctness erase_bders erase_intern lexer_correct_Some option.distinct(1))
  apply(subst bmkeps_retrieve)
  apply (metis L_bders_simp L_bsimp_erase Posix_mkeps bders_simp.simps(1) bders_simp.simps(2) bders_simp_append blexer_correctness blexer_def bnullable_correctness erase_bders erase_intern lexer.simps(1) lexer_correct_None lexer_correctness(2))
  apply(subst bder_retrieve)
  apply (metis bders.simps(1) bders.simps(2) bders_append blexer_correctness blexer_def bnullable_correctness erase_bder erase_bders erase_intern lexer_correct_Some mkeps_nullable option.distinct(1))
  apply(subst bder_retrieve)
  apply (metis L_bders_simp L_bsimp_erase Posix_mkeps bders_simp.simps(1) bders_simp.simps(2) bders_simp_append blexer_correctness blexer_def bnullable_correctness erase_bder erase_bders erase_intern lexer.simps(1) lexer_correct_None lexer_correctness(2) mkeps_nullable)
    
  apply(drule_tac x="bder a r" in meta_spec)
  apply(drule_tac  meta_mp)
  apply (me tis erase_bder lexer.simps(2) lexer_correct_None option.simps(4))
  apply(simp)
  oops


lemma
  shows "blexer r s = blexer_simp r s"
  apply(induct s arbitrary: r rule: rev_induct)
   apply(simp add: blexer_def blexer_simp_def)
  apply(case_tac "xs @ [x] \<in> L r")
   defer
   apply(subgoal_tac "blexer (ders xs r) [x] = None")
    prefer 2
    apply(subst blexer_correctness)
    apply(simp (no_asm) add: lexer_correct_None)
    apply(simp add: nullable_correctness)
    apply(simp add: der_correctness ders_correctness)
  apply(simp add: Der_def Ders_def)
apply(subgoal_tac "blexer r (xs @ [x]) = None")
    prefer 2
    apply (simp add: blexer_correctness)
  using lexer_correct_None apply auto[1]
  apply(simp)
   apply(subgoal_tac "blexer_simp (ders xs r) [x] = None")
    prefer 2
  apply (metis L_bders_simp Posix_injval Posix_mkeps bders.simps(2) blexer_correctness blexer_simp_def bnullable_correctness ders.simps(1) erase_bder erase_bders erase_intern lexer_correct_None lexer_correctness(2))
   apply(subgoal_tac "[] \<notin> L (erase (bders_simp (intern r) (xs @ [x])))")
  prefer 2
  apply(metis L_bders_simp Posix_injval bders.simps(2) blexer_correctness ders.simps(1) ders_append erase_bder erase_bders erase_intern lexer_correct_None lexer_correctness(2))
  using blexer_simp_def bnullable_correctness lexer_correct_None apply auto[1]
(* case xs @ [x] \<in> L r *)
  apply(subgoal_tac "\<exists>v. blexer (ders xs r) [x] = Some v \<and> [x] \<in> (ders xs r) \<rightarrow> v")
   prefer 2  
  using blexer_correctness lexer_correct_Some apply auto[1]
    apply (simp add: Posix_injval Posix_mkeps)
  apply (metis ders.simps(1) ders.simps(2) ders_append lexer_correct_None lexer_flex)    
  apply(clarify)
  apply(subgoal_tac "blexer_simp (ders xs r) [x] = Some v")
   prefer 2
   apply(simp add: blexer_simp_def)
   apply(auto)[1]
    apply (metis bders.simps(1) bders.simps(2) blexer_def bmkeps_simp option.simps(3))
  using b3 blexer_def apply fastforce
  apply(subgoal_tac "blexer_simp (ders xs r) [x] = blexer_simp r (xs @ [x])")
   prefer 2
   apply(simp add: blexer_simp_def)
  
   apply(simp)
  
  
  
   apply(simp)
  apply(subst blexer_simp_def)
  apply(simp)
  apply(auto)
  apply(drule_tac x="der a r" in meta_spec)
  apply(subst blexer_def)
   apply(subgoal_tac "bnullable (bders (intern r) (a # s))")
    prefer 2
    apply (metis Posix_injval blexer_correctness blexer_def lexer_correctness(2))
  apply(simp)
  


lemma
  shows "blexer r s = blexer_simp r s"
  apply(induct s arbitrary: r)
   apply(simp add: blexer_def blexer_simp_def)
  apply(case_tac "s \<in> L (der a r)")
   defer
   apply(subgoal_tac "blexer (der a r) s = None")
    prefer 2
    apply (simp add: blexer_correctness lexer_correct_None)
apply(subgoal_tac "blexer r (a # s) = None")
    prefer 2
    apply (simp add: blexer_correctness)
   apply(simp)
  
   apply(subst blexer_simp_def)
   apply(simp)
  apply(drule_tac  x="der a r" in  meta_spec)
   apply(subgoal_tac "s \<notin> L(erase (bder a (intern  r)))")
  prefer 2
    apply simp
   
   apply(simp only:)
   apply(subst blexer_simp_def)
   apply(subgoal_tac "\<not> bnullable (bders_simp (intern r) (a # s))")
    apply(simp)
   apply(subst bnullable_correctness[symmetric])
  apply(simp)
  


end