
theory Antiquotes
imports "../Base"
begin


section {* Useful Document Antiquotations *}

text {*
  {\bf Problem:} 
  How to keep your ML-code inside a document synchronised with the actual code?\smallskip

  {\bf Solution:} This can be achieved using document antiquotations.\smallskip

  Document antiquotations can be used for ensuring consistent type-setting of
  various entities in a document. They can also be used for sophisticated
  \LaTeX-hacking.

  Below we give the code for two antiquotations that can be used to typeset
  ML-code and also to check whether the given code actually compiles. This
  provides a sanity check for the code and also allows one to keep documents
  in sync with other code, for example Isabelle.

  We first describe the antiquotation @{text "@{ML_checked \"expr\"}"}. This
  antiquotation takes a piece of code as argument. The code is checked
  by sending the ML-expression @{text [quotes] "val _ = \<dots>"} containing the
  given argument to the ML-compiler (i.e.~the function @{ML "ML_Context.eval_in"}
  in Line 4 below). The complete code of the antiquotation is as follows:

*}

ML %linenumbers {*fun ml_val code_txt = "val _ = " ^ code_txt

fun output_ml src ctxt code_txt =
  (ML_Context.eval_in (SOME ctxt) false Position.none (ml_val code_txt); 
  ThyOutput.output_list (fn _ => fn s => Pretty.str s) src ctxt 
                                            (space_explode "\n" code_txt))

val _ = ThyOutput.add_commands
 [("ML_checked", ThyOutput.args (Scan.lift Args.name) output_ml)]
*}

text {*

  Note that the parser @{ML "(Scan.lift Args.name)"} in line 9 parses a string, 
  in this case the code given as argument. As mentioned before, this argument 
  is send to the ML-compiler in the line 4 using the function @{ML ml_val},
  which constructs an apropriate ML-expression.
  If the code is ``approved'' by the compiler, then the output function @{ML
  "ThyOutput.output_list (fn _ => fn s => Pretty.str s)"} in the next line pretty prints the
  code. This function expects that the code is a list of strings where each
  string correspond to a line in the output. Therefore the use of 
  @{ML "(space_explode \"\\n\" txt)" for txt} 
  which produces this list according to linebreaks.  There are a number of options for antiquotations
  that are observed by @{ML ThyOutput.output_list} when printing the code (for
  example @{text "[display]"}, @{text "[quotes]"} and @{text "[source]"}).

  \begin{readmore}
  For more information about options of antiquotations see \rsccite{sec:antiq}).
  \end{readmore}

  Since we used the argument @{ML "Position.none"}, the compiler cannot give specific 
  information about the line number, in case an error is detected. We 
  can improve the code above slightly by writing 

*}

ML %linenumbers {* fun output_ml src ctxt (code_txt,pos) =
  (ML_Context.eval_in (SOME ctxt) false pos (ml_val code_txt);
  ThyOutput.output_list (fn _ => fn s => Pretty.str s) src ctxt 
                                            (space_explode "\n" code_txt))

val _ = ThyOutput.add_commands
 [("ML_checked", ThyOutput.args 
       (Scan.lift (OuterParse.position Args.name)) output_ml)]
*}

text {*
  where in Lines 1 and 2 the positional information is properly treated.

  (FIXME: say something about OuterParse.position)

  We can now write in a document @{text "@{ML_checked \"2 + 3\"}"} in order to
  obtain @{ML_checked "2 + 3"} and be sure that this code compiles until
  somebody changes the definition of \mbox{@{ML "(op +)"}}.


  The second antiquotation extends the first by allowing also to give
  hints what the result of the ML-code should be and to check the consistency of 
  the actual result with these hints. For this we are going to implement the 
  antiquotation  
  @{text "@{ML_response \"expr\" \"pat\"}"}
  whose first argument is the ML-code and the second is a pattern specifying
  the result. To add some convenience and allow dealing with large outputs,
  the user can give a partial specification including abbreviations 
  @{text [quotes] "\<dots>"}.

  Whereas in the antiquotation @{text "@{ML_checked \"expr\"}"} above, we have
  sent the expression 
  @{text [quotes] "val _ = expr"} to the compiler, in this antiquotation the wildcard
  @{text "_"} we will be replaced by a proper pattern constructed the hints. To
  do this we need to replace the @{text [quotes] "\<dots>"} by @{text [quotes] "_"} 
  before sending the code to the compiler. The following function will do this:

*}

ML {* 
fun ml_pat (code_txt, pat) =
   let val pat' = 
         implode (map (fn "\<dots>" => "_" | s => s) (Symbol.explode pat))
   in 
     "val " ^ pat' ^ " = " ^ code_txt 
   end
*}

text {* 
  Next we like to add a response indicator to the result using:
*}


ML {*
fun add_response_indicator pat =
  map (fn s => "> " ^ s) (space_explode "\n" pat)
*}

text {* 
  The rest of the code of the antiquotation is
  *}

ML {*
fun output_ml_response src ctxt ((code_txt,pat),pos) = 
  (ML_Context.eval_in (SOME ctxt) false pos (ml_pat (code_txt,pat));
   let 
     val output = (space_explode "\n" code_txt) @ (add_response_indicator pat)
   in 
     ThyOutput.output_list (fn _ => fn s => Pretty.str s) src ctxt output 
   end)

val _ = ThyOutput.add_commands
 [("ML_response", 
     ThyOutput.args 
      (Scan.lift (OuterParse.position (Args.name -- Args.name))) 
        output_ml_response)]
*}

text {*
  This extended antiquotation allows us to write 
  @{text [display] "@{ML_response [display] \"true andalso false\" \"false\"}"}
  to obtain

@{ML_response [display] "true andalso false" "false"} 

  or 

@{text [display] "@{ML_response [display] \"let val i = 3 in (i * i,\"foo\") end\" \"(9,\<dots>)\"}"}
  
  to obtain

@{ML_response [display] "let val i = 3 in (i * i,\"foo\") end" "(9,\<dots>)"} 

  In both cases, the check by the compiler ensures that code and result match. A limitation
  of this antiquotation is that the hints can only be given for results that can actually
  be constructed as a pattern. This excludes values that are abstract datatypes, like 
  theorems or cterms.

*}


end
  



