
theory Antiquotes
imports "../Appendix"
begin

section \<open>Useful Document Antiquotations\label{rec:docantiquotations}\<close>

text \<open>
  {\bf Problem:} 
  How to keep your ML-code inside a document synchronised with the actual code?\smallskip

  {\bf Solution:} This can be achieved with document antiquotations.\smallskip

  Document antiquotations can be used for ensuring consistent type-setting of
  various entities in a document. They can also be used for sophisticated
  \LaTeX-hacking. If you type on the Isabelle level
\<close>

print_antiquotations

text \<open>
  you obtain a list of all currently available document antiquotations and
  their options.  

  Below we will give the code for two additional document
  antiquotations both of which are intended to typeset ML-code. The crucial point
  of these document antiquotations is that they not just print the ML-code, but also 
  check whether it compiles. This will provide a sanity check for the code
  and also allows you to keep documents in sync with other code, for example
  Isabelle.

  We first describe the antiquotation \<open>ML_checked\<close> with the syntax:
 
  @{text [display] \<open>@{ML_checked "a_piece_of_code"}\<close>}

  The code is checked by sending the ML-expression @{text [quotes] "val _ =
  a_piece_of_code"} to the ML-compiler (i.e.~the function @{ML \<open>ML_Context.eval_source_in\<close>} in Line 7 below). The complete code of the
  document antiquotation is as follows:

\<close>
ML \<open>Input.pos_of\<close>
ML%linenosgray\<open>fun ml_enclose bg en source =
  ML_Lex.read bg @ ML_Lex.read_source source @ ML_Lex.read en;\<close>

ML%linenosgray\<open>fun ml_val code_txt = (ml_enclose "val _ = " "" code_txt)

fun output_ml ctxt code_txt =
let
  val _ = ML_Context.eval_in (SOME ctxt) ML_Compiler.flags (Input.pos_of code_txt) (ml_val code_txt)
in 
   Pretty.str (fst (Input.source_content code_txt))
end

val ml_checked_setup = Thy_Output.antiquotation_pretty_source @{binding "ML_checked"} (Scan.lift Args.text_input) output_ml\<close>

setup \<open>ml_checked_setup\<close>


text \<open>
  The parser @{ML \<open>(Scan.lift Args.name)\<close>} in Line 7 parses a string, in this
  case the code, and then calls the function @{ML output_ml}. As mentioned
  before, the parsed code is sent to the ML-compiler in Line 4 using the
  function @{ML ml_val}, which constructs the appropriate ML-expression, and
  using @{ML \<open>eval_in\<close> in ML_Context}, which calls the compiler.  If the code is
  ``approved'' by the compiler, then the output function @{ML \<open>output\<close> in
  Document_Antiquotation} in the next line pretty prints the code. This function expects
  that the code is a list of (pretty)strings where each string correspond to a
  line in the output. Therefore the use of @{ML \<open>(space_explode "\\n" txt)\<close>
  for txt} which produces such a list according to linebreaks.  There are a
  number of options for antiquotations that are observed by the function 
  @{ML \<open>output\<close> in Document_Antiquotation} when printing the code (including \<open>[display]\<close> 
  and \<open>[quotes]\<close>). The function @{ML \<open>antiquotation_raw\<close> in Thy_Output} in 
  Line 7 sets up the new document antiquotation.

  \begin{readmore}
  For more information about options of document antiquotations see \rsccite{sec:antiq}).
  \end{readmore}

  Since we used the argument @{ML \<open>Position.none\<close>}, the compiler cannot give specific 
  information about the line number, in case an error is detected. We 
  can improve the code above slightly by writing 
\<close>
(* FIXME: remove
ML%linenosgray{*fun output_ml ctxt (code_txt, pos) =
let
  val srcpos = {delimited = false, pos = pos, text = ml_val code_txt}
in
  (ML_Context.eval_source_in (SOME ctxt) ML_Compiler.flags srcpos;
   code_txt 
   |> space_explode "\n"
   |> map Pretty.str
   |> Pretty.list "" ""
   |> Document_Antiquotation.output ctxt
   |> Latex.string)
end

val ml_checked_setup2 = Thy_Output.antiquotation @{binding "ML_checked2"}
         (Scan.lift (Parse.position Args.name)) output_ml *}

setup {* ml_checked_setup2 *}
*)
text \<open>
  where in Lines 1 and 2 the positional information is properly treated. The
  parser @{ML Parse.position} encodes the positional information in the 
  result.

  We can now write \<open>@{ML_checked2 "2 + 3"}\<close> in a document in order to
  obtain @{ML_checked "2 + 3"} and be sure that this code compiles until
  somebody changes the definition of addition.


  The second document antiquotation we describe extends the first by a pattern
  that specifies what the result of the ML-code should be and checks the
  consistency of the actual result with the given pattern. For this we are
  going to implement the document antiquotation:

  
  @{text [display] \<open>@{ML_resp "a_piece_of_code" "a_pattern"}\<close>}
  
  To add some convenience and also to deal with large outputs, the user can
  give a partial specification by using ellipses. For example \<open>(\<dots>, \<dots>)\<close>
  for specifying a pair.  In order to check consistency between the pattern
  and the output of the code, we have to change the ML-expression that is sent 
  to the compiler: in \<open>ML_checked2\<close> we sent the expression @{text [quotes]
  "val _ = a_piece_of_code"} to the compiler; now the wildcard \<open>_\<close>
  must be be replaced by the given pattern. However, we have to remove all
  ellipses from it and replace them by @{text [quotes] "_"}. The following 
  function will do this:
\<close>

ML%linenosgray\<open>fun ml_pat pat code =
  ML_Lex.read "val" @ ML_Lex.read_source pat @ ML_Lex.read " = " @ ML_Lex.read_source code\<close>

(*
ML %grayML{*fun ml_pat code_txt pat =
let val pat' = 
         implode (map (fn "\<dots>" => "_" | s => s) (Symbol.explode pat))
in 
  ml_enclose ("val " ^ pat' ^ " = ") "" code_txt 
end*}
*)
text \<open>
  Next we add a response indicator to the result using:
\<close>


ML %grayML\<open>fun add_resp pat = map (fn s => "> " ^ s) pat\<close>

text \<open>
  The rest of the code of \<open>ML_resp\<close> is: 
\<close>

ML %linenosgray\<open>
fun output_ml_resp ctxt (code_txt, pat) =
let
  val _ = ML_Context.eval_in (SOME ctxt) ML_Compiler.flags (Input.pos_of code_txt) (ml_pat pat code_txt)
  val code = space_explode "\n" (fst (Input.source_content code_txt))
  val resp = add_resp (space_explode "\n" (fst (Input.source_content pat)))
in 
   Pretty.str (cat_lines (code @ resp))
end

val ml_response_setup = Thy_Output.antiquotation_pretty_source @{binding "ML_resp"} (Scan.lift (Args.text_input -- Args.text_input)) output_ml_resp

\<close>

(*
ML %linenosgray{*fun output_ml_resp {context = ctxt, ...} ((code_txt, pat), pos) = 
  (let
     val srcpos = {delimited = false, text = ml_pat (code_txt, pat), pos = pos}
   in
     ML_Context.eval_source_in (SOME ctxt) ML_Compiler.flags srcpos 
   end;
   let 
     val code_output = space_explode "\n" code_txt 
     val resp_output = add_resp (space_explode "\n" pat)
   in 
     Thy_Output.output ctxt (map Pretty.str (code_output @ resp_output)) 
   end)


val ml_resp_setup = Thy_Output.antiquotation @{binding "ML_resp"} 
          (Scan.lift (Parse.position (Args.text_input -- Args.text_input))) 
             output_ml_resp*}
*)
setup \<open>ml_response_setup\<close>

(* FIXME *)
text \<open>
  In comparison with \<open>ML_checked\<close>, we only changed the line about 
  the compiler (Line~2), the lines about
  the output (Lines 4 to 7) and the parser in the setup (Line 11). Now 
  you can write
 
  @{text [display] \<open>@{ML_resp [display] "true andalso false" "false"}\<close>}

  to obtain

  @{ML_resp [display] "true andalso false" "false"} 

  or 

  @{text [display] \<open>@{ML_resp [display] "let val i = 3 in (i * i, "foo") end" "(9, \<dots>)"}\<close>}
  
  to obtain

  @{ML_resp [display] "let val i = 3 in (i * i, \"foo\") end" "(9, _)"} 

  In both cases, the check by the compiler ensures that code and result
  match. A limitation of this document antiquotation, however, is that the
  pattern can only be given for values that can be constructed. This excludes
  values that are abstract datatypes, like @{ML_type thm}s and @{ML_type cterm}s.

\<close>
end
