abstract class Rexp 
case object ZERO extends Rexp
case object ONE extends Rexp
case class CHAR(c: Char) extends Rexp
case class ALT(r1: Rexp, r2: Rexp) extends Rexp
case class SEQ(r1: Rexp, r2: Rexp) extends Rexp 
case class STAR(r: Rexp) extends Rexp 
case class NTIMES(r: Rexp, n: Int) extends Rexp 

def simp(r: Rexp): Rexp = r match {
  case ALT(r1, r2) => {
    (simp(r1), simp(r2)) match {
      case (ZERO, r2s) => r2s
      case (r1s, ZERO) => r1s
      case (r1s, r2s) => if (r1s == r2s) r1s else ALT(r1s, r2s)
    }
  }
  case SEQ(r1, r2) => {
    (simp(r1), simp(r2)) match {
      case (ZERO, _) => ZERO
      case (_, ZERO) => ZERO
      case (ONE, r2s) => r2s
      case (r1s, ONE) => r1s
      case (r1s, r2s) => SEQ(r1s, r2s)
    }
  }
  case NTIMES(r, n) => NTIMES(simp(r), n)    
  case r => r
}

def nullable (r: Rexp) : Boolean = r match {
  case ZERO => false
  case ONE => true
  case CHAR(_) => false
  case ALT(r1, r2) => nullable(r1) || nullable(r2)
  case SEQ(r1, r2) => nullable(r1) && nullable(r2)
  case STAR(_) => true
  case NTIMES(r, i) => if (i == 0) true else nullable(r)
}

def der (c: Char, r: Rexp) : Rexp = r match {
  case ZERO => ZERO
  case ONE => ZERO
  case CHAR(d) => if (c == d) ONE else ZERO
  case ALT(r1, r2) => ALT(der(c, r1), der(c, r2))
  case SEQ(r1, r2) => 
    if (nullable(r1)) ALT(SEQ(der(c, r1), r2), der(c, r2))
    else SEQ(der(c, r1), r2)
  case STAR(r) => SEQ(der(c, r), STAR(r))
  case NTIMES(r, i) => 
    if (i == 0) ZERO else SEQ(der(c, r), NTIMES(r, i - 1))
}

def ders (s: List[Char], r: Rexp) : Rexp = s match {
  case Nil => r
  case c::s => ders(s, simp(der(c, r)))
}

def matches(r: Rexp, s: String) : Boolean = nullable(ders(s.toList, r))


//one or zero
def OPT(r: Rexp) = ALT(r, ONE)

def EVIL(n: Int) = SEQ(NTIMES(OPT(CHAR('a')), n), NTIMES(CHAR('a'), n))

def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  (end - start)/(i * 1.0e9)
}

/*
for (i <- 1 to 9001 by 1000) {
  println(i + " " + "%.5f".format(time_needed(1, matches(EVIL(i), "a" * i))))
}
*/

// some convenience for typing in regular expressions
import scala.language.implicitConversions    
import scala.language.reflectiveCalls 

def charlist2rexp(s : List[Char]) : Rexp = s match {
  case Nil => ONE
  case c::Nil => CHAR(c)
  case c::s => SEQ(CHAR(c), charlist2rexp(s))
}
implicit def string2rexp(s : String) : Rexp = charlist2rexp(s.toList)

implicit def RexpOps (r: Rexp) = new {
  def | (s: Rexp) = ALT(r, s)
  def % = STAR(r)
  def ~ (s: Rexp) = SEQ(r, s)
}

implicit def stringOps (s: String) = new {
  def | (r: Rexp) = ALT(s, r)
  def | (r: String) = ALT(s, r)
  def % = STAR(s)
  def ~ (r: Rexp) = SEQ(s, r)
  def ~ (r: String) = SEQ(s, r)
}


def PLUS(r: Rexp) = SEQ(r, STAR(r))
def RANGE(s: List[Char]) : Rexp = s match {
  case Nil => ZERO
  case c::s => ALT(CHAR(c), RANGE(s))
} 
def NMTIMES(r: Rexp, n: Int, m: Int) : Rexp =
  if (n >= m) NTIMES(r, n) else ALT(NTIMES(r, m), NMTIMES(r, n, m - 1))


println("Coursework:")
val REG1 = PLUS(PLUS("a" ~ "a" ~ "a"))
val REG2 = PLUS(PLUS("aaaaaaaaaaaaaaaaaaa" ~ OPT("a")))  // 19 as ~ a?


//40 * aaa matches
//43 * aaa + aa does not
//45 * aaa + a

println("EVIL1:")
println(matches(REG1, "aaa" * 40))
println(matches(REG1, "aaa" * 43 + "aa"))
println(matches(REG1, "aaa" * 45 + "a"))
println("EVIL2:")
println(matches(REG2, "aaa" * 40))
println(matches(REG2, "aaa" * 43 + "aa"))
println(matches(REG2, "aaa" * 45 + "a"))

println("EMAIL:")
val LOWERCASE = "abcdefghijklmnopqrstuvwxyz"
val DIGITS = "0123456789"
val EMAIL = PLUS(RANGE((LOWERCASE + DIGITS + "_" + "." + "-").toList)) ~ "@" ~ 
            PLUS(RANGE((LOWERCASE + DIGITS + "." + "-").toList)) ~ "." ~
            NMTIMES(RANGE((LOWERCASE + ".").toList), 2, 6)

val my_email = "christian.urban@kcl.ac.uk"

println(matches(EMAIL, my_email))

println(matches(NTIMES("a", 2), "a"))
println(matches(NTIMES("a", 2), "aa"))
println(matches(NTIMES("a", 2), "aaa"))

println(matches(NMTIMES("a", 2, 3), "a"))
println(matches(NMTIMES("a", 2, 3), "aa"))
println(matches(NMTIMES("a", 2, 3), "aaa"))
println(matches(NMTIMES("a", 2, 3), "aaaa"))
