package object matcher {

// regular expressions 
// including constructors for NOT and ALLC
sealed abstract class Rexp

case object NULL extends Rexp
case object EMPTY extends Rexp
case object ALLC extends Rexp            // recognises any character
case class CHAR(c: Char) extends Rexp
case class ALT(r1: Rexp, r2: Rexp) extends Rexp
case class SEQ(r1: Rexp, r2: Rexp) extends Rexp
case class STAR(r: Rexp) extends Rexp
case class NOT(r: Rexp) extends Rexp     // negation of a regular expression


// nullable function: tests whether the regular 
// expression can recognise the empty string
def nullable (r: Rexp) : Boolean = r match {
  case NULL => false
  case EMPTY => true
  case ALLC => false
  case CHAR(_) => false
  case ALT(r1, r2) => nullable(r1) || nullable(r2)
  case SEQ(r1, r2) => nullable(r1) && nullable(r2)
  case STAR(_) => true
  case NOT(r) => !(nullable(r))
}

// tests whether a regular expression 
// cannot recognise more
def no_more (r: Rexp) : Boolean = r match {
  case NULL => true
  case EMPTY => false
  case ALLC => false
  case CHAR(_) => false
  case ALT(r1, r2) => no_more(r1) && no_more(r2)
  case SEQ(r1, r2) => if (nullable(r1)) (no_more(r1) && no_more(r2)) else no_more(r1)
  case STAR(_) => false
  case NOT(r) => !(no_more(r))
}


// derivative of a regular expression w.r.t. a character
def der (c: Char, r: Rexp) : Rexp = r match {
  case NULL => NULL
  case EMPTY => NULL  
  case ALLC => EMPTY
  case CHAR(d) => if (c == d) EMPTY else NULL
  case ALT(r1, r2) => ALT(der(c, r1), der(c, r2))
  case SEQ(r1, r2) => 
    if (nullable(r1)) ALT(SEQ(der(c, r1), r2), der(c, r2))
    else SEQ(der(c, r1), r2)
  case STAR(r) => SEQ(der(c, r), STAR(r))
  case NOT(r) => NOT(der (c, r))
}

// main class for the tokenizer
case class Tokenizer[T](rules: List[(Rexp, List[Char] => T)], excl: List[T] = Nil) {

def munch(r: Rexp, action: List[Char] => T, s: List[Char], t: List[Char]) : Option[(List[Char], T)] = 
  s match {
    case Nil if (nullable(r)) => Some(Nil, action(t))
    case Nil => None
    case c::s if (no_more(der (c, r)) && nullable(r)) => Some(c::s, action(t))
    case c::s if (no_more(der (c, r))) => None
    case c::s => munch(der (c, r), action, s, t ::: List(c))
  }

def one_token(s: List[Char]) : Either[(List[Char], T), String] = {
  val somes = rules.map { (r) => munch(r._1, r._2, s, Nil) }.flatten
  if (somes == Nil) Right(s.mkString) 
  else Left(somes sortBy (_._1.length) head)
}

def tokenize(cs: List[Char]) : List[T] = cs match {
  case Nil => Nil
  case _ => one_token(cs) match {
    case Left((rest, token)) => token :: tokenize(rest)
    case Right(s) => { println("Cannot tokenize: \"" + s + "\""); Nil } 
  }
}

def fromString(s: String) : List[T] = 
  tokenize(s.toList).filterNot(excl.contains(_))

def fromFile(name: String) : List[T] = 
  fromString(io.Source.fromFile(name).mkString)

}


// regular expression for specifying 
// ranges of characters
def Range(s : List[Char]) : Rexp = s match {
  case Nil => NULL
  case c::Nil => CHAR(c)
  case c::s => ALT(CHAR(c), Range(s))
}
def RANGE(s: String) = Range(s.toList)


// one or more
def PLUS(r: Rexp) = SEQ(r, STAR(r))

// many alternatives
def Alts(rs: List[Rexp]) : Rexp = rs match {
  case Nil => NULL
  case r::Nil => r
  case r::rs => ALT(r, Alts(rs))
}
def ALTS(rs: Rexp*) = Alts(rs.toList)

// repetitions
def Seqs(rs: List[Rexp]) : Rexp = rs match {
  case Nil => NULL
  case r::Nil => r
  case r::rs => SEQ(r, Seqs(rs))
}
def SEQS(rs: Rexp*) = Seqs(rs.toList)

// some convenience for typing in regular expressions
def charlist2rexp(s : List[Char]) : Rexp = s match {
  case Nil => EMPTY
  case c::Nil => CHAR(c)
  case c::s => SEQ(CHAR(c), charlist2rexp(s))
}
implicit def string2rexp(s : String) : Rexp = charlist2rexp(s.toList)

}
