import scala.language.implicitConversions    
import scala.language.reflectiveCalls 
import scala.util._
import scala.annotation.tailrec
import scala.sys.process._

def fromFile(name: String) : String = 
  io.Source.fromFile(name).mkString

abstract class Rexp 
case object NULL extends Rexp
case object EMPTY extends Rexp
case class CHAR(c: Char) extends Rexp
case class ALT(r1: Rexp, r2: Rexp) extends Rexp 
case class RANGE(cs: List[Char]) extends Rexp 
case class SEQ(r1: Rexp, r2: Rexp) extends Rexp 
case class PLUS(r: Rexp) extends Rexp 
case class STAR(r: Rexp) extends Rexp 
case class NTIMES(r: Rexp, n: Int) extends Rexp 
case class NUPTOM(r: Rexp, n: Int, m: Int) extends Rexp

object RANGE {
  def apply(s: String) : RANGE = RANGE(s.toList)
}
def NMTIMES(r: Rexp, n: Int, m: Int) = {
  if (m < n) throw new IllegalArgumentException("the number m cannot be smaller than n.")
  else NUPTOM(r, n, m - n)
}

case class NOT(r: Rexp) extends Rexp 
case class OPT(r: Rexp) extends Rexp 

// some convenience for typing in regular expressions
def charlist2rexp(s : List[Char]) : Rexp = s match {
  case Nil => EMPTY
  case c::Nil => CHAR(c)
  case c::s => SEQ(CHAR(c), charlist2rexp(s))
}
implicit def string2rexp(s : String) : Rexp = charlist2rexp(s.toList)

implicit def RexpOps (r: Rexp) = new {
  def | (s: Rexp) = ALT(r, s)
  def % = STAR(r)
  def ~ (s: Rexp) = SEQ(r, s)
}

implicit def stringOps (s: String) = new {
  def | (r: Rexp) = ALT(s, r)
  def | (r: String) = ALT(s, r)
  def % = STAR(s)
  def ~ (r: Rexp) = SEQ(s, r)
  def ~ (r: String) = SEQ(s, r)
}


// nullable function: tests whether the regular 
// expression can recognise the empty string
def nullable (r: Rexp) : Boolean = r match {
  case NULL => false
  case EMPTY => true
  case CHAR(_) => false
  case ALT(r1, r2) => nullable(r1) || nullable(r2)
  case SEQ(r1, r2) => nullable(r1) && nullable(r2)
  case STAR(_) => true
  case PLUS(r) => nullable(r)
  case NTIMES(r, i) => if (i == 0) true else nullable(r)
  case NUPTOM(r, i, j) => if (i == 0) true else nullable(r)
  case RANGE(_) => false
  case NOT(r) => !(nullable(r))
  case OPT(_) => true
}

// derivative of a regular expression w.r.t. a character
def der (c: Char, r: Rexp) : Rexp = r match {
  case NULL => NULL
  case EMPTY => NULL
  case CHAR(d) => if (c == d) EMPTY else NULL
  case ALT(r1, r2) => ALT(der(c, r1), der(c, r2))
  case SEQ(r1, r2) => 
    if (nullable(r1)) ALT(SEQ(der(c, r1), r2), der(c, r2))
    else SEQ(der(c, r1), r2)
  case STAR(r) => SEQ(der(c, r), STAR(r))
  case PLUS(r) => SEQ(der(c, r), STAR(r))
  case NTIMES(r, i) => 
    if (i == 0) NULL else der(c, SEQ(r, NTIMES(r, i - 1)))
  case NUPTOM(r, i, j) =>
    if (i == 0 && j == 0) NULL else 
    if (i == 0) ALT(der(c, NTIMES(r, j)), der(c, NUPTOM(r, 0, j - 1)))
    else der(c, SEQ(r, NUPTOM(r, i - 1, j)))
  case RANGE(cs) => if (cs contains c) EMPTY else NULL
  case NOT(r) => NOT(der (c, r))
  case OPT(r) => der(c, r)
}

def zeroable (r: Rexp) : Boolean = r match {
  case NULL => true
  case EMPTY => false
  case CHAR(_) => false
  case ALT(r1, r2) => zeroable(r1) && zeroable(r2)
  case SEQ(r1, r2) => zeroable(r1) || zeroable(r2)
  case STAR(_) => false
  case PLUS(r) => zeroable(r)
  case NTIMES(r, i) => if (i == 0) false else zeroable(r)
  case NUPTOM(r, i, j) => if (i == 0) false else zeroable(r)
  case RANGE(_) => false
  case NOT(r) => !(zeroable(r))     // bug: incorrect definition for NOT
  case OPT(_) => false
}

// derivative w.r.t. a string (iterates der)
def ders (s: List[Char], r: Rexp) : Rexp = s match {
  case Nil => r
  case c::s => ders(s, der(c, r))
}


// regular expressions for the While language
val SYM = RANGE("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_".toList)
val DIGIT = RANGE("0123456789".toList)
val ID = SYM ~ (SYM | DIGIT).% 
val NUM = PLUS(DIGIT)
val KEYWORD : Rexp = "if" | "then" | "else" | "write" | "def"
val SEMI: Rexp = ";"
val COMMA: Rexp = ","
val OP: Rexp = ":=" | "==" | "-" | "+" | "*" | "!=" | "<=" | "=>" | "<" | ">" | "%" | "=" | "/"
val WHITESPACE = PLUS(" " | "\n" | "\t")
val RPAREN: Rexp = ")"
val LPAREN: Rexp = "("
val ALL = SYM | DIGIT | OP | " " | ":" | ";" | "\"" | "=" | "," | "(" | ")"
val ALL2 = ALL | "\n"
val COMMENT = ("/*" ~ ALL2.% ~ "*/") | ("//" ~ ALL.% ~ "\n")


// token for While language
abstract class Token
case object T_WHITESPACE extends Token
case object T_SEMI extends Token
case object T_COMMA extends Token
case object T_LPAREN extends Token
case object T_RPAREN extends Token
case object T_COMMENT extends Token
case class T_ID(s: String) extends Token
case class T_OP(s: String) extends Token
case class T_NUM(s: String) extends Token
case class T_KWD(s: String) extends Token
case class T_ERR(s: String) extends Token // special error token


type TokenFun = String => Token
type LexRules = List[(Rexp, TokenFun)]
val While_lexing_rules: LexRules = 
  List((KEYWORD, (s) => T_KWD(s)),
       (ID, (s) => T_ID(s)),
       (COMMENT, (s) => T_COMMENT),
       (OP, (s) => T_OP(s)),
       (NUM, (s) => T_NUM(s)),
       (SEMI, (s) => T_SEMI),
       (COMMA, (s) => T_COMMA),
       (LPAREN, (s) => T_LPAREN),
       (RPAREN, (s) => T_RPAREN),
       (WHITESPACE, (s) => T_WHITESPACE))


// calculates derivatives until all of them are zeroable
@tailrec
def munch(s: List[Char], 
          pos: Int, 
          rs: LexRules, 
          last: Option[(Int, TokenFun)]): Option[(Int, TokenFun)] = {
  rs match {
  case Nil => last
  case rs if (s.length <= pos) => last
  case rs => {
    val ders = rs.map({case (r, tf) => (der(s(pos), r), tf)})
    val rs_nzero = ders.filterNot({case (r, _) => zeroable(r)})
    val rs_nulls = ders.filter({case (r, _) => nullable(r)})
    val new_last = if (rs_nulls != Nil) Some((pos, rs_nulls.head._2)) else last
    munch(s, 1 + pos, rs_nzero, new_last)
  }
}}

// iterates the munching function and returns a Token list
def tokenize(s: String, rs: LexRules) : List[Token] = munch(s.toList, 0, rs, None) match {
  case None if (s == "") => Nil
  case None => List(T_ERR(s"Lexing error: $s"))
  case Some((n, tf)) => {
    val (head, tail) = s.splitAt(n + 1)
    tf(head)::tokenize(tail, rs)
  }
}

def tokenizer(s:String) : List[Token] = 
  tokenize(s, While_lexing_rules).filter {
    case T_ERR(s) => { println(s); sys.exit(-1) }
    case T_WHITESPACE => false
    case T_COMMENT => false
    case _ => true
  } 



// Parser - Abstract syntax trees
abstract class Exp
abstract class BExp 
abstract class Decl

case class Def(name: String, args: List[String], body: Exp) extends Decl
case class Main(e: Exp) extends Decl

case class Call(name: String, args: List[Exp]) extends Exp
case class If(a: BExp, e1: Exp, e2: Exp) extends Exp
case class Write(e: Exp) extends Exp
case class Var(s: String) extends Exp
case class Num(i: Int) extends Exp
case class Aop(o: String, a1: Exp, a2: Exp) extends Exp
case class Sequ(e1: Exp, e2: Exp) extends Exp

case class Bop(o: String, a1: Exp, a2: Exp) extends BExp

// calculating the maximal needed stack size
def max_stack_exp(e: Exp): Int = e match {
  case Call(_, args) => args.map(max_stack_exp).sum
  case If(a, e1, e2) => max_stack_bexp(a) + (List(max_stack_exp(e1), max_stack_exp(e2)).max)
  case Write(e) => max_stack_exp(e) + 1
  case Var(_) => 1
  case Num(_) => 1
  case Aop(_, a1, a2) => max_stack_exp(a1) + max_stack_exp(a2)
  case Sequ(e1, e2) => List(max_stack_exp(e1), max_stack_exp(e2)).max
}
def max_stack_bexp(e: BExp): Int = e match {
  case Bop(_, a1, a2) => max_stack_exp(a1) + max_stack_exp(a2)
}

// Parser combinators
abstract class Parser[I <% Seq[_], T] {
  def parse(ts: I): Set[(T, I)]

  def parse_all(ts: I) : Set[T] =
    for ((head, tail) <- parse(ts); if (tail.isEmpty)) yield head

  def parse_single(ts: I) : T = parse_all(ts).toList match {
    case List(t) => t
    case _ => { println ("Parse Error") ; sys.exit(-1) }
  }
}

class SeqParser[I <% Seq[_], T, S](p: => Parser[I, T], q: => Parser[I, S]) extends Parser[I, (T, S)] {
  def parse(sb: I) = 
    for ((head1, tail1) <- p.parse(sb); 
         (head2, tail2) <- q.parse(tail1)) yield ((head1, head2), tail2)
}

class AltParser[I <% Seq[_], T](p: => Parser[I, T], q: => Parser[I, T]) extends Parser[I, T] {
  def parse(sb: I) = p.parse(sb) ++ q.parse(sb)   
}

class FunParser[I <% Seq[_], T, S](p: => Parser[I, T], f: T => S) extends Parser[I, S] {
  def parse(sb: I) = 
    for ((head, tail) <- p.parse(sb)) yield (f(head), tail)
}

case class TokParser(tok: Token) extends Parser[List[Token], Token] {
  def parse(ts: List[Token]) = ts match {
    case t::ts if (t == tok) => Set((t, ts)) 
    case _ => Set ()
  }
}

implicit def token2tparser(t: Token) = TokParser(t)

case object NumParser extends Parser[List[Token], Int] {
  def parse(ts: List[Token]) = ts match {
    case T_NUM(s)::ts => Set((s.toInt, ts)) 
    case _ => Set ()
  }
}

case object IdParser extends Parser[List[Token], String] {
  def parse(ts: List[Token]) = ts match {
    case T_ID(s)::ts => Set((s, ts)) 
    case _ => Set ()
  }
}


implicit def ParserOps[I<% Seq[_], T](p: Parser[I, T]) = new {
  def || (q : => Parser[I, T]) = new AltParser[I, T](p, q)
  def ==>[S] (f: => T => S) = new FunParser[I, T, S](p, f)
  def ~[S] (q : => Parser[I, S]) = new SeqParser[I, T, S](p, q)
}
implicit def TokOps(t: Token) = new {
  def || (q : => Parser[List[Token], Token]) = new AltParser[List[Token], Token](t, q)
  def ==>[S] (f: => Token => S) = new FunParser[List[Token], Token, S](t, f)
  def ~[S](q : => Parser[List[Token], S]) = new SeqParser[List[Token], Token, S](t, q)
}

def ListParser[I <% Seq[_], T, S](p: => Parser[I, T], q: => Parser[I, S]): Parser[I, List[T]] = {
  (p ~ q ~ ListParser(p, q)) ==> { case ((x, y), z) => x :: z : List[T] } ||
  (p ==> ((s) => List(s)))
}


// expressions
lazy val Exp: Parser[List[Token], Exp] = 
  (T_KWD("if") ~ BExp ~ T_KWD("then") ~ Exp ~ T_KWD("else") ~ Exp) ==>
    { case (((((x, y), z), u), v), w) => If(y, u, w): Exp } ||
  (M ~ T_SEMI ~ Exp) ==> { case ((x, y), z) => Sequ(x, z): Exp } || M
lazy val M: Parser[List[Token], Exp] =
  (T_KWD("write") ~ L) ==> { case (x, y) => Write(y): Exp } || L
lazy val L: Parser[List[Token], Exp] = 
  (T ~ T_OP("+") ~ Exp) ==> { case ((x, y), z) => Aop("+", x, z): Exp } ||
  (T ~ T_OP("-") ~ Exp) ==> { case ((x, y), z) => Aop("-", x, z): Exp } || T  
lazy val T: Parser[List[Token], Exp] = 
  (F ~ T_OP("*") ~ T) ==> { case ((x, y), z) => Aop("*", x, z): Exp } || 
  (F ~ T_OP("/") ~ T) ==> { case ((x, y), z) => Aop("/", x, z): Exp } || 
  (F ~ T_OP("%") ~ T) ==> { case ((x, y), z) => Aop("%", x, z): Exp } || F
lazy val F: Parser[List[Token], Exp] = 
  (IdParser ~ T_LPAREN ~ ListParser(Exp, T_COMMA) ~ T_RPAREN) ==> 
    { case (((x, y), z), w) => Call(x, z): Exp } ||
  (T_LPAREN ~ Exp ~ T_RPAREN) ==> { case ((x, y), z) => y: Exp } || 
  IdParser ==> { case x => Var(x): Exp } || 
  NumParser ==> { case x => Num(x): Exp }

// boolean expressions
lazy val BExp: Parser[List[Token], BExp] = 
  (Exp ~ T_OP("==") ~ Exp) ==> { case ((x, y), z) => Bop("==", x, z): BExp } || 
  (Exp ~ T_OP("!=") ~ Exp) ==> { case ((x, y), z) => Bop("!=", x, z): BExp } || 
  (Exp ~ T_OP("<") ~ Exp) ==> { case ((x, y), z) => Bop("<", x, z): BExp } || 
  (Exp ~ T_OP(">") ~ Exp) ==> { case ((x, y), z) => Bop("<", z, x): BExp } || 
  (Exp ~ T_OP("<=") ~ Exp) ==> { case ((x, y), z) => Bop("<=", x, z): BExp } || 
  (Exp ~ T_OP("=>") ~ Exp) ==> { case ((x, y), z) => Bop("<=", z, x): BExp }  

lazy val Defn: Parser[List[Token], Decl] =
   (T_KWD("def") ~ IdParser ~ T_LPAREN ~ ListParser(IdParser, T_COMMA) ~ T_RPAREN ~ T_OP("=") ~ Exp) ==>
     { case ((((((x, y), z), w), u), v), r) => Def(y, w, r): Decl }

lazy val Prog: Parser[List[Token], List[Decl]] =
  (Defn ~ T_SEMI ~ Prog) ==> { case ((x, y), z) => x :: z : List[Decl] } ||
  (Exp ==> ((s) => List(Main(s)) : List[Decl]))


// compiler - built-in functions 
// copied from http://www.ceng.metu.edu.tr/courses/ceng444/link/jvm-cpm.html
//

val library = """
.class public XXX.XXX
.super java/lang/Object

.method public <init>()V
        aload_0
        invokenonvirtual java/lang/Object/<init>()V
        return
.end method

.method public static write(I)V 
        .limit locals 1 
        .limit stack 2 
        getstatic java/lang/System/out Ljava/io/PrintStream; 
        iload 0
        invokevirtual java/io/PrintStream/println(I)V 
        return 
.end method

"""

// for generating new labels
var counter = -1

def Fresh(x: String) = {
  counter += 1
  x ++ "_" ++ counter.toString()
}

type Mem = Map[String, Int]
type Instrs = List[String]

def compile_exp(a: Exp, env : Mem) : Instrs = a match {
  case Num(i) => List("ldc " + i.toString + "\n")
  case Var(s) => List("iload " + env(s).toString + "\n")
  case Aop("+", a1, a2) => compile_exp(a1, env) ++ compile_exp(a2, env) ++ List("iadd\n")
  case Aop("-", a1, a2) => compile_exp(a1, env) ++ compile_exp(a2, env) ++ List("isub\n")
  case Aop("*", a1, a2) => compile_exp(a1, env) ++ compile_exp(a2, env) ++ List("imul\n")
  case Aop("/", a1, a2) => compile_exp(a1, env) ++ compile_exp(a2, env) ++ List("idiv\n")
  case Aop("%", a1, a2) => compile_exp(a1, env) ++ compile_exp(a2, env) ++ List("irem\n")
  case If(b, a1, a2) => {
    val if_else = Fresh("If_else")
    val if_end = Fresh("If_end")
    compile_bexp(b, env, if_else) ++
    compile_exp(a1, env) ++
    List("goto " + if_end + "\n") ++
    List("\n" + if_else + ":\n\n") ++
    compile_exp(a2, env) ++
    List("\n" + if_end + ":\n\n")
  }
  case Call(n, args) => {
    val is = "I" * args.length
    args.flatMap(a => compile_exp(a, env)) ++
    List ("invokestatic XXX/XXX/" + n + "(" + is + ")I\n")
  }
  case Sequ(a1, a2) => {
    compile_exp(a1, env) ++ List("pop\n") ++ compile_exp(a2, env)
  }
  case Write(a1) => {
    compile_exp(a1, env) ++
    List("dup\n",
         "invokestatic XXX/XXX/write(I)V\n")
  }
}

def compile_bexp(b: BExp, env : Mem, jmp: String) : Instrs = b match {
  case Bop("==", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ List("if_icmpne " + jmp + "\n")
  case Bop("!=", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ List("if_icmpeq " + jmp + "\n")
  case Bop("<", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ List("if_icmpge " + jmp + "\n")
  case Bop("<=", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ List("if_icmpgt " + jmp + "\n")
}

def compile_decl(d: Decl) : Instrs = d match {
  case Def(name, args, a) => { 
    val env = args.zipWithIndex.toMap
    val is = "I" * args.length
    List(".method public static " + name + "(" + is + ")I \n",
         ".limit locals " + args.length.toString + "\n",
         ".limit stack " + (1 + max_stack_exp(a)).toString + "\n",
         name + "_Start:\n") ++   
    compile_exp(a, env) ++
    List("ireturn\n",
         ".end method \n\n")
  }
  case Main(a) => {
    List(".method public static main([Ljava/lang/String;)V\n",
         ".limit locals 200\n",
         ".limit stack 200\n") ++
    compile_exp(a, Map()) ++
    List("invokestatic XXX/XXX/write(I)V\n",
         "return\n",
         ".end method\n")
  }
}

def compile(class_name: String, input: String) : String = {
  val tks = tokenizer(input)
  println(Prog.parse_single(tks).mkString("\n"))
  val ast = Prog.parse_single(tks)
  val instructions = ast.flatMap(compile_decl).mkString
  (library + instructions).replaceAllLiterally("XXX", class_name)
}




def compile_file(file_name: String) = {
  val class_name = file_name.split('.')(0)
  val output = compile(class_name, fromFile(file_name))
  val fw = new java.io.FileWriter(class_name + ".j") 
  fw.write(output) 
  fw.close()
}

def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  (end - start)/(i * 1.0e9)
}

def compile_run(file_name: String) : Unit = {
  val class_name = file_name.split('.')(0)
  compile_file(file_name)
  val test = ("java -jar jvm/jasmin-2.4/jasmin.jar " + class_name + ".j").!!
  println("Time: " + time_needed(2, ("java " + class_name + "/" + class_name).!))
}


//examples
compile_file("fact.rec")
//compile_run("defs.rec")
//compile_run("fact.rec")
