// regular expressions
abstract class Rexp

case object NULL extends Rexp
case object EMPTY extends Rexp
case class CHAR(c: Char) extends Rexp
case class ALT(r1: Rexp, r2: Rexp) extends Rexp
case class SEQ(r1: Rexp, r2: Rexp) extends Rexp
case class STAR(r: Rexp) extends Rexp
case class NOT(r: Rexp) extends Rexp


// some convenience for typing in regular expressions
def charlist2rexp(s : List[Char]) : Rexp = s match {
  case Nil => EMPTY
  case c::Nil => CHAR(c)
  case c::s => SEQ(CHAR(c), charlist2rexp(s))
}
implicit def string2rexp(s : String) : Rexp = charlist2rexp(s.toList)


// nullable function: tests whether the regular 
// expression can recognise the empty string
def nullable (r: Rexp) : Boolean = r match {
  case NULL => false
  case EMPTY => true
  case CHAR(_) => false
  case ALT(r1, r2) => nullable(r1) || nullable(r2)
  case SEQ(r1, r2) => nullable(r1) && nullable(r2)
  case STAR(_) => true
  case NOT(r) => !(nullable(r))
}

// tests whether a regular expression 
// recognises nothing
def zeroable (r: Rexp) : Boolean = r match {
  case NULL => true
  case EMPTY => false
  case CHAR(_) => false
  case ALT(r1, r2) => zeroable(r1) && zeroable(r2)
  case SEQ(r1, r2) => zeroable(r1) || zeroable(r2)
  case STAR(_) => false
  case NOT(r) => !(zeroable(r))
}

def starts_with (r: Rexp, c: Char) : Boolean = r match {
  case NULL => false
  case EMPTY => false
  case CHAR(d) => (c == d)
  case ALT(r1, r2) => starts_with(r1, c) || starts_with(r2, c)
  case SEQ(r1, r2) => if (nullable(r1)) (starts_with(r1, c) || starts_with(r2, c))
                      else starts_with(r1, c)
  case STAR(r) => starts_with(r, c)
  case NOT(r) => !(starts_with(r, c))
}

// derivative of a regular expression w.r.t. a character
def der (c: Char, r: Rexp) : Rexp = r match {
  case NULL => NULL
  case EMPTY => NULL
  case CHAR(d) => if (c == d) EMPTY else NULL
  case ALT(r1, r2) => ALT(der(c, r1), der(c, r2))
  case SEQ(r1, r2) => 
    if (nullable(r1)) ALT(SEQ(der(c, r1), r2), der(c, r2))
    else SEQ(der(c, r1), r2)
  case STAR(r) => SEQ(der(c, r), STAR(r))
  case NOT(r) => NOT(der (c, r))
}

// derivative w.r.t. a string (iterates der)
def ders (s: List[Char], r: Rexp) : Rexp = s match {
  case Nil => r
  case c::s => ders(s, der(c, r))
}

// main matcher function
def matcher(r: Rexp, s: String) : Boolean = nullable(ders(s.toList, r))


// regular expression for specifying 
// ranges of characters
def RANGE(s : List[Char]) : Rexp = s match {
  case Nil => NULL
  case c::Nil => CHAR(c)
  case c::s => ALT(CHAR(c), RANGE(s))
}

//one or more
def PLUS(r: Rexp) = SEQ(r, STAR(r))


//some regular expressions
val LOWERCASE = RANGE("abcdefghijklmnopqrstuvwxyz".toList)
val UPPERCASE = RANGE("ABCDEFGHIJKLMNOPQRSTUVWXYZ".toList)
val LETTER = ALT(LOWERCASE, UPPERCASE)
val DIGITS = RANGE("0123456789".toList)
val NONZERODIGITS = RANGE("123456789".toList)

val IDENT = SEQ(LETTER, STAR(ALT(LETTER,DIGITS)))
val NUMBER = ALT(SEQ(NONZERODIGITS, STAR(DIGITS)), "0")
val WHITESPACE = RANGE(" \n".toList)
val SYMBOLS = RANGE("/*".toList)

val ALL = ALT(ALT(ALT(LETTER, DIGITS), WHITESPACE), SYMBOLS)

val COMMENT = SEQ(SEQ("/*", NOT(SEQ(SEQ(STAR(ALL), "*/"), STAR(ALL)))), "*/")

println(matcher(NUMBER, "0"))
println(matcher(NUMBER, "01"))
println(matcher(NUMBER, "123450"))

println(matcher(SEQ(STAR("a"), STAR("b")), "bbaaa"))
println(matcher(ALT(STAR("a"), STAR("b")), ""))
println(matcher("abc", ""))
println(matcher(STAR(ALT(EMPTY, "a")), ""))
println(matcher(STAR(EMPTY), "a"))
println(matcher("cab","cab"))
println(matcher(STAR("a"),"aaa"))
println(matcher("cab" ,"cab"))
println(matcher(STAR("a"),"aaa"))

println(matcher(COMMENT, "/* */"))
println(matcher(COMMENT, "/* foobar comment */"))
println(matcher(COMMENT, "/* test */ test */"))

// an example list of regular expressions
val regs: List[Rexp]=  List("if", "then", "else", "+", IDENT, NUMBER, COMMENT, WHITESPACE) 


def error (s: String) = throw new IllegalArgumentException ("Could not lex " + s)

def munch(r: Rexp, s: List[Char], t: List[Char]) : Option[(List[Char], List[Char])] = 
  if (zeroable(r)) None else s match {
  case Nil => if (nullable(r)) Some(Nil, t) else None
  case c::s if (zeroable(der (c, r)) && nullable(r)) => Some(c::s, t)
  //case c::s if (zeroable(der (c, r))) => None
  case c::s => munch(der (c, r), s, t ::: List(c))
}


def lex_one (regs: List[Rexp], s: List[Char]) : (List[Char], List[Char]) = {
 val somes = regs.map { munch(_, s, Nil) } .flatten
 if (somes == Nil) error(s.mkString) else (somes sortBy (_._1.length) head)
}

def lex_all (regs: List[Rexp], s: List[Char]) : List[String] = s match {
  case Nil => Nil
  case _ => lex_one(regs, s) match {
    case (rest, s) => s.mkString :: lex_all(regs, rest) 
  }
}



starts_with(der('/', COMMENT), '*')

munch(COMMENT, "/*ifff if 34 */".toList, Nil)
val COMMENT2 = NOT(SEQ(SEQ(STAR(ALL), "*/"), STAR(ALL)))

der('a', COMMENT2)
zeroable(der('a', COMMENT2))

matcher(COMMENT2, "ifff if 34")
munch(COMMENT2, "ifff if 34".toList, Nil)
starts_with(COMMENT2, 'i')
lex_all(regs, "ifff if 34".toList)
lex_all(regs, "ifff $ if 34".toList)

