// A Small Compiler for a Simple Functional Language
// (includes an external lexer and parser)
//
// call with 
//
//     scala fun.scala fact
//
//     scala fun.scala defs
//
// this will generate a .j file and run the jasmin
// assembler (installed at jvm/jasmin-2.4/jasmin.jar)
// it runs the resulting JVM file twice for timing 
// purposes.




//object Compiler {

import java.io._  
import scala.util._
import scala.sys.process._

// Abstract syntax trees for the Fun language
abstract class Exp extends Serializable 
abstract class BExp extends Serializable 
abstract class Decl extends Serializable 

case class Def(name: String, args: List[String], body: Exp) extends Decl
case class Main(e: Exp) extends Decl

case class Call(name: String, args: List[Exp]) extends Exp
case class If(a: BExp, e1: Exp, e2: Exp) extends Exp
case class Write(e: Exp) extends Exp
case class Var(s: String) extends Exp
case class Num(i: Int) extends Exp
case class Aop(o: String, a1: Exp, a2: Exp) extends Exp
case class Sequence(e1: Exp, e2: Exp) extends Exp
case class Bop(o: String, a1: Exp, a2: Exp) extends BExp


// compiler - built-in functions 
// copied from http://www.ceng.metu.edu.tr/courses/ceng444/link/jvm-cpm.html
//


// for generating new labels
var counter = -1

def Fresh(x: String) = {
  counter += 1
  x ++ "_" ++ counter.toString()
}



// Abstract syntax trees for the Fun language
abstract class KExp

case class KVar(s: String) extends KExp
case class KNum(i: Int) extends KExp
case class KAop(o: String, x1: String, x2: String) extends KExp
case class KIfeq(x1: String, x2: String, e1: KExp, e2: KExp) extends KExp
case class KCall(o: String, vrs: List[String]) extends KExp
case class KLet(x: String, e1: KExp, e2: KExp) extends KExp {
  override def toString = s"let $x = $e1 in \n$e2" 
}

def K(e: Exp) : KExp = e match {
  case Var(s) => KVar(s) 
  case Num(i) => KNum(i)
  case Aop(o, a1, a2) => {
    val x1 = Fresh("tmp")
    val x2 = Fresh("tmp") 
    KLet(x1, K(a1), KLet(x2, K(a2), KAop(o, x1, x2)))
  } 
  case Call(name: String, args: List[Exp]) => {
    val args_new = args.map{a => (Fresh("x"), K(a))}
    def aux(as: List[(String, KExp)]) : KExp = as match {
      case Nil => KCall(name, args_new.map(_._1))
      case (x, a)::rest => KLet(x, a, aux(rest))
    }
    aux(args_new)
  }
  
}

def Denest(e: KExp) : KExp = e match {
  case KLet(xt, e1, e2) => {
    def insert(e: KExp) : KExp = e match {
      case KLet(yt, e3, e4) => KLet(yt, e3, insert(e4))
      case e => KLet(xt, e, Denest(e2))
    }
    insert(Denest(e1))  
  }
  case _ => e
}

val e = Aop("*", Aop("+", Num(1), Call("foo", List(Num(2), Num(3)))), Num(4))
println(K(e))
println(Denest(K(e)))



// convenient string interpolations 
// for instructions, labels and methods
import scala.language.implicitConversions
import scala.language.reflectiveCalls

implicit def sring_inters(sc: StringContext) = new {
    def i(args: Any*): String = "   " ++ sc.s(args:_*) ++ "\n"
    def l(args: Any*): String = sc.s(args:_*) ++ ":\n"
    def m(args: Any*): String = sc.s(args:_*) ++ "\n"
}


type Env = Map[String, Int]



// compile expressions
def compile_exp(a: Exp, env : Env, k: Int) : (String, Int) = a match {
  case Num(i) => (i"%$k = add i32 0, i32 $i", k)
  case Var(s) => (i"%$k = add i32 0, i32 %${env(s)}", k)
  case Aop("+", a1, a2) => {
    val (cs1, k1) = compile_exp(a1, env, k)
    val (cs2, k2) = compile_exp(a2, env, k1 + 1)
    (cs1 ++ cs2 ++ i"%${k2+1} = add i32 %$k1, i32 %$k2", k2 + 1)
  }
  case Aop("-", a1, a2) => {
    val (cs1, k1) = compile_exp(a1, env, k)
    val (cs2, k2) = compile_exp(a2, env, k1 + 1)
    (cs1 ++ cs2 ++ i"%${k2+1} = sub i32 %$k1, i32 %$k2", k2 + 1)
  }
  case Aop("*", a1, a2) => {
    val (cs1, k1) = compile_exp(a1, env, k)
    val (cs2, k2) = compile_exp(a2, env, k1 + 1)
    (cs1 ++ cs2 ++ i"%${k2+1} = mul i32 %$k1, i32 %$k2", k2 + 1)
  }
  /*
  case If(b, a1, a2) => {
    val if_else = Fresh("If_else")
    val if_end = Fresh("If_end")
    compile_bexp(b, env, if_else) ++
    compile_exp(a1, env) ++
    i"goto $if_end" ++
    l"$if_else" ++
    compile_exp(a2, env) ++
    l"$if_end"
  }
  case Call(name, args) => {
    val is = "I" * args.length
    args.map(a => compile_exp(a, env)).mkString ++
    i"invokestatic XXX/XXX/$name($is)I"
  }
  case Sequence(a1, a2) => {
    compile_exp(a1, env) ++ i"pop" ++ compile_exp(a2, env)
  }
  case Write(a1) => {
    compile_exp(a1, env) ++
    i"dup" ++
    i"invokestatic XXX/XXX/write(I)V"
  }
  */
}

val e = Aop("*", Aop("+", Num(2), Num(4)), Num(5))
compile_exp(e, Map(), 1)._1.mkString


// compile boolean expressions
def compile_bexp(b: BExp, env : Env, jmp: String) : String = b match {
  case Bop("==", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"if_icmpne $jmp"
  case Bop("!=", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"if_icmpeq $jmp"
  case Bop("<", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"if_icmpge $jmp"
  case Bop("<=", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"if_icmpgt $jmp"
}

// compile function for declarations and main
def compile_decl(d: Decl) : String = d match {
  case Def(name, args, a) => { 
    val env = args.zipWithIndex.toMap
    val is = "I" * args.length
    m".method public static $name($is)I" ++
    m".limit locals ${args.length.toString}" ++
    m".limit stack ${1 + max_stack_exp(a)}" ++
    l"${name}_Start" ++   
    compile_exp(a, env) ++
    i"ireturn" ++
    m".end method\n"
  }
  case Main(a) => {
    m".method public static main([Ljava/lang/String;)V" ++
    m".limit locals 200" ++
    m".limit stack 200" ++
    compile_exp(a, Map()) ++
    i"invokestatic XXX/XXX/write(I)V" ++
    i"return" ++
    m".end method\n"
  }
}

// main compiler functions

def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  (end - start)/(i * 1.0e9)
}

def deserialise[T](fname: String) : Try[T] = {
  import scala.util.Using
  Using(new ObjectInputStream(new FileInputStream(fname))) {
    in => in.readObject.asInstanceOf[T]
  }
}

def compile(class_name: String) : String = {
  val ast = deserialise[List[Decl]](class_name ++ ".prs").getOrElse(Nil) 
  val instructions = ast.map(compile_decl).mkString
  (library + instructions).replaceAllLiterally("XXX", class_name)
}

def compile_to_file(class_name: String) = {
  val output = compile(class_name)
  scala.tools.nsc.io.File(s"${class_name}.j").writeAll(output)
}

def compile_and_run(class_name: String) : Unit = {
  compile_to_file(class_name)
  (s"java -jar jvm/jasmin-2.4/jasmin.jar ${class_name}.j").!!
  println("Time: " + time_needed(2, (s"java ${class_name}/${class_name}").!))
}


// some examples of .fun files
//compile_to_file("fact")
//compile_and_run("fact")
//compile_and_run("defs")


def main(args: Array[String]) = 
   compile_and_run(args(0))


}