
// regular expressions including NOT
abstract class Rexp

case object NULL extends Rexp
case object EMPTY extends Rexp
case class CHAR(c: Char) extends Rexp
case class ALT(r1: Rexp, r2: Rexp) extends Rexp
case class SEQ(r1: Rexp, r2: Rexp) extends Rexp
case class STAR(r: Rexp) extends Rexp
case class NOT(r: Rexp) extends Rexp


// some convenience for typing in regular expressions
def charlist2rexp(s : List[Char]) : Rexp = s match {
  case Nil => EMPTY
  case c::Nil => CHAR(c)
  case c::s => SEQ(CHAR(c), charlist2rexp(s))
}
implicit def string2rexp(s : String) : Rexp = charlist2rexp(s.toList)


// nullable function: tests whether the regular 
// expression can recognise the empty string
def nullable (r: Rexp) : Boolean = r match {
  case NULL => false
  case EMPTY => true
  case CHAR(_) => false
  case ALT(r1, r2) => nullable(r1) || nullable(r2)
  case SEQ(r1, r2) => nullable(r1) && nullable(r2)
  case STAR(_) => true
  case NOT(r) => !(nullable(r))
}

// tests whether a regular expression 
// cannot recognise more
def no_more (r: Rexp) : Boolean = r match {
  case NULL => true
  case EMPTY => false
  case CHAR(_) => false
  case ALT(r1, r2) => no_more(r1) && no_more(r2)
  case SEQ(r1, r2) => if (nullable(r1)) (no_more(r1) && no_more(r2)) else no_more(r1)
  case STAR(_) => false
  case NOT(r) => !(no_more(r))
}


// derivative of a regular expression w.r.t. a character
def der (c: Char, r: Rexp) : Rexp = r match {
  case NULL => NULL
  case EMPTY => NULL
  case CHAR(d) => if (c == d) EMPTY else NULL
  case ALT(r1, r2) => ALT(der(c, r1), der(c, r2))
  case SEQ(r1, r2) => 
    if (nullable(r1)) ALT(SEQ(der(c, r1), r2), der(c, r2))
    else SEQ(der(c, r1), r2)
  case STAR(r) => SEQ(der(c, r), STAR(r))
  case NOT(r) => NOT(der (c, r))
}


// regular expression for specifying 
// ranges of characters
def RANGE(s : List[Char]) : Rexp = s match {
  case Nil => NULL
  case c::Nil => CHAR(c)
  case c::s => ALT(CHAR(c), RANGE(s))
}

//one or more
def PLUS(r: Rexp) = SEQ(r, STAR(r))


//some regular expressions
val LOWERCASE = RANGE("abcdefghijklmnopqrstuvwxyz".toList)
val UPPERCASE = RANGE("ABCDEFGHIJKLMNOPQRSTUVWXYZ".toList)
val LETTER = ALT(LOWERCASE, UPPERCASE)
val DIGIT = RANGE("0123456789".toList)
val NONZERODIGIT = RANGE("123456789".toList)

val IDENT = SEQ(LETTER, STAR(ALT(LETTER,DIGIT)))
val NUMBER = ALT(SEQ(NONZERODIGIT, STAR(DIGIT)), "0")
val WHITESPACE = RANGE(" \n".toList)
val WHITESPACES = PLUS(WHITESPACE)

val ALL = ALT(ALT(LETTER, DIGIT), WHITESPACE)
val COMMENT = SEQ(SEQ("/*", NOT(SEQ(SEQ(STAR(ALL), "*/"), STAR(ALL)))), "*/")


// an example list of regular expressions
val regs: List[Rexp]=  List("if", "then", "else", "+", IDENT, NUMBER, WHITESPACES, COMMENT) 


def error (s: String) = throw new IllegalArgumentException ("Could not lex " + s)

def munch(r: Rexp, s: List[Char], t: List[Char]) : Option[(List[Char], List[Char])] = 
  s match {
    case Nil if (nullable(r)) => Some(Nil, t)
    case Nil => None
    case c::s if (no_more(der (c, r)) && nullable(r)) => Some(c::s, t)
    case c::s if (no_more(der (c, r))) => None
    case c::s => munch(der (c, r), s, t ::: List(c))
  }

def one_string (regs: List[Rexp], s: List[Char]) : (List[Char], List[Char]) = {
 val somes = regs.map { munch(_, s, Nil) } .flatten
 if (somes == Nil) error(s.mkString) else (somes sortBy (_._1.length) head)
}

def tokenize (regs: List[Rexp], s: List[Char]) : List[String] = s match {
  case Nil => Nil
  case _ => one_string(regs, s) match {
    case (rest, s) => s.mkString :: tokenize(regs, rest) 
  }
}

//examples
println(tokenize(regs, "if true then then 42 else +".toList))
println(tokenize(regs, "if+true+then+then+42+else +".toList))
println(tokenize(regs, "ifff if     34 34".toList))
println(tokenize(regs, "/*ifff if */ hhjj /*34 */".toList))
println(tokenize(regs, "/* if true then */ then 42 else +".toList))
//println(tokenize(regs, "ifff $ if 34".toList)) // causes an error because of the symbol $
