// Version with simplification during derivatives;
// this keeps the regular expressions small, which
// is good for run-time
 

abstract class Rexp
case object ZERO extends Rexp
case object ONE extends Rexp
case class CHAR(c: Char) extends Rexp
case class ALT(r1: Rexp, r2: Rexp) extends Rexp 
case class SEQ(r1: Rexp, r2: Rexp) extends Rexp 
case class STAR(r: Rexp) extends Rexp 
case class NTIMES(r: Rexp, n: Int) extends Rexp 


// string of a regular expressions - for testing purposes 
def string(r: Rexp) : String = r match {
  case ZERO => "0"
  case ONE => "1"
  case CHAR(c) => c.toString 
  case ALT(r1, r2) => s"(${string(r1)} + ${string(r2)})"
  case SEQ(CHAR(c), CHAR(d)) => s"${c}${d}"
  case SEQ(ONE, CHAR(c)) => s"1${c}"
  case SEQ(r1, r2) => s"(${string(r1)} ~ ${string(r2)})"
  case STAR(r) => s"(${string(r)})*"
  case NTIMES(r, n) =>  s"(${string(r)}){${n}}"
}

// size of a regular expressions - for testing purposes 
def size(r: Rexp) : Int = r match {
  case ZERO => 1
  case ONE => 1
  case CHAR(_) => 1
  case ALT(r1, r2) => 1 + size(r1) + size(r2)
  case SEQ(r1, r2) => 1 + size(r1) + size(r2)
  case STAR(r) => 1 + size(r)
  case NTIMES(r, _) => 1 + size(r)
}



// nullable function: tests whether the regular 
// expression can recognise the empty string
def nullable (r: Rexp) : Boolean = r match {
  case ZERO => false
  case ONE => true
  case CHAR(_) => false
  case ALT(r1, r2) => nullable(r1) || nullable(r2)
  case SEQ(r1, r2) => nullable(r1) && nullable(r2)
  case STAR(_) => true
  case NTIMES(r, i) => if (i == 0) true else nullable(r)
}

// derivative of a regular expression w.r.t. a character
def der (c: Char, r: Rexp) : Rexp = r match {
  case ZERO => ZERO
  case ONE => ZERO
  case CHAR(d) => if (c == d) ONE else ZERO
  case ALT(r1, r2) => ALT(der(c, r1), der(c, r2))
  case SEQ(r1, r2) => 
    if (nullable(r1)) ALT(SEQ(der(c, r1), r2), der(c, r2))
    else SEQ(der(c, r1), r2)
  case STAR(r1) => SEQ(der(c, r1), STAR(r1))
  case NTIMES(r, i) => 
    if (i == 0) ZERO else SEQ(der(c, r), NTIMES(r, i - 1))
}



def simp(r: Rexp, seen: Set[Rexp]) : (Rexp, Set[Rexp]) = r match {
  case ALT(r1, r2) => {
    val (r1s, seen1) = simp(r1, seen)
    val (r2s, seen2) = simp(r2, seen1)
    (r1s, r2s) match {
      case (ZERO, r2s) => (r2s, seen2)
      case (r1s, ZERO) => (r1s, seen2)
      case (r1s, r2s) => (ALT(r1s, r2s), seen2)
    }
  }
  case SEQ(r1, r2) =>  {
    val (r1s, _) = simp(r1, Set())
    val (r2s, _) = simp(r2, Set())
    if (seen.contains(SEQ(r1s, r2s))) (ZERO, seen)
    else (r1s, r2s) match {
      case (ZERO, _) => (ZERO, seen)
      case (_, ZERO) => (ZERO, seen)
      case (ONE, r2s) => (r2s, seen + r2s)
      case (r1s, ONE) => (r1s, seen + r1s)
      case (r1s, r2s) => (SEQ(r1s, r2s), seen + SEQ(r1s, r2s))
    }
  }
  case r => if (seen.contains(r)) (ZERO, seen) else (r, seen + r)
}


// derivative w.r.t. a string (iterates der)
def ders (s: List[Char], r: Rexp) : Rexp = s match {
  case Nil => r
  case c::s => ders(s, simp(der(c, r), Set())._1)
}


// main matcher function
def matcher(r: Rexp, s: String) : Boolean = nullable(ders(s.toList, r))


//one or zero
def OPT(r: Rexp) = ALT(r, ONE)









def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  (end - start)/(i * 1.0e9)
}


// star example
val Tstar = STAR(STAR(STAR(CHAR('a'))))

string(ders("".toList, Tstar))
size(ders("".toList, Tstar))      // 4
string(ders("a".toList, Tstar))
size(ders("a".toList, Tstar))     // 11
string(ders("aa".toList, Tstar))
size(ders("aa".toList, Tstar))    // 11
string(ders("aaa".toList, Tstar))
size(ders("aaa".toList, Tstar))   // 11
string(ders("aaaa".toList, Tstar))
size(ders("aaaa".toList, Tstar))  // 11
string(ders("aaaa".toList, Tstar))
size(ders("aaaaa".toList, Tstar)) // 11
string(ders("aaaab".toList, Tstar))
size(ders("aaaaab".toList, Tstar)) // 1

// test: ("a" | "aa")*
val EVIL3 = STAR(ALT(CHAR('a'), SEQ(CHAR('a'), CHAR('a'))))

for (i <- 1 to 100 by 1) {
  println(i + " " + "%.5f".format(time_needed(2, matcher(EVIL3, "a" * i))) + 
	  " size: " + size(ders(("a" * i).toList, EVIL3)))
}


println("start " + string(EVIL3) + "    " + size(EVIL3))
val t1  = der('a', EVIL3)
println(string(t1) + "    " + size(t1))
val t1s = simp(t1, Set())._1 
println("simplified " + string(t1s) + "    " + size(t1s))
val t2  = der('a', t1s)
println(string(t2) + "    " + size(t2))
val t2s = simp(t2, Set())._1 
println("simplified " + string(t2s) + "    " + size(t2s))
val t3  = der('a', t2s)
println(string(t3) + "    " + size(t3))
val t3s = simp(t3, Set())._1 
println("simplified " + string(t3s) + "    " + size(t3s))
val t4  = der('a', t3s)
val t4s = simp(t4, Set())._1 








println(string(t4) + "    " + size(t4))
println("simplified " + string(t4s) + "    " + size(t4s))


// Test Cases

//evil regular expressions: (a?){n} a{n}  and (a*)* b
def EVIL1(n: Int) = SEQ(NTIMES(OPT(CHAR('a')), n), NTIMES(CHAR('a'), n))
val EVIL2 = SEQ(STAR(STAR(CHAR('a'))), CHAR('b'))


def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  (end - start)/(i * 1.0e9)
}


//test: (a?{n}) (a{n})
for (i <- 1 to 8001 by 1000) {
  println(i + " " + "%.5f".format(time_needed(2, matcher(EVIL1(i), "a" * i))))
}

for (i <- 1 to 8001 by 1000) {
  println(i + " " + "%.5f".format(time_needed(2, matcher(EVIL1(i), "a" * i))))
}

//test: (a*)* b
for (i <- 1 to 6000001 by 500000) {
  println(i + " " + "%.5f".format(time_needed(2, matcher(EVIL2, "a" * i))))
}

for (i <- 1 to 6000001 by 500000) {
  println(i + " " + "%.5f".format(time_needed(2, matcher(EVIL2, "a" * i))))
}



// size of a regular expressions - for testing purposes 
def size(r: Rexp) : Int = r match {
  case ZERO => 1
  case ONE => 1
  case CHAR(_) => 1
  case ALT(r1, r2) => 1 + size(r1) + size(r2)
  case SEQ(r1, r2) => 1 + size(r1) + size(r2)
  case STAR(r) => 1 + size(r)
  case NTIMES(r, _) => 1 + size(r)
}


// now the size of the derivatives grows 
// much, much slower

size(ders("".toList, EVIL2))      // 5
size(ders("a".toList, EVIL2))     // 8
size(ders("aa".toList, EVIL2))    // 8
size(ders("aaa".toList, EVIL2))   // 8
size(ders("aaaa".toList, EVIL2))  // 8
size(ders("aaaaa".toList, EVIL2)) // 8





// Examples from the Sulzmann paper
val sulzmann = STAR(ALT(ALT(CHAR('a'), CHAR('b')), SEQ(CHAR('a'), CHAR('b'))))


for (i <- 1 to 4501 by 500) {
  println(i + ": " + "%.5f".format(time_needed(1, matcher(sulzmann, "a" * i))))
}

for (i <- 1 to 4501 by 500) {
  println(i + ": " + "%.5f".format(time_needed(1, matcher(sulzmann, "ab" * i))))
}




(((1 + 1a) ~ ((a + aa))*) + (((0 + 1) ~ ((a + aa))*) + ((1 + 1a) ~ ((a + aa))*)))
