// A Small Compiler for the WHILE Language
// (stub for including arrays)

// the abstract syntax trees
abstract class Stmt
abstract class AExp
abstract class BExp 
type Block = List[Stmt]

// statements
case object Skip extends Stmt
case class If(a: BExp, bl1: Block, bl2: Block) extends Stmt
case class While(b: BExp, bl: Block) extends Stmt
case class Assign(s: String, a: AExp) extends Stmt
case class AssignA(s: String, a1: AExp, a2: AExp) extends Stmt
case class Write(s: String) extends Stmt  // writes out a variable
case class Array(s: String, n: Int) extends Stmt 

// arithmetic expressions
case class Var(s: String) extends AExp
case class Num(i: Int) extends AExp
case class Aop(o: String, a1: AExp, a2: AExp) extends AExp
case class Ref(s: String, a1: AExp) extends AExp

// boolean expressions
case object True extends BExp
case object False extends BExp
case class Bop(o: String, a1: AExp, a2: AExp) extends BExp


// compiler headers needed for the JVM
// (contains an init method, as well as methods for read and write)
val beginning = """
.class public XXX.XXX
.super java/lang/Object

.method public <init>()V
   aload_0
   invokenonvirtual java/lang/Object/<init>()V
   return
.end method

.method public static write(I)V 
    .limit locals 1 
    .limit stack 2 
    getstatic java/lang/System/out Ljava/io/PrintStream; 
    iload 0
    i2c
    invokevirtual java/io/PrintStream/print(C)V 
    return 
.end method

.method public static main([Ljava/lang/String;)V
   .limit locals 200
   .limit stack 200

"""

val ending = """

   return

.end method
"""

println("Start compilation")


// for generating new labels
var counter = -1

def Fresh(x: String) = {
  counter += 1
  x ++ "_" ++ counter.toString()
}

// environments and instructions
type Env = Map[String, String]
type Instrs = List[String]

// arithmetic expression compilation
def compile_aexp(a: AExp, env : Env) : Instrs = a match {
  case Num(i) => List("ldc " + i.toString + "\n")
  case Var(s) => List("iload " + env(s) + "\n")
  case Aop("+", a1, a2) => 
    compile_aexp(a1, env) ++ 
    compile_aexp(a2, env) ++ List("iadd\n")
  case Aop("-", a1, a2) => 
    compile_aexp(a1, env) ++ compile_aexp(a2, env) ++ List("isub\n")
  case Aop("*", a1, a2) => 
    compile_aexp(a1, env) ++ compile_aexp(a2, env) ++ List("imul\n")
  case Ref(s, a1) => 
    List("aload " + env(s) + "\n") ++ compile_aexp(a1, env) ++ List("iaload \n")
}

// boolean expression compilation
def compile_bexp(b: BExp, env : Env, jmp: String) : Instrs = b match {
  case True => Nil
  case False => List("goto " + jmp + "\n")
  case Bop("=", a1, a2) => 
    compile_aexp(a1, env) ++ compile_aexp(a2, env) ++ 
    List("if_icmpne " + jmp + "\n")
  case Bop("!=", a1, a2) => 
    compile_aexp(a1, env) ++ compile_aexp(a2, env) ++ 
    List("if_icmpeq " + jmp + "\n")
  case Bop("<", a1, a2) => 
    compile_aexp(a1, env) ++ compile_aexp(a2, env) ++ 
    List("if_icmpge " + jmp + "\n")
}

// statement compilation
def compile_stmt(s: Stmt, env: Env) : (Instrs, Env) = s match {
  case Skip => (Nil, env)
  case Assign(x, a) => {
    val index = if (env.isDefinedAt(x)) env(x) else 
                    env.keys.size.toString
    (compile_aexp(a, env) ++ 
     List("istore " + index + "\n"), env + (x -> index))
  } 
  case If(b, bl1, bl2) => {
    val if_else = Fresh("If_else")
    val if_end = Fresh("If_end")
    val (instrs1, env1) = compile_block(bl1, env)
    val (instrs2, env2) = compile_block(bl2, env1)
    (compile_bexp(b, env, if_else) ++
     instrs1 ++
     List("goto " + if_end + "\n") ++
     List("\n" + if_else + ":\n\n") ++
     instrs2 ++
     List("\n" + if_end + ":\n\n"), env2)
  }
  case While(b, bl) => {
    val loop_begin = Fresh("Loop_begin")
    val loop_end = Fresh("Loop_end")
    val (instrs1, env1) = compile_block(bl, env)
    (List("\n" + loop_begin + ":\n\n") ++
     compile_bexp(b, env, loop_end) ++
     instrs1 ++
     List("goto " + loop_begin + "\n") ++
     List("\n" + loop_end + ":\n\n"), env1)
  }
  case Write(x) => 
    (List("iload " + env(x) + "\n" + 
           "invokestatic XXX/XXX/write(I)V\n"), env)
  case Array(s, n) => {
    val index = if (env.isDefinedAt(s)) throw new Exception("Array already defined") else 
                    env.keys.size.toString
    (List("ldc " ++ n.toString ++ "\n",
          "newarray int \n",
          "astore " ++ index ++ "\n"), env + (s -> index))
  } 
  case AssignA(s, a1, a2) => {
    val index = if (env.isDefinedAt(s)) env(s) else 
                    throw new Exception("Array not yet defined")
    (List("aload " + index + "\n") ++
     compile_aexp(a1, env) ++
     compile_aexp(a2, env) ++
     List("iastore \n"), env)
  } 
}

// compilation of a block (i.e. list of instructions)
def compile_block(bl: Block, env: Env) : (Instrs, Env) = bl match {
  case Nil => (Nil, env)
  case s::bl => {
    val (instrs1, env1) = compile_stmt(s, env)
    val (instrs2, env2) = compile_block(bl, env1)
    (instrs1 ++ instrs2, env2)
  }
}

// main compilation function for blocks
def compile(bl: Block, class_name: String) : String = {
  val instructions = compile_block(bl, Map.empty)._1
  (beginning ++ instructions.mkString ++ ending).replaceAllLiterally("XXX", class_name)
}


// compiling and running files
//
// JVM files can be assembled with 
//
//    java -jar jvm/jasmin-2.4/jasmin.jar fib.j
//
// and started with
//
//    java fib/fib



import scala.util._
import scala.sys.process._
import scala.io

def compile_tofile(bl: Block, class_name: String) = {
  val output = compile(bl, class_name)
  val fw = new java.io.FileWriter(class_name + ".j") 
  fw.write(output) 
  fw.close()
}

def compile_all(bl: Block, class_name: String) : Unit = {
  compile_tofile(bl, class_name)
  println("compiled ")
  val test = ("java -jar jvm/jasmin-2.4/jasmin.jar " + class_name + ".j").!!
  println("assembled ")
}

def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  (end - start)/(i * 1.0e9)
}


def compile_run(bl: Block, class_name: String) : Unit = {
  println("Start compilation")
  compile_all(bl, class_name)
  println("Time: " + time_needed(1, ("java " + class_name + "/" + class_name).!))
}


// Fibonacci numbers as a test-case
val fib_test = 
  List(Assign("n", Num(10)),            //  n := 10;                     
       Assign("minus1",Num(0)),         //  minus1 := 0;
       Assign("minus2",Num(1)),         //  minus2 := 1;
       Assign("temp",Num(0)),           //  temp := 0;
       While(Bop("<",Num(0),Var("n")),  //  while n > 0 do  {
          List(Assign("temp",Var("minus2")),    //  temp := minus2;
               Assign("minus2",Aop("+",Var("minus1"),Var("minus2"))), 
                                        //  minus2 := minus1 + minus2;
               Assign("minus1",Var("temp")), //  minus1 := temp;
               Assign("n",Aop("-",Var("n"),Num(1))))), //  n := n - 1 };
       Write("minus1"))                 //  write minus1


compile_run(fib_test, "fib")


val arr_test = 
  List(Array("a", 10),                 // a[10]
       Array("b", 2),                  // b[2]
       AssignA("a", Num(0), Num(10)),  // a[0] := 10
       Assign("x", Ref("a", Num(0))),  // x := a[0]
       Write("x"),                     // write x
       AssignA("b", Num(1), Num(5)),   // b[1] := 5
       Assign("x", Ref("b", Num(1))),  // x := b[1]
       Write("x"))                     // write x

compile_run(arr_test, "a")


//====================
// Parser Combinators
//====================

import scala.language.implicitConversions
import scala.language.reflectiveCalls


abstract class Parser[I <% Seq[_], T] {
  def parse(ts: I): Set[(T, I)]

  def parse_all(ts: I) : Set[T] =
    for ((head, tail) <- parse(ts); if (tail.isEmpty)) yield head
}

class SeqParser[I <% Seq[_], T, S](p: => Parser[I, T], q: => Parser[I, S]) extends Parser[I, (T, S)] {
  def parse(sb: I) = 
    for ((head1, tail1) <- p.parse(sb); 
         (head2, tail2) <- q.parse(tail1)) yield ((head1, head2), tail2)
}

class AltParser[I <% Seq[_], T](p: => Parser[I, T], q: => Parser[I, T]) extends Parser[I, T] {
  def parse(sb: I) = p.parse(sb) ++ q.parse(sb)   
}

class FunParser[I <% Seq[_], T, S](p: => Parser[I, T], f: T => S) extends Parser[I, S] {
  def parse(sb: I) = 
    for ((head, tail) <- p.parse(sb)) yield (f(head), tail)
}


import scala.util.matching.Regex
case class RegexParser(reg: Regex) extends Parser[String, String] {
  def parse(sb: String) = reg.findPrefixMatchOf(sb) match {
    case None => Set()
    case Some(m) => Set((m.matched, m.after.toString))  
  }
}

def StringParser(s: String) = RegexParser(Regex.quote(s).r)


implicit def string2parser(s : String) = StringParser(s)

implicit def ParserOps[I<% Seq[_], T](p: Parser[I, T]) = new {
  def | (q : => Parser[I, T]) = new AltParser[I, T](p, q)
  def ==>[S] (f: => T => S) = new FunParser[I, T, S](p, f)
  def ~[S] (q : => Parser[I, S]) = new SeqParser[I, T, S](p, q)
}

implicit def StringOps(s: String) = new {
  def | (q : => Parser[String, String]) = new AltParser[String, String](s, q)
  def | (r: String) = new AltParser[String, String](s, r)
  def ==>[S] (f: => String => S) = new FunParser[String, String, S](s, f)
  def ~[S](q : => Parser[String, S]) = 
    new SeqParser[String, String, S](s, q)
  def ~ (r: String) = 
    new SeqParser[String, String, String](s, r)
}


val NumParser = RegexParser("[0-9]+".r) ==> (s => s.toInt : Int)
val IdParser = RegexParser("[a-z][a-z,0-9]*".r)



// Grammar Rules

lazy val AExp: Parser[String, AExp] = 
  (Te ~ "+" ~ AExp) ==> { case ((x, _), z) => Aop("+", x, z):AExp } |
  (Te ~ "-" ~ AExp) ==> { case ((x, _), z) => Aop("-", x, z):AExp } | Te
lazy val Te: Parser[String, AExp] = 
  (Fa ~ "*" ~ Te) ==> { case ((x, _), z) => Aop("*", x, z):AExp } | Fa
lazy val Fa: Parser[String, AExp] = 
   ("(" ~ AExp ~ ")") ==> { case ((_, y), _) => y } | 
   (IdParser ~ "[" ~ AExp ~ "]") ==> { case (((x, _), y), _) => Ref(x, y) } |
   IdParser ==> Var | 
   NumParser ==> Num

// boolean expressions
lazy val BExp: Parser[String, BExp] = 
   (AExp ~ "=" ~ AExp) ==> { case ((x, y), z) => Bop("=", x, z):BExp } | 
   (AExp ~ "!=" ~ AExp) ==> { case ((x, y), z) => Bop("!=", x, z):BExp } | 
   (AExp ~ "<" ~ AExp) ==> { case ((x, y), z) => Bop("<", x, z):BExp } | 
   (AExp ~ ">" ~ AExp) ==> { case ((x, y), z) => Bop("<", z, x):BExp } | 
   ("true" ==> ((_) => True:BExp )) | 
   ("false" ==> ((_) => False:BExp )) |
   ("(" ~ BExp ~ ")") ==> { case ((x, y), z) => y}

lazy val Stmt: Parser[String, Stmt] =
   ("skip" ==> (_ => Skip: Stmt)) |
   (IdParser ~ ":=" ~ AExp) ==> { case ((x, y), z) => Assign(x, z): Stmt } |
   (IdParser ~ "[" ~ AExp ~ "]" ~ ":=" ~ AExp) ==> { 
     case (((((x, y), z), v), w), u) => AssignA(x, z, u): Stmt } |
   ("if" ~ BExp ~ "then" ~ Block ~ "else" ~ Block) ==>
    { case (((((x,y),z),u),v),w) => If(y, u, w): Stmt } |
   ("while" ~ BExp ~ "do" ~ Block) ==> { case (((x, y), z), w) => While(y, w) } |
   ("new" ~ IdParser ~ "[" ~ NumParser ~ "]") ==> { case ((((x, y), z), u), v) => Array(y, u) } |
   ("write" ~ IdParser) ==> { case (_, y) => Write(y) } 
 
lazy val Stmts: Parser[String, Block] =
  (Stmt ~ ";" ~ Stmts) ==> { case ((x, y), z) => x :: z : Block } |
  (Stmt ==> ((s) => List(s) : Block)) 


lazy val Block: Parser[String, Block] =
   ("{" ~ Stmts ~ "}") ==> { case ((x, y), z) => y} | 
   (Stmt ==> (s => List(s)))

Stmts.parse_all("x2:=5+a")
Stmts.parse_all("x2:=5+a[3+a]")
Stmts.parse_all("a[x2+3]:=5+a[3+a]")
Block.parse_all("{x:=5;y:=8}")
Block.parse_all("if(false)then{x:=5}else{x:=10}")



val fib = """
   n := 10;
   minus1 := 0;
   minus2 := 1;
   temp:=0;
   while (n > 0) do {
     temp := minus2; 
     minus2 := minus1 + minus2;
     minus1 := temp;
     n := n - 1};
   result := minus2;
   write result
   """.replaceAll("\\s+", "")

val fib_prog = Stmts.parse_all(fib).toList
//compile_run(fib_prog.head, "fib")


// BF 

// simple instructions
def instr(c: Char) : String = c match {
  case '>' => "ptr := ptr + 1;"
  case '<' => "ptr := ptr - 1;"
  case '+' => "field[ptr] := field[ptr] + 1;"
  case '-' => "field[ptr] := field[ptr] - 1;"
  case '.' => "x := field[ptr]; write x;"
  //case ',' => "XXX" // "ptr = getchar();\n"
  case '['  => "while (field[ptr] != 0) do {"
  case ']'  => "skip};"
  case _ => ""
}

def instrs(prog: String) : String =
  prog.toList.map(instr).mkString


def splice(cs: List[Char], acc: List[(Char, Int)]) : List[(Char, Int)] = (cs, acc) match {
  case (Nil, acc) => acc
  case (c :: cs, Nil) => splice(cs, List((c, 1)))
  case (c :: cs, (d, n) :: acc) => 
    if (c == d) splice(cs, (c, n + 1) :: acc)
    else splice(cs, (c, 1) :: (d, n) :: acc)
}

def spl(s: String) = splice(s.toList, Nil).reverse

def instr2(c: Char, n: Int) : String = c match {
  case '>' => "ptr := ptr + " + n.toString + ";"
  case '<' => "ptr := ptr - " + n.toString + ";"
  case '+' => "field[ptr] := field[ptr] + " + n.toString + ";"
  case '-' => "field[ptr] := field[ptr] - " + n.toString + ";"
  case '.' => "x := field[ptr]; write x;" //* n
  //case ',' => "*ptr = getchar();\n" * n
  case '['  => "while (field[ptr] != 0) do {" * n 
  case ']'  => "skip};" * n
  case _ => ""
}

def instrs2(prog: String) : String =
  spl(prog).map{ case (c, n) => instr2(c, n) }.mkString


def bf_str(prog: String) : String = {
  "\n" ++
  //"new field[30000];\n" ++
  "ptr := 15000;" ++
  instrs2(prog) ++
  "skip"
}

def bf_run(prog: String, name: String) = {
  println("BF processing start")
  val bf_string = bf_str(prog).replaceAll("\\s", "")
  println(s"BF parsing start (string length ${bf_string.length})")
  val bf_prog = Stmts.parse_all(bf_string).toList.head
  println("BF Compile start")
  compile_run(Array("field", 30000) :: bf_prog, name)
}



val bf1 = """++++++++[>+>++++<<-]>++>>+<[-[>>+<<-]+>>]>+[-<<<[
      ->[+[-]+>++>>>-<<]<[<]>>++++++[<<+++++>>-]+<<++.[-]<<
      ]>.>+[>>]>+]"""

bf_run(bf1, "sier")

bf_run("""++++++++[>++++[>++>+++>+++>+<<<<-]>+>+>->>+[<]<-]>>.>---.+++++++
       ..+++.>>.<-.<.+++.------.--------.>>+.>++.""", "hello")

bf_run("""+++++++++++
      >+>>>>++++++++++++++++++++++++++++++++++++++++++++
      >++++++++++++++++++++++++++++++++<<<<<<[>[>>>>>>+>
      +<<<<<<<-]>>>>>>>[<<<<<<<+>>>>>>>-]<[>++++++++++[-
      <-[>>+>+<<<-]>>>[<<<+>>>-]+<[>[-]<[-]]>[<<[>>>+<<<
      -]>>[-]]<<]>>>[>>+>+<<<-]>>>[<<<+>>>-]+<[>[-]<[-]]
      >[<<+>>[-]]<<<<<<<]>>>>>[+++++++++++++++++++++++++
      +++++++++++++++++++++++.[-]]++++++++++<[->-<]>++++
      ++++++++++++++++++++++++++++++++++++++++++++.[-]<<
      <<<<<<<<<<[>>>+>+<<<<-]>>>>[<<<<+>>>>-]<-[>>.>.<<<
      [-]]<<[>>+>+<<<-]>>>[<<<+>>>-]<<[<+>-]>[<+>-]<<<-]""", "fibs")
