// A Small Compiler for a Simple Functional Language
// (includes an external lexer and parser)
//
// call with 
//
//     scala fun.scala fact
//
//     scala fun.scala defs
//
// this will generate a .j file and run the jasmin
// assembler (installed at jvm/jasmin-2.4/jasmin.jar)
// it runs the resulting JVM file twice for timing 
// purposes.




object Compiler {

import java.io._  
import scala.util._
import scala.sys.process._

// Abstract syntax trees for the Fun language
abstract class Exp extends Serializable 
abstract class BExp extends Serializable 
abstract class Decl extends Serializable 

case class Def(name: String, args: List[String], body: Exp) extends Decl
case class Main(e: Exp) extends Decl

case class Call(name: String, args: List[Exp]) extends Exp
case class If(a: BExp, e1: Exp, e2: Exp) extends Exp
case class Write(e: Exp) extends Exp
case class Var(s: String) extends Exp
case class Num(i: Int) extends Exp
case class Aop(o: String, a1: Exp, a2: Exp) extends Exp
case class Sequence(e1: Exp, e2: Exp) extends Exp
case class Bop(o: String, a1: Exp, a2: Exp) extends BExp


// compiler - built-in functions 
// copied from http://www.ceng.metu.edu.tr/courses/ceng444/link/jvm-cpm.html
//

val library = """
.class public XXX.XXX
.super java/lang/Object

.method public <init>()V
        aload_0
        invokenonvirtual java/lang/Object/<init>()V
        return
.end method

.method public static write(I)V 
        .limit locals 1 
        .limit stack 2 
        getstatic java/lang/System/out Ljava/io/PrintStream; 
        iload 0
        invokevirtual java/io/PrintStream/println(I)V 
        return 
.end method

"""

// calculating the maximal needed stack size
def max_stack_exp(e: Exp): Int = e match {
  case Call(_, args) => args.map(max_stack_exp).sum
  case If(a, e1, e2) => max_stack_bexp(a) + (List(max_stack_exp(e1), max_stack_exp(e2)).max)
  case Write(e) => max_stack_exp(e) + 1
  case Var(_) => 1
  case Num(_) => 1
  case Aop(_, a1, a2) => max_stack_exp(a1) + max_stack_exp(a2)
  case Sequence(e1, e2) => List(max_stack_exp(e1), max_stack_exp(e2)).max
}

def max_stack_bexp(e: BExp): Int = e match {
  case Bop(_, a1, a2) => max_stack_exp(a1) + max_stack_exp(a2)
}


// for generating new labels
var counter = -1

def Fresh(x: String) = {
  counter += 1
  x ++ "_" ++ counter.toString()
}

// convenient string interpolations 
// for instructions, labels and methods
import scala.language.implicitConversions
import scala.language.reflectiveCalls

implicit def sring_inters(sc: StringContext) = new {
    def i(args: Any*): String = "   " ++ sc.s(args:_*) ++ "\n"
    def l(args: Any*): String = sc.s(args:_*) ++ ":\n"
    def m(args: Any*): String = sc.s(args:_*) ++ "\n"
}


type Env = Map[String, Int]

// compile expressions
def compile_exp(a: Exp, env : Env) : String = a match {
  case Num(i) => i"ldc $i"
  case Var(s) => i"iload ${env(s)}"
  case Aop("+", a1, a2) => compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"iadd"
  case Aop("-", a1, a2) => compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"isub"
  case Aop("*", a1, a2) => compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"imul"
  case Aop("/", a1, a2) => compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"idiv"
  case Aop("%", a1, a2) => compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"irem"
  case If(b, a1, a2) => {
    val if_else = Fresh("If_else")
    val if_end = Fresh("If_end")
    compile_bexp(b, env, if_else) ++
    compile_exp(a1, env) ++
    i"goto $if_end" ++
    l"$if_else" ++
    compile_exp(a2, env) ++
    l"$if_end"
  }
  case Call(name, args) => {
    val is = "I" * args.length
    args.map(a => compile_exp(a, env)).mkString ++
    i"invokestatic XXX/XXX/$name($is)I"
  }
  case Sequence(a1, a2) => {
    compile_exp(a1, env) ++ i"pop" ++ compile_exp(a2, env)
  }
  case Write(a1) => {
    compile_exp(a1, env) ++
    i"dup" ++
    i"invokestatic XXX/XXX/write(I)V"
  }
}

// compile boolean expressions
def compile_bexp(b: BExp, env : Env, jmp: String) : String = b match {
  case Bop("==", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"if_icmpne $jmp"
  case Bop("!=", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"if_icmpeq $jmp"
  case Bop("<", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"if_icmpge $jmp"
  case Bop("<=", a1, a2) => 
    compile_exp(a1, env) ++ compile_exp(a2, env) ++ i"if_icmpgt $jmp"
}

// compile function for declarations and main
def compile_decl(d: Decl) : String = d match {
  case Def(name, args, a) => { 
    val env = args.zipWithIndex.toMap
    val is = "I" * args.length
    m".method public static $name($is)I" ++
    m".limit locals ${args.length}" ++
    m".limit stack ${1 + max_stack_exp(a)}" ++
    l"${name}_Start" ++   
    compile_exp(a, env) ++
    i"ireturn" ++
    m".end method\n"
  }
  case Main(a) => {
    m".method public static main([Ljava/lang/String;)V" ++
    m".limit locals 200" ++
    m".limit stack 200" ++
    compile_exp(a, Map()) ++
    i"invokestatic XXX/XXX/write(I)V" ++
    i"return" ++
    m".end method\n"
  }
}

// main compiler functions

def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  (end - start)/(i * 1.0e9)
}

def deserialise[T](fname: String) : Try[T] = {
  import scala.util.Using
  Using(new ObjectInputStream(new FileInputStream(fname))) {
    in => in.readObject.asInstanceOf[T]
  }
}

def compile(class_name: String) : String = {
  val ast = deserialise[List[Decl]](class_name ++ ".prs").getOrElse(Nil) 
  val instructions = ast.map(compile_decl).mkString
  (library + instructions).replaceAllLiterally("XXX", class_name)
}

def compile_to_file(class_name: String) = {
  val output = compile(class_name)
  scala.tools.nsc.io.File(s"${class_name}.j").writeAll(output)
}

def compile_and_run(class_name: String) : Unit = {
  compile_to_file(class_name)
  (s"java -jar jvm/jasmin-2.4/jasmin.jar ${class_name}.j").!!
  println("Time: " + time_needed(2, (s"java ${class_name}/${class_name}").!))
}


// some examples of .fun files
//compile_to_file("fact")
//compile_and_run("fact")
//compile_and_run("defs")


def main(args: Array[String]) : Unit = 
   compile_and_run(args(0))


}
