// regular expressions
abstract class Rexp

case object NULL extends Rexp
case object EMPTY extends Rexp
case class CHAR(c: Char) extends Rexp
case class ALT(r1: Rexp, r2: Rexp) extends Rexp
case class SEQ(r1: Rexp, r2: Rexp) extends Rexp
case class STAR(r: Rexp) extends Rexp

// some convenience for typing in regular expressions
def charlist2rexp(s : List[Char]) : Rexp = s match {
  case Nil => EMPTY
  case c::Nil => CHAR(c)
  case c::s => SEQ(CHAR(c), charlist2rexp(s))
}
implicit def string2rexp(s : String) : Rexp = charlist2rexp(s.toList)


// for example
println(STAR("abc"))

// produces STAR(SEQ(CHAR(a),SEQ(CHAR(b),SEQ(CHAR(c),EMPTY))))



// a simple-minded regular expression matcher:
// it loops for examples like STAR(EMPTY) with
// strings this regular expression does not match

def smatchers(rs: List[Rexp], s: List[Char]) : Boolean = (rs, s) match {
  case (NULL::rs, s) => false
  case (EMPTY::rs, s) => smatchers(rs, s)
  case (CHAR(c)::rs, Nil) => false
  case (CHAR(c)::rs, d::s) => (c ==d) && smatchers(rs, s) 
  case (ALT(r1, r2)::rs, s) => smatchers(r1::rs, s) || smatchers(r2::rs, s)
  case (SEQ(r1, r2)::rs, s) => smatchers(r1::r2::rs, s) 
  case (STAR(r)::rs, s) => smatchers(rs, s) || smatchers(r::STAR(r)::rs, s) 
  case (Nil, s) => s == Nil
}

def smatcher(r: Rexp, s: String) = smatchers(List(r), s.toList)

// regular expression: a
println(smatcher(CHAR('a'), "ab")) 

// regular expression: a + (b o c)
println(smatcher(ALT(CHAR('a'), SEQ(CHAR('b'), CHAR('c'))), "ab")) 

// regular expression: a + (b o c)
println(smatcher(ALT(CHAR('a'), SEQ(CHAR('b'), CHAR('c'))), "bc")) 

// loops for regular expression epsilon* 
//println(smatcher(STAR(EMPTY), "a"))



// Regular expression matcher that works properly
//================================================

// nullable function: tests whether the regular 
// expression can recognise the empty string
def nullable (r: Rexp) : Boolean = r match {
  case NULL => false
  case EMPTY => true
  case CHAR(_) => false
  case ALT(r1, r2) => nullable(r1) || nullable(r2)
  case SEQ(r1, r2) => nullable(r1) && nullable(r2)
  case STAR(_) => true
}

// derivative of a regular expression w.r.t. a character
def der (c: Char, r: Rexp) : Rexp = r match {
  case NULL => NULL
  case EMPTY => NULL
  case CHAR(d) => if (c == d) EMPTY else NULL
  case ALT(r1, r2) => ALT(der(c, r1), der(c, r2))
  case SEQ(r1, r2) => 
    if (nullable(r1)) ALT(SEQ(der(c, r1), r2), der(c, r2))
    else SEQ(der(c, r1), r2)
  case STAR(r) => SEQ(der(c, r), STAR(r))
}

// derivative w.r.t. a string (iterates der)
def ders (s: List[Char], r: Rexp) : Rexp = s match {
  case Nil => r
  case c::s => ders(s, der(c, r))
}

// main matcher function
def matcher(r: Rexp, s: String) : Boolean = nullable(ders(s.toList, r))


//examples

println(matcher(SEQ(STAR("a"), STAR("b")), "bbaaa"))
println(matcher(ALT(STAR("a"), STAR("b")), ""))
println(matcher("abc", ""))
println(matcher(STAR(ALT(EMPTY, "a")), ""))
println(matcher(STAR(EMPTY), "a"))
println(matcher("cab","cab"))
println(matcher(STAR("a"),"aaa"))
println(matcher("cab" ,"cab"))
println(matcher(STAR("a"),"aaa"))


