// A parser and evaluator for the WHILE language
// 
import matcher._
import parser._


// some regular expressions
val SYM = RANGE("ABCDEFGHIJKLMNOPQRSTUVXYZabcdefghijklmnopqrstuvwxyz_")
val DIGIT = RANGE("0123456789")
val ID = SEQ(SYM, STAR(ALT(SYM, DIGIT))) 
val NUM = PLUS(DIGIT)
val KEYWORD = ALTS("skip", "while", "do", "if", "then", "else", "true", "false", "write") 
val SEMI: Rexp = ";"
val OP: Rexp = ALTS(":=", "=", "-", "+", "*", "!=", "<", ">")
val WHITESPACE = PLUS(RANGE(" \n"))
val RPAREN: Rexp = ")"
val LPAREN: Rexp = "("
val BEGIN: Rexp = "{"
val END: Rexp = "}"
val COMMENT = SEQS("/*", NOT(SEQS(STAR(ALLC), "*/", STAR(ALLC))), "*/")

// tokens for classifying the strings that have been recognised
abstract class Token
case object T_WHITESPACE extends Token
case object T_COMMENT extends Token
case object T_SEMI extends Token
case object T_LPAREN extends Token
case object T_RPAREN extends Token
case object T_BEGIN extends Token
case object T_END extends Token
case class T_ID(s: String) extends Token
case class T_OP(s: String) extends Token
case class T_NUM(s: String) extends Token
case class T_KWD(s: String) extends Token

val lexing_rules: List[(Rexp, List[Char] => Token)] = 
  List((KEYWORD, (s) => T_KWD(s.mkString)),
       (ID, (s) => T_ID(s.mkString)),
       (OP, (s) => T_OP(s.mkString)),
       (NUM, (s) => T_NUM(s.mkString)),
       (SEMI, (s) => T_SEMI),
       (LPAREN, (s) => T_LPAREN),
       (RPAREN, (s) => T_RPAREN),
       (BEGIN, (s) => T_BEGIN),
       (END, (s) => T_END),
       (WHITESPACE, (s) => T_WHITESPACE),
       (COMMENT, (s) => T_COMMENT))

// the tokenizer
val Tok = Tokenizer(lexing_rules, List(T_WHITESPACE, T_COMMENT))

// the abstract syntax trees
abstract class Stmt
abstract class AExp
abstract class BExp 
type Block = List[Stmt]
case object Skip extends Stmt
case class If(a: BExp, bl1: Block, bl2: Block) extends Stmt
case class While(b: BExp, bl: Block) extends Stmt
case class Assign(s: String, a: AExp) extends Stmt
case class Write(s: String) extends Stmt

case class Var(s: String) extends AExp
case class Num(i: Int) extends AExp
case class Aop(o: String, a1: AExp, a2: AExp) extends AExp

case object True extends BExp
case object False extends BExp
case class Bop(o: String, a1: AExp, a2: AExp) extends BExp

// atomic parsers
case class TokParser(tok: Token) extends Parser[List[Token], Token] {
  def parse(ts: List[Token]) = ts match {
    case t::ts if (t == tok) => Set((t, ts)) 
    case _ => Set ()
  }
}
implicit def token2tparser(t: Token) = TokParser(t)

case object NumParser extends Parser[List[Token], Int] {
  def parse(ts: List[Token]) = ts match {
    case T_NUM(s)::ts => Set((s.toInt, ts)) 
    case _ => Set ()
  }
}

case object IdParser extends Parser[List[Token], String] {
  def parse(ts: List[Token]) = ts match {
    case T_ID(s)::ts => Set((s, ts)) 
    case _ => Set ()
  }
}


// arithmetic expressions
lazy val AExp: Parser[List[Token], AExp] = 
  (T ~ T_OP("+") ~ AExp) ==> { case ((x, y), z) => Aop("+", x, z): AExp } ||
  (T ~ T_OP("-") ~ AExp) ==> { case ((x, y), z) => Aop("-", x, z): AExp } || T  
lazy val T: Parser[List[Token], AExp] = 
  (F ~ T_OP("*") ~ T) ==> { case ((x, y), z) => Aop("*", x, z): AExp } || F
lazy val F: Parser[List[Token], AExp] = 
  (T_LPAREN ~> AExp <~ T_RPAREN) || 
  IdParser ==> Var || 
  NumParser ==> Num

// boolean expressions
lazy val BExp: Parser[List[Token], BExp] = 
  (T_KWD("true") ==> ((_) => True: BExp)) || 
  (T_KWD("false") ==> ((_) => False: BExp)) ||
  (T_LPAREN ~> BExp <~ T_RPAREN) ||
  (AExp ~ T_OP("=") ~ AExp) ==> { case ((x, y), z) => Bop("=", x, z): BExp } || 
  (AExp ~ T_OP("!=") ~ AExp) ==> { case ((x, y), z) => Bop("!=", x, z): BExp } || 
  (AExp ~ T_OP("<") ~ AExp) ==> { case ((x, y), z) => Bop("<", x, z): BExp } || 
  (AExp ~ T_OP(">") ~ AExp) ==> { case ((x, y), z) => Bop("<", z, x): BExp }

lazy val Stmt: Parser[List[Token], Stmt] =
  (T_KWD("skip") ==> ((_) => Skip: Stmt)) ||
  (IdParser ~ T_OP(":=") ~ AExp) ==> { case ((x, y), z) => Assign(x, z): Stmt } ||
  (T_KWD("if") ~ BExp ~ T_KWD("then") ~ Block ~ T_KWD("else") ~ Block) ==>
    { case (((((x,y),z),u),v),w) => If(y, u, w): Stmt } ||
  (T_KWD("while") ~ BExp ~ T_KWD("do") ~ Block) ==> { case (((x, y), z), w) => While(y, w) } || 
  (T_KWD("write") ~ IdParser) ==> { case (x, y) => Write(y) } 

lazy val Stmts: Parser[List[Token], Block] =
  (Stmt ~ T_SEMI ~ Stmts) ==> { case ((x, y), z) => x :: z : Block } ||
  (Stmt ==> ((s) => List(s) : Block))

lazy val Block: Parser[List[Token], Block] =
  (T_BEGIN ~> Stmts <~ T_END) || 
  (Stmt ==> ((s) => List(s)))

// interpreter
type Env = Map[String, Int]

def eval_bexp(b: BExp, env: Env) : Boolean = b match {
  case True => true
  case False => false
  case Bop("=", a1, a2) => eval_aexp(a1, env) == eval_aexp(a2, env)
  case Bop("!=", a1, a2) => !(eval_aexp(a1, env) == eval_aexp(a2, env))
  case Bop("<", a1, a2) => eval_aexp(a1, env) < eval_aexp(a2, env)
}

def eval_aexp(a: AExp, env : Env) : Int = a match {
  case Num(i) => i
  case Var(s) => env(s)
  case Aop("+", a1, a2) => eval_aexp(a1, env) + eval_aexp(a2, env)
  case Aop("-", a1, a2) => eval_aexp(a1, env) - eval_aexp(a2, env)
  case Aop("*", a1, a2) => eval_aexp(a1, env) * eval_aexp(a2, env)
}

def eval_stmt(s: Stmt, env: Env) : Env = s match {
  case Skip => env
  case Assign(x, a) => env + (x -> eval_aexp(a, env))
  case If(b, bl1, bl2) => if (eval_bexp(b, env)) eval_bl(bl1, env) else eval_bl(bl2, env) 
  case While(b, bl) => 
    if (eval_bexp(b, env)) eval_stmt(While(b, bl), eval_bl(bl, env))
    else env
  case Write(x) => { println(env(x)); env }
}

def eval_bl(bl: Block, env: Env) : Env = bl match {
  case Nil => env
  case s::bl => eval_bl(bl, eval_stmt(s, env))
}

def eval_prog(name: String) : Env = {
  val tks = Tok.fromFile(name)
  val ast = Stmts.parse_single(tks)
  eval_bl(ast, Map.empty)
}


//examples

//eval_prog("loops.while")
eval_prog("fib.while")


def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  (end - start)/(i * 1.0e9)
}


val test_prog = """
start := XXX;
x := start;
y := start;
z := start;
while 0 < x do {
 while 0 < y do {
  while 0 < z do {
    z := z - 1
  };
  z := start;
  y := y - 1
 };     
 y := start;
 x := x - 1
}
"""



def eval_test(n: Int) : Unit = {
  val tks = Tok.fromString(test_prog.replaceAllLiterally("XXX", n.toString))
  val ast = Stmts.parse_single(tks)
  println(n + " " + time_needed(2, eval_bl(ast, Map.empty)))
}

List(1, 200, 400, 600, 800, 1000, 1200, 1400, 1600).map(eval_test(_))







