// Version with explicit an explicit n-times regular expression;
// this keeps the overall regular expression in the EVIL1 regular 
// expression small

abstract class Rexp 
case object ZERO extends Rexp
case object ONE extends Rexp
case class CHAR(c: Char) extends Rexp
case class ALT(r1: Rexp, r2: Rexp) extends Rexp 
case class SEQ(r1: Rexp, r2: Rexp) extends Rexp 
case class STAR(r: Rexp) extends Rexp 
case class NTIMES(r: Rexp, n: Int) extends Rexp   //explicit constructor for n-times


def nullable (r: Rexp) : Boolean = r match {
  case ZERO => false
  case ONE => true
  case CHAR(_) => false
  case ALT(r1, r2) => nullable(r1) || nullable(r2)
  case SEQ(r1, r2) => nullable(r1) && nullable(r2)
  case STAR(_) => true
  case NTIMES(r, i) => if (i == 0) true else nullable(r)
}


def der (c: Char, r: Rexp) : Rexp = r match {
  case ZERO => ZERO
  case ONE => ZERO
  case CHAR(d) => if (c == d) ONE else ZERO
  case ALT(r1, r2) => ALT(der(c, r1), der(c, r2))
  case SEQ(r1, r2) => 
    if (nullable(r1)) ALT(SEQ(der(c, r1), r2), der(c, r2))
    else SEQ(der(c, r1), r2)
  case STAR(r1) => SEQ(der(c, r1), STAR(r1))
  case NTIMES(r1, i) => 
    if (i == 0) ZERO else SEQ(der(c, r1), NTIMES(r1, i - 1))
}

def ders (s: List[Char], r: Rexp) : Rexp = s match {
  case Nil => r
  case c::s => ders(s, der(c, r))
}

def matches(r: Rexp, s: String) : Boolean = nullable(ders(s.toList, r))


//optional regular expression: one or zero times
def OPT(r: Rexp) = ALT(r, ONE)


// Test Cases

//evil regular expressions
def EVIL1(n: Int) = SEQ(NTIMES(OPT(CHAR('a')), n), NTIMES(CHAR('a'), n))
val EVIL2 = SEQ(STAR(STAR(CHAR('a'))), CHAR('b'))

def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  (end - start)/(i * 1.0e9)
}


//test: (a?{n}) (a{n})
for (i <- 1 to 1201 by 100) {
  println(i + " " + "%.5f".format(time_needed(2, matches(EVIL1(i), "a" * i))))
}

for (i <- 1 to 1201 by 100) {
  println(i + " " + "%.5f".format(time_needed(2, matches(EVIL1(i), "a" * i))))
}


//test: (a*)* b
for (i <- 1 to 20) {
  println(i + " " + "%.5f".format(time_needed(2, matches(EVIL2, "a" * i))))
}

for (i <- 1 to 20) {
  println(i + " " + "%.5f".format(time_needed(2, matches(EVIL2, "a" * i))))
}



// size of a regular expressions - for testing purposes 
def size(r: Rexp) : Int = r match {
  case ZERO => 1
  case ONE => 1
  case CHAR(_) => 1
  case ALT(r1, r2) => 1 + size(r1) + size(r2)
  case SEQ(r1, r2) => 1 + size(r1) + size(r2)
  case STAR(r) => 1 + size(r)
  case NTIMES(r, _) => 1 + size(r)
}

// EVIL1(n) has now a constant size, no matter
// what n is 

size(EVIL1(1))  // 7
size(EVIL1(3))  // 7
size(EVIL1(5))  // 7
size(EVIL1(7))  // 7


// but the size of the derivatives can grow 
// quite dramatically

size(ders("".toList, EVIL2))     // 5
size(ders("a".toList, EVIL2))    // 12
size(ders("aa".toList, EVIL2))   // 28
size(ders("aaa".toList, EVIL2))  // 58
size(ders("aaaa".toList, EVIL2)) // 116
