// A Small Compiler for the WHILE Language with Arrays
// (it does not use a parser and lexer)

// the new parts are
//    - declaring an array
//    - references an array cell
//    - assigning an array cell

// the abstract syntax trees
abstract class Stmt
abstract class AExp
abstract class BExp 
type Block = List[Stmt]

// statements
case object Skip extends Stmt
case class If(a: BExp, bl1: Block, bl2: Block) extends Stmt
case class While(b: BExp, bl: Block) extends Stmt
case class Assign(s: String, a: AExp) extends Stmt
case class AssignA(s: String, a1: AExp, a2: AExp) extends Stmt
case class Write(s: String) extends Stmt
case class Array(s: String, n: Int) extends Stmt 

// arithmetic expressions
case class Var(s: String) extends AExp
case class Num(i: Int) extends AExp
case class Aop(o: String, a1: AExp, a2: AExp) extends AExp
case class Ref(s: String, a1: AExp) extends AExp

// boolean expressions
case object True extends BExp
case object False extends BExp
case class Bop(o: String, a1: AExp, a2: AExp) extends BExp


// compiler headers needed for the JVM
// (contains an init method, as well as methods for read and write)
val beginning = """
.class public XXX.XXX
.super java/lang/Object

.method public <init>()V
   aload_0
   invokenonvirtual java/lang/Object/<init>()V
   return
.end method

.method public static write(I)V 
    .limit locals 1 
    .limit stack 2 
    getstatic java/lang/System/out Ljava/io/PrintStream; 
    iload 0
    invokevirtual java/io/PrintStream/println(I)V 
    return 
.end method

.method public static main([Ljava/lang/String;)V
   .limit locals 200
   .limit stack 200

; COMPILED CODE STARTS

"""

val ending = """
; COMPILED CODE ENDS
   return

.end method
"""


// Compiler functions

// for generating new labels
var counter = -1

def Fresh(x: String) = {
  counter += 1
  x ++ "_" ++ counter.toString()
}

// convenient string interpolations 
// for instructions and labels
import scala.language.implicitConversions
import scala.language.reflectiveCalls

implicit def sring_inters(sc: StringContext) = new {
    def i(args: Any*): String = "   " ++ sc.s(args:_*) ++ "\n"
    def l(args: Any*): String = sc.s(args:_*) ++ ":\n"
}


// environments 
type Env = Map[String, String]

// arithmetic expression compilation
def compile_aexp(a: AExp, env : Env) : String = a match {
  case Num(i) => i"ldc $i"
  case Var(s) => i"iload ${env(s)}"
  case Aop("+", a1, a2) => 
    compile_aexp(a1, env) ++ compile_aexp(a2, env) ++ i"iadd"
  case Aop("-", a1, a2) => 
    compile_aexp(a1, env) ++ compile_aexp(a2, env) ++ i"isub"
  case Aop("*", a1, a2) => 
    compile_aexp(a1, env) ++ compile_aexp(a2, env) ++ i"imul"
  case Ref(s, a1) => 
    i"aload ${env(s)}" ++ compile_aexp(a1, env) ++ i"iaload"  
}

// boolean expression compilation
def compile_bexp(b: BExp, env : Env, jmp: String) : String = b match {
  case True => ""
  case False => i"goto $jmp"
  case Bop("=", a1, a2) => 
    compile_aexp(a1, env) ++ compile_aexp(a2, env) ++ i"if_icmpne $jmp"
  case Bop("!=", a1, a2) => 
    compile_aexp(a1, env) ++ compile_aexp(a2, env) ++ i"if_icmpeq $jmp"
  case Bop("<", a1, a2) => 
    compile_aexp(a1, env) ++ compile_aexp(a2, env) ++ i"if_icmpge $jmp"
}

// statement compilation
def compile_stmt(s: Stmt, env: Env) : (String, Env) = s match {
  case Skip => ("", env)
  case Assign(x, a) => {
    val index = if (env.isDefinedAt(x)) env(x) else 
                    env.keys.size.toString
    (compile_aexp(a, env) ++ i"istore $index", env + (x -> index))
  } 
  case If(b, bl1, bl2) => {
    val if_else = Fresh("If_else")
    val if_end = Fresh("If_end")
    val (instrs1, env1) = compile_block(bl1, env)
    val (instrs2, env2) = compile_block(bl2, env1)
    (compile_bexp(b, env, if_else) ++
     instrs1 ++
     i"goto $if_end" ++
     l"$if_else" ++
     instrs2 ++
     l"$if_end", env2)
  }
  case While(b, bl) => {
    val loop_begin = Fresh("Loop_begin")
    val loop_end = Fresh("Loop_end")
    val (instrs1, env1) = compile_block(bl, env)
    (l"$loop_begin" ++
     compile_bexp(b, env, loop_end) ++
     instrs1 ++
     i"goto $loop_begin" ++
     l"$loop_end", env1)
  }
  case Write(x) => 
    (i"iload ${env(x)}" ++ 
     i"invokestatic XXX/XXX/write(I)V", env)
  case Array(s, n) => {
    val index = if (env.isDefinedAt(s)) throw new Exception("Array already defined") else 
                    env.keys.size.toString
    (i"ldc $n" ++
     i"newarray int" ++
     i"astore $index", env + (s -> index))
  } 
  case AssignA(s, a1, a2) => {
    val index = if (env.isDefinedAt(s)) env(s) else 
                    throw new Exception("Array not yet defined")
    (i"aload $index" ++
     compile_aexp(a1, env) ++
     compile_aexp(a2, env) ++
     i"iastore", env)
  } 
}

// compilation of a block (i.e. list of instructions)
def compile_block(bl: Block, env: Env) : (String, Env) = bl match {
  case Nil => ("", env)
  case s::bl => {
    val (instrs1, env1) = compile_stmt(s, env)
    val (instrs2, env2) = compile_block(bl, env1)
    (instrs1 ++ instrs2, env2)
  }
}

// main compilation function for blocks
def compile(bl: Block, class_name: String) : String = {
  val instructions = compile_block(bl, Map.empty)._1
  (beginning ++ instructions.mkString ++ ending).replaceAllLiterally("XXX", class_name)
}


// main compiler functions
import scala.util._
import scala.sys.process._
import scala.io

def compile_tofile(bl: Block, class_name: String) = {
  val output = compile(bl, class_name)
  scala.tools.nsc.io.File(s"${class_name}.j").writeAll(output)
}

def compile_all(bl: Block, class_name: String) : Unit = {
  compile_tofile(bl, class_name)
  println("compiled ")
  (s"java -jar jvm/jasmin-2.4/jasmin.jar ${class_name}.j").!!
  println("assembled ")
}

def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  (end - start)/(i * 1.0e9)
}


def compile_run(bl: Block, class_name: String) : Unit = {
  println("Start compilation")
  compile_all(bl, class_name)
  println("Start running")
  println("Time: " + time_needed(1, (s"java ${class_name}/${class_name}").!))
}

// a simple test case
val arr_test = 
  List(Array("a", 10),                 // a[10]
       Array("b", 2),                  // b[2]
       AssignA("a", Num(0), Num(10)),  // a[0] := 10
       Assign("x", Ref("a", Num(0))),  // x := a[0]
       Write("x"),                     // write x
       AssignA("b", Num(1), Num(5)),   // b[1] := 5
       Assign("x", Ref("b", Num(1))),  // x := b[1]
       Write("x"))                     // write x

compile_run(arr_test, "a")

