// A Small Compiler for a Simple Functional Language
// (includes an external lexer and parser)
//
// call with 
//
//     scala fun.scala fact
//
//     scala fun.scala defs
//
// this will generate a .j file and run the jasmin
// assembler (installed at jvm/jasmin-2.4/jasmin.jar)
// it runs the resulting JVM file twice for timing 
// purposes.




object Compiler {

import java.io._  
import scala.util._
import scala.sys.process._

// Abstract syntax trees for the Fun language
abstract class Exp extends Serializable 
abstract class BExp extends Serializable 
abstract class Decl extends Serializable 

case class Def(name: String, args: List[String], body: Exp) extends Decl
case class Main(e: Exp) extends Decl

case class Call(name: String, args: List[Exp]) extends Exp
case class If(a: BExp, e1: Exp, e2: Exp) extends Exp
case class Write(e: Exp) extends Exp
case class Var(s: String) extends Exp
case class Num(i: Int) extends Exp
case class Aop(o: String, a1: Exp, a2: Exp) extends Exp
case class Sequence(e1: Exp, e2: Exp) extends Exp
case class Bop(o: String, a1: Exp, a2: Exp) extends BExp


// compiler - built-in functions 
// copied from http://www.ceng.metu.edu.tr/courses/ceng444/link/jvm-cpm.html
//


// for generating new labels
var counter = -1

def Fresh(x: String) = {
  counter += 1
  x ++ "_" ++ counter.toString()
}



// Abstract syntax trees for the Fun language
abstract class KExp
abstract class KVal

case class KVar(s: String) extends KVal
case class KNum(i: Int) extends KVal
case class KAop(o: String, v1: KVal, v2: KVal) extends KVal
case class KBop(o: String, v1: KVal, v2: KVal) extends KVal
case class KCall(o: String, vrs: List[KVal]) extends KVal

case class KIf(x1: String, e1: KExp, e2: KExp) extends KExp {
  override def toString = s"KIf $x1\nIF\n$e1\nELSE\n$e2"
}
case class KLet(x: String, e1: KVal, e2: KExp) extends KExp {
  override def toString = s"let $x = $e1 in \n$e2" 
}
case class KReturn(v: KVal) extends KExp

def CPS(e: Exp)(k: KVal => KExp) : KExp = e match {
  case Var(s) => k(KVar(s)) 
  case Num(i) => k(KNum(i))
  case Aop(o, e1, e2) => {
    val z = Fresh("tmp")
    CPS(e1)(y1 => 
      CPS(e2)(y2 => KLet(z, KAop(o, y1, y2), k(KVar(z)))))
  }
  case If(Bop(o, b1, b2), e1, e2) => {
    val z = Fresh("tmp")
    CPS(b1)(y1 => 
      CPS(b2)(y2 => KLet(z, KBop(o, y1, y2), KIf(z, CPS(e1)(k), CPS(e2)(k)))))
  }
  case Call(name, args) => {
    def aux(args: List[Exp], vs: List[KVal]) : KExp = args match {
      case Nil => {
          val z = Fresh("tmp")
          KLet(z, KCall(name, vs), k(KVar(z)))
      }
      case e::es => CPS(e)(y => aux(es, vs ::: List(y)))
    }
    aux(args, Nil)
  }
  case Sequence(e1, e2) => {
    val z = Fresh("tmp")
    CPS(e1)(y1 => 
      CPS(e2)(y2 => KLet("_", y1, KLet(z, y2, k(KVar(z))))))
  }
}   

def CPSi(e: Exp) = CPS(e)(KReturn)

val e1 = Aop("*", Var("a"), Num(3))
CPS(e1)(KReturn)

val e2 = Aop("+", Aop("*", Var("a"), Num(3)), Num(4))
CPS(e2)(KReturn)

val e3 = Aop("+", Num(2), Aop("*", Var("a"), Num(3)))
CPS(e3)(KReturn)

val e4 = Aop("+", Aop("-", Num(1), Num(2)), Aop("*", Var("a"), Num(3)))
CPS(e4)(KReturn)

val e5 = If(Bop("==", Num(1), Num(1)), Num(3), Num(4))
CPS(e5)(KReturn)

val e6 = If(Bop("!=", Num(10), Num(10)), e5, Num(40))
CPS(e6)(KReturn)

val e7 = Call("foo", List(Num(3)))
CPS(e7)(KReturn)

val e8 = Call("foo", List(Num(3), Num(4), Aop("+", Num(5), Num(6))))
CPS(e8)(KReturn)

val e9 = Sequence(Aop("*", Var("a"), Num(3)), Aop("+", Var("b"), Num(6)))
CPS(e9)(KReturn)

val e = Aop("*", Aop("+", Num(1), Call("foo", List(Var("a"), Num(3)))), Num(4))
CPS(e)(KReturn)




// convenient string interpolations 
// for instructions, labels and methods
import scala.language.implicitConversions
import scala.language.reflectiveCalls

implicit def sring_inters(sc: StringContext) = new {
    def i(args: Any*): String = "   " ++ sc.s(args:_*) ++ "\n"
    def l(args: Any*): String = sc.s(args:_*) ++ ":\n"
    def m(args: Any*): String = sc.s(args:_*) ++ "\n"
}

def compile_op(op: String) = op match {
  case "+" => "add i32 "
  case "*" => "mul i32 "
  case "-" => "sub i32 "
  case "==" => "icmp eq i32 "
}

def compile_val(v: KVal) : String = v match {
  case KNum(i) => s"$i"
  case KVar(s) => s"%$s"
  case KAop(op, x1, x2) => 
    s"${compile_op(op)} ${compile_val(x1)}, ${compile_val(x2)}"
  case KBop(op, x1, x2) => 
    s"${compile_op(op)} ${compile_val(x1)}, ${compile_val(x2)}"
  case KCall(x1, args) => 
    s"call i32 @$x1 (${args.map(compile_val).mkString("i32 ", ", i32 ", "")})"
}

// compile K expressions
def compile_exp(a: KExp) : String = a match {
  case KReturn(v) =>
    i"ret i32 ${compile_val(v)}"
  case KLet(x: String, v: KVal, e: KExp) => 
    i"%$x = ${compile_val(v)}" ++ compile_exp(e)
  case KIf(x, e1, e2) => {
    val if_br = Fresh("if_br")
    val else_br = Fresh("else_br")
    i"br i1 %$x, label %$if_br, label %$else_br" ++
    l"\n$if_br" ++
    compile_exp(e1) ++
    l"\n$else_br" ++ 
    compile_exp(e2)
  }
}

/*  case Write(a1) => {
    compile_exp(a1, env) ++
    i"dup" ++
    i"invokestatic XXX/XXX/write(I)V"
  }
*/



// compile function for declarations and main
def compile_decl(d: Decl) : String = d match {
  case Def(name, args, body) => { 
    //println(s"DEF\n $name ($args) = \nBODY:")
    //println(CPSi(body))
    //println()
    //counter = -1
    m"define i32 @$name (${args.mkString("i32 %", ", i32 %", "")}) {" ++
    compile_exp(CPSi(body)) ++
    m"}\n"
  }
  case Main(body) => {
    m"define i32 @main() {" ++
    compile_exp(CPSi(body)) ++
    m"}\n"
  }
}

// main compiler functions

def time_needed[T](i: Int, code: => T) = {
  val start = System.nanoTime()
  for (j <- 1 to i) code
  val end = System.nanoTime()
  (end - start)/(i * 1.0e9)
}

def deserialise[T](fname: String) : Try[T] = {
  import scala.util.Using
  Using(new ObjectInputStream(new FileInputStream(fname))) {
    in => in.readObject.asInstanceOf[T]
  }
}

def compile(class_name: String) : String = {
  val ast = deserialise[List[Decl]](class_name ++ ".prs").getOrElse(Nil) 
  //println(ast(0).toString ++ "\n")
  ast.map(compile_decl).mkString
}

/*
def compile_to_file(class_name: String) = {
  val output = compile(class_name)
  scala.tools.nsc.io.File(s"${class_name}.j").writeAll(output)
}

def compile_and_run(class_name: String) : Unit = {
  compile_to_file(class_name)
  (s"java -jar jvm/jasmin-2.4/jasmin.jar ${class_name}.j").!!
  println("Time: " + time_needed(2, (s"java ${class_name}/${class_name}").!))
}
*/

// some examples of .fun files
//compile_to_file("fact")
//compile_and_run("fact")
//compile_and_run("defs")


def main(args: Array[String]) : Unit = 
   println(compile(args(0)))


}